package ags.rougedc.movement;

import java.util.List;
import java.util.Map;
import java.util.HashMap;

import robocode.Event;
import robocode.HitByBulletEvent;
import robocode.BulletHitBulletEvent;
import robocode.Bullet;
import static robocode.util.Utils.isNear;

import ags.rougedc.waves.*;
import ags.rougedc.robots.StatusRobot;
import ags.utils.newtree.KdTree;
import ags.utils.points.*;

/**
 * Watches the enemies target!
 * Stores stats in a statically defined HashMap (saves between rounds)
 * with the robot name as the key.
 * 
 * @author Alexander Schultz
 */
public class EnemyTargetingWatcher {
    private final EnemyWaveManager ewm;
    private static Map<String, KdTree<Double>> stats;
    private boolean need_wave_dangers_updated = false;
    
    public final ags.rougedc.research.SituationRecorder rec = new ags.rougedc.research.SituationRecorder();
    
    static private int hits = 0; 
    
    public EnemyTargetingWatcher(EnemyWaveManager ewm) {
        this.ewm = ewm;
        if (stats == null)
            stats = new HashMap<String, KdTree<Double>>();
    }
    
    public void handleEvents(Iterable<Event> i) {
        for (Event e : i) {
            if (e instanceof HitByBulletEvent) handleHit((HitByBulletEvent)e);
            if (e instanceof BulletHitBulletEvent) handleHit((BulletHitBulletEvent)e);
        }
        
        for (EnemyWave wave : ewm.getAll()) {
            if (need_wave_dangers_updated || wave.getDangerPoints() == null) {
                wave.setDangerPoints(getDangerFactors(wave));
            }
        }
        
        for (EnemyWave wave : ewm.getAllHeat()) {
            if (need_wave_dangers_updated || wave.getDangerPoints() == null) {
                wave.setDangerPoints(getDangerFactors(wave));
            }
        }
        
        need_wave_dangers_updated = false;
    }
    
    // Returns a list of weighted dangerous guessfactors
    public double[][] getDangerFactors(EnemyWave wave) {
        String sourcename = wave.getSource().getName();
        if (!stats.containsKey(sourcename))
            return new double[2][0];
        
        double[] p = getSegmentation(wave);
        List<KdTree.Entry<Double>> l = KdTree.nearestNeighbor(stats.get(sourcename), p, 20);
        
        double nweight = (50/l.size());
        
        double ptotal = 0;
        
        double[][] list = new double[2][l.size()];
        for (int i=0; i<l.size(); i++) {
            list[0][i] = l.get(i).value;
            
            double pweight = 1+Math.sqrt(l.get(i).distance);
            ptotal += pweight;
            
            list[1][i] = nweight/pweight;
        }
        ptotal /= l.size();
        for (int i=0; i<l.size(); i++) {
            list[1][i] /= ptotal;
        }
        
        
        // TODO: Add interpolation-focusing!
        
        return list;
    }
    
    
    // JGAP SEGMENTATION RESEARCH:
    // Lat:0.49533494812047046 Adv:0.47888435173735894 Dist:0.024915706369010034 Time:8.64993773160576E-4
    // USED BELOW
    private static final int segments = 2;
    private static final double[] segmentweights = {0.49533494812047046, 0.47888435173735894, 0.024915706369010034, 8.64993773160576E-4};
    private double[] getSegmentation(EnemyWave wave) {
        StatusRobot self = wave.getStaticTarget();
        double[] segmentdata = new double[segments];
        
        double lateralVelocity = (self.getVelocity().magnitude * Math.sin(self.getVelocity().getDirection() - wave.getHOTAngle()));
        double advancingVelocity = (self.getVelocity().magnitude * -Math.cos(self.getVelocity().getDirection() - wave.getHOTAngle()));
        // double distance = self.getLocation().distance(wave.getOrigin());
        
        segmentdata[0] = Math.abs(lateralVelocity);
        segmentdata[1] = advancingVelocity;
        //segmentdata[2] = distance;
        //segmentdata[3] = ((double)(stateholder.getTotalTime()-stateholder.getTime())+self.getTime());
        
        // Apply segment weights!
        for (int x = 0; x < segments; x++)
            segmentdata[x] *= segmentweights[x];
        
        return segmentdata;
    }
    private void hitAt(EnemyWave wave, AbsolutePoint bulletpos) {
        double guessfactor = wave.getGuessFactor(bulletpos);
        
        String sourcename = wave.getSource().getName();
        double[] p = getSegmentation(wave);
        if (!stats.containsKey(sourcename))
            stats.put(sourcename, new KdTree<Double>(segments));
        KdTree.addPoint(stats.get(sourcename), p, guessfactor);
        
        // The wave dangers may need updating now
        need_wave_dangers_updated = true;
        
        hits++;
    }
    
    // Handle a bullet hitting us
    private void handleHit(HitByBulletEvent hit) {
        //Bullet bullet = hit.getBullet();
        //EnemyWave wave = findMatchingWave(bullet);
        //if (wave != null) {
            //String message = "GuessFactor: "+wave.getGuessFactor(AbsolutePoint.fromXY(bullet.getX(), bullet.getY()));
            //String message2 = "GF range: "+ wave.getNewHitRange() + "   Radius:" + wave.getRadius() + "   Power:" + wave.getPower();
            //stateholder.writeImage(hit.getBullet(), message, message2);
        //}
        handleHit(hit.getBullet());
    }
    
    // Handle a bullet hitting one of ours
    private void handleHit(BulletHitBulletEvent hit) {
        handleHit(hit.getHitBullet());
    }
    
    // Handle a bullet hitting something
    private void handleHit(Bullet bullet) {
        EnemyWave wave = findMatchingWave(bullet);
        // If there's no match? Oh well then, nothing to do with this hit
        if (wave != null) {
            AbsolutePoint bulletpos = AbsolutePoint.fromXY(bullet.getX(), bullet.getY());
            
            // Record data!
            rec.recordSituation(wave, bullet);
            
            // Deal with the bullet hitting us at a given point!
            hitAt(wave, bulletpos);
            
            // Remove the old wave
            ewm.remove(wave);
        } else {
            // Warn that we were hit by a bullet we didn't know about!
            System.out.println("Warning: Hit not matched!");
        }
    }
    
    private EnemyWave findMatchingWave(Bullet bullet) {
        AbsolutePoint bulletpos = AbsolutePoint.fromXY(bullet.getX(), bullet.getY());
        List<EnemyWave> waves = ewm.getAll();
        
        for (EnemyWave wave: waves) { 
            // If it's not from the right bot, it's wrong
            if (wave.getSource().getName() != bullet.getName())
                continue;
            // Not our wave if power is wrong
            if (Math.abs(wave.getPower() - bullet.getPower()) > 0.01)
                continue;
            // If the wave is not over the bullet, it's wrong
            if (!wave.isNear(bulletpos, 22))
                continue;
            if (!isNear(wave.getAngleTo(bulletpos),bullet.getHeadingRadians()))
                continue;
            return wave;
        }
        return null;
    }
}
