package ags.rougedc.movement;

import java.util.List;
import ags.utils.points.*;

import ags.rougedc.base.*;
import ags.rougedc.robots.VirtualRobot;
import ags.rougedc.waves.EnemyWave;
import static robocode.util.Utils.normalAbsoluteAngle;
import static robocode.util.Utils.normalRelativeAngle;

/**
 * Drives perpendicular, goes to a goal distance
 * 
 * @author Alexander Schultz
 */
public class OrbitDriver implements RobotDriver {
    final private Rules rules;
    final private double diroffset;
    final private double goaldistance;
    private double turn;
    private double acceleration;
    
    public OrbitDriver(Rules rules, double goaldistance, double diroffset) {
        this.rules = rules;
        this.diroffset = diroffset;
        this.goaldistance = goaldistance;
    }

    public void run(VirtualRobot bot, List<EnemyWave> waves) {
        WallSmoother smoother = new WallSmoother(rules);
        
        if (waves.size() == 0) {
            turn = 0;
            acceleration = -bot.getVelocity().magnitude;
            return;
        }
        
        RelativePoint origin = RelativePoint.fromPP(bot.getLocation(), waves.get(0).getOrigin());
        
        // Add a distancing factor!
        double distancing_factor=1.0;
        distancing_factor -= Math.min(Math.max((origin.magnitude - goaldistance)/200, -0.5), .5);
        
        //System.out.println("Distancing factor: "+distancing_factor);
        
        double direction = normalAbsoluteAngle(origin.direction+distancing_factor*diroffset);
        
        direction = smoother.smoothAngle(bot.getLocation().x, bot.getLocation().y, direction, -(int)Math.signum(diroffset));
        //direction = smoother.smoothAngle2(bot.getLocation(), direction, bot.getVelocity());
        
        turn = normalRelativeAngle(direction - bot.getVelocity().direction);
        
        // Back as front
        if (Math.abs(turn) > Math.PI/2) {
            turn = normalRelativeAngle(turn + Math.PI);
            acceleration = Double.NEGATIVE_INFINITY;
        } else {
            acceleration = Double.POSITIVE_INFINITY;
        }
    }
    
    public double getAcceleration() {
        return acceleration;
    }

    public double getAngularVelocity() {
        return turn;
    }

}
