package sgp;
import robocode.*;
import java.util.*;
import java.awt.Color;

/**
 * MadHatterStrategy - a class by Simon Parker
 This class does the MadHatter 4.13 robot strategy
 */
public class MadHatterStrategy extends Strategy
{
	private final String moduleName = "-MH-";
	JollyNinja robot = null;
	boolean isLocked = false;
	int unsuccessfulScanCounter = 0;
//	String targetName;
//
//	double currentTargetPosX = 200;
//	double currentTargetPosY = 200;
//	double currentTargetHeading = 0;
//	double currentTargetVelocity = 0;
//
//
//	double lastTargetHeading_deg = 0;
//	double currentTargetAnglularVelocity = 0;

	double currentVelocity = 0;

	private int bulletHitCount = 0;
	private int bulletMissCount = 0;

	double MOVE_DISTANCE = 150;

	double OPTIMUM_TARGET_DISTANCE = 200.0;
	double WALL_DEFLECTION_DISTANCE = 100.0;
	double POSSIBLE_MOVEMENT_RANGE = 60;
	double MIN_MOVE_DISTANCE = 120;
	double WALL_DISTANCE_WEIGHT = 3.0;
	double TARGET_BEARING_WEIGHT_1_ON_1 = 0.1;

	final double TARGET_BEARING_WEIGHT_MELEE = 0.0;
	final double CENTRE_DISTANCE_WEIGHT_MELEE = 1.0;
	final double CENTRE_DISTANCE_WEIGHT_1_ON_1 = 0.0;


	final int MAX_VELOCITY = 8;
	final double MIN_BULLET_POWER = 0.5;
	final double MAX_BULLET_POWER = 3.0;
	final int MAX_SCAN_FAIL_COUNT = 2;
	public static double ROBOT_RADIUS = 45.0/2;
	final double ANGULAR_VELOCITY_DEG_PER_TURN = 10.0;
	final double RADAR_TURN_ANGLE = 40;

	private JiggleAnalyser jiggleAnalyser = null;
	private double previousHeading = 0;
	private Coordinate currentPosition = new Coordinate();
	
	private Target currentTarget = null;
	

	public void reset()
	{
		super.reset();
		OPTIMUM_TARGET_DISTANCE = robot.parameters.optimumDuelTargetDistance.getValue();
		WALL_DEFLECTION_DISTANCE = robot.parameters.wallDeflectionDistance.getValue();
		POSSIBLE_MOVEMENT_RANGE = robot.parameters.possibleMovementRange.getValue();
		MIN_MOVE_DISTANCE = robot.parameters.minimumMoveDistance.getValue();
		ROBOT_RADIUS = robot.parameters.robotRadius.getValue();
		WALL_DISTANCE_WEIGHT = robot.parameters.mhWallDistanceWeight.getValue();
		TARGET_BEARING_WEIGHT_1_ON_1 = robot.parameters.mhTargetBearingWeight.getValue();
		
		if (currentTarget != null)
		{
			currentTarget.reset();
		}
	}

	public MadHatterStrategy(StrategyManager theStrategyManager)
	{
		super(theStrategyManager);
		robot = JollyNinja.getInstance();
		jiggleAnalyser = new JiggleAnalyser();
	}

	//any init that needs doing at the start of the turn
	public void startTurn()
	{
		robot.angularVelocity = normalRelativeAngle(robot.getHeading() - previousHeading);
		currentPosition.set(robot.getX(), robot.getY());
		robot.jiggleAnalyser.update(currentPosition, robot.getVelocity(), robot.getHeading());

		if (unsuccessfulScanCounter < 0)
		{
			unsuccessfulScanCounter = 0;
		}

		if (robot.getOthers() < 2)
		{
			strategyManager.setStrategyId(strategyManager.EVASION_STRATEGY_ID);
		}
	}

	public void endTurn()
	{
		previousHeading = robot.getHeading();
	}

	public void setMovement() //sets the turning angle and the ahead/back movement
	{
		if ( robot.getDistanceRemaining() == 0 )
		{
			MOVE_DISTANCE = Math.random() * POSSIBLE_MOVEMENT_RANGE + MIN_MOVE_DISTANCE;

			if (currentVelocity <= 0)
			{
				robot.setAhead(MOVE_DISTANCE);
				currentVelocity = MAX_VELOCITY;
			}
			else
			{
				robot.setBack(MOVE_DISTANCE);
				currentVelocity = -MAX_VELOCITY;
			}
		}

		double predictedPosX = robot.getX() + currentVelocity * Math.sin(Math.toRadians(robot.getHeading() + ANGULAR_VELOCITY_DEG_PER_TURN));
		double predictedPosY = robot.getY() + currentVelocity * Math.cos(Math.toRadians(robot.getHeading() + ANGULAR_VELOCITY_DEG_PER_TURN));
		Coordinate turnRightPosition = new Coordinate(predictedPosX, predictedPosY);

		predictedPosX = robot.getX() + currentVelocity * Math.sin(Math.toRadians(robot.getHeading() - ANGULAR_VELOCITY_DEG_PER_TURN));
		predictedPosY = robot.getY() + currentVelocity * Math.cos(Math.toRadians(robot.getHeading() - ANGULAR_VELOCITY_DEG_PER_TURN));
		Coordinate turnLeftPosition = new Coordinate(predictedPosX, predictedPosY);

		if (currentTarget == null) return;
		
		double leftEval = evaluatePosition(turnLeftPosition, currentTarget.position, (robot.getHeading() + ANGULAR_VELOCITY_DEG_PER_TURN), currentTarget.heading_deg);
		double rightEval = evaluatePosition(turnRightPosition, currentTarget.position, (robot.getHeading() - ANGULAR_VELOCITY_DEG_PER_TURN), currentTarget.heading_deg);

		if (leftEval >= rightEval)
		{
			robot.setTurnLeft(360);
		}
		else
		{
			robot.setTurnRight(360);
		}
	}

	//control the radar motion
	public void setScan()
	{
		if (isLocked)
		{
			if (unsuccessfulScanCounter > MAX_SCAN_FAIL_COUNT)
			{
				unsuccessfulScanCounter = 0;
				isLocked = false;

				//???robot.out.println("Lost Lock");
				return;
			}

			unsuccessfulScanCounter++;
			robot.scan();

			Coordinate robotPosition = new Coordinate(robot.getX(), robot.getY());

			double targetRadarAngle = robotPosition.headingTo(currentTarget.position);
			double radarTurnRightAngle = normalRelativeAngle(targetRadarAngle - robot.getRadarHeading());

			if (radarTurnRightAngle > 0)
			{
				radarTurnRightAngle += 20;
			}
			else
			{
				radarTurnRightAngle -= 20;
			}

			robot.setTurnRadarRight(radarTurnRightAngle);

		}
		else
		{
			// look for new robots
			robot.setTurnRadarRight(720);
		}
	}
	public void setGunRotation() //control the gun motion and fires if neccessary
	{
		if (currentTarget == null) return;
		if (currentTarget.position == null) return;		
		Coordinate estimatedTargetPosition = currentTarget.getEstimatedPosition(robot.getTime());
		double targetDistance = currentPosition.distanceFrom(estimatedTargetPosition);

		double bulletPower = calculateBulletPower(currentPosition, estimatedTargetPosition);

		Intercept intercept = currentTarget.gunManager.getIntercept(currentPosition, bulletPower);
		if (intercept == null) 
		{
			robot.out.println("null intercept");
			return;
		}

		double turnAngle = normalRelativeAngle(intercept.bulletHeading_deg - robot.getGunHeading());
		// move gun to target angle
		robot.setTurnGunRight(turnAngle);

		if (Math.abs(turnAngle) <= intercept.angleThreshold)
		{
robot.out.print("d (" + intercept.impactPoint.x + "," + intercept.impactPoint.y + ")");
			if 	(
					(intercept.impactPoint.x > 0) &&
					(intercept.impactPoint.x < robot.getBattleFieldWidth()) &&
					(intercept.impactPoint.y > 0) &&
					(intercept.impactPoint.y < robot.getBattleFieldHeight())
				)
			{
robot.out.print("e");
				if (
					(robot.getGunHeat() == 0) && 
					(robot.getEnergy() > bulletPower)// && 
					//(currentTarget.getActiveBulletCount() < 7) 
					)
				{
robot.out.print("f");
					currentTarget.fire(bulletPower);
				}
			}
robot.out.println("|");
		}
		


		
//		
//		Intercept intercept;
//
//		if (jiggleAnalyser.isUpdated)
//		{
//			//robot.out.println("Jiggle " + jiggleAnalyser.isJiggling + ", Osc " + jiggleAnalyser.isOscilating);
//		}
//
//		if (jiggleAnalyser.isJiggling)
//		{
//			intercept = new JiggleIntercept(jiggleAnalyser);
//		}
//		else
//		{
//			intercept = new CircularIntercept();
//		}
//
//		// calculate target angle
//		double posX = robot.getX();
//		double posY = robot.getY();
//		double dX = currentTargetPosX - posX;
//		double dY = currentTargetPosY - posY;
//
//
//		double bulletPower = calculateBulletPower
//		(
//			posX, posY, robot.getGunHeading(),
//			currentTargetPosX, currentTargetPosY, currentTargetHeading,
//			currentTargetVelocity
//		);
//
//		intercept.calculate
//		(
//			posX, posY, currentTargetPosX, currentTargetPosY,
//			currentTargetHeading, currentTargetVelocity, bulletPower, currentTargetAnglularVelocity
//		);
//
//		double turnAngle = normalRelativeAngle(intercept.bulletHeading_deg - robot.getGunHeading());
//		// move gun to target angle
//		robot.setTurnGunRight(turnAngle);
//		if 	(Math.abs(turnAngle) <= intercept.angleThreshold)
//		{
//			if 	(
//					(intercept.impactPoint.x < 0) ||
//					(intercept.impactPoint.x > robot.getBattleFieldWidth()) ||
//					(intercept.impactPoint.y < 0) ||
//					(intercept.impactPoint.y > robot.getBattleFieldHeight())
//				)
//			{
//				//impact point off the field
//				//robot.out.println("T " + intercept.impactTime + ", x " + intercept.impactPoint.x + ", y " + intercept.impactPoint.y);
//			}
//			else
//			{
//				if (isLocked)
//				{
//					robot.fire(bulletPower);
//				}
//			}
//		}
//		else
//		{
//			// gun not in position
//			//robot.out.println("ta " + turnAngle + ", th " + intercept.angleThreshold);
//		}

	}

	public void onHitWall(HitWallEvent e)
	{
		if (Math.abs(e.getBearing()) >= 90)
		{
			robot.setAhead(MOVE_DISTANCE);
			currentVelocity = MAX_VELOCITY;
		}
		else
		{
			robot.setAhead(-MOVE_DISTANCE);
			currentVelocity = -MAX_VELOCITY;
		}
		robot.execute();
	}

	public void onHitRobot(HitRobotEvent e) {
		if (e.getBearing() > -90 && e.getBearing() < 90)
		{
			robot.setBack(MOVE_DISTANCE);
			currentVelocity = -MAX_VELOCITY;
		}
		else
		{
			robot.setAhead(MOVE_DISTANCE);
			currentVelocity = MAX_VELOCITY;
		}
		robot.execute();
	}

	public void onScannedRobot(ScannedRobotEvent e)
	{
		unsuccessfulScanCounter--;

		Coordinate robotPosition = new Coordinate(robot.getX(), robot.getY());

		double distance = e.getDistance();
		
		if (currentTarget != null)
		{
			if (
				(currentTarget.name != e.getName()) && 
				(distance > robotPosition.distanceFrom(currentTarget.position)) && 
				isLocked
			   )
			{
				return;
			}
		}

		boolean isNewTarget = true;
		if (currentTarget != null) 
		{
			isNewTarget = (currentTarget.name != e.getName());
		}
		
		if (isNewTarget) 
		{
			currentTarget = new Target(moduleName, e.getName(), robot);
			currentTarget.reset();
			//???robot.out.println("Locked onto " + currentTarget.name);
		}

		isLocked = true;

		currentTarget.update(e, robot.getTime());
	}

	public void onHitByBullet(HitByBulletEvent e)
	{

	}
	public void onDeath(DeathEvent event)
	{
	}

	private double evaluatePosition(Coordinate robotPosition, Coordinate targetPosition, double robotHeading, double targetHeading)
	{

		double dX = targetPosition.x - robotPosition.x;
		double dY = targetPosition.y - robotPosition.y;
		double distance = targetPosition.distanceFrom(robotPosition);

		double targetGoodness = - Math.abs(distance - OPTIMUM_TARGET_DISTANCE);

		double wallGoodnessX = 0;
		double wallGoodnessY = 0;
		if (robotPosition.x < WALL_DEFLECTION_DISTANCE)
		{
			wallGoodnessX = -WALL_DISTANCE_WEIGHT * (WALL_DEFLECTION_DISTANCE - robotPosition.x);
		}

		if (robotPosition.y < WALL_DEFLECTION_DISTANCE)
		{
			wallGoodnessY = -WALL_DISTANCE_WEIGHT * (WALL_DEFLECTION_DISTANCE - robotPosition.y);
		}

		if (robotPosition.x > (robot.getBattleFieldWidth()-WALL_DEFLECTION_DISTANCE))
		{
			wallGoodnessX = WALL_DISTANCE_WEIGHT * (robot.getBattleFieldWidth()-WALL_DEFLECTION_DISTANCE - robotPosition.x);
		}

		if (robotPosition.y > (robot.getBattleFieldHeight()-WALL_DEFLECTION_DISTANCE))
		{
			wallGoodnessY = WALL_DISTANCE_WEIGHT * (robot.getBattleFieldHeight()-WALL_DEFLECTION_DISTANCE - robotPosition.y);
		}

		double bearing_deg = normalRelativeAngle(robotHeading-Math.toDegrees(Math.atan2(dX,dY)));

		double bearingWeight = TARGET_BEARING_WEIGHT_MELEE;
		if (robot.getOthers() == 1) bearingWeight = TARGET_BEARING_WEIGHT_1_ON_1;

		double bearingGoodness = - bearingWeight * Math.abs(Math.abs(Math.abs(bearing_deg) - 90) -90);

		Coordinate centrePosition = new Coordinate(robot.getBattleFieldWidth()/2, robot.getBattleFieldHeight()/2);
		double centreDistanceWeight = CENTRE_DISTANCE_WEIGHT_MELEE;
		if (robot.getOthers() == 1) centreDistanceWeight = CENTRE_DISTANCE_WEIGHT_1_ON_1;
		double centreGoodness = robotPosition.distanceFrom(centrePosition) * centreDistanceWeight;


		return targetGoodness + wallGoodnessX + wallGoodnessY + bearingGoodness + centreGoodness;
	}

	private double calculateBulletPower
	(
		Coordinate robotPosition,
		Coordinate targetPosition
	)
	{
		double distance = robotPosition.distanceFrom(targetPosition);

		double power = MAX_BULLET_POWER - (distance-2*ROBOT_RADIUS)/300.0;
		power = Math.max(MIN_BULLET_POWER, power);
		power = Math.min(MAX_BULLET_POWER, power);
		//return 0.1;
		return power;
	}


}
