package gh.mini;
import robocode.*;
import robocode.util.Utils;
import java.awt.Color;
import java.awt.geom.*;
/**
 * GrubbmOne - a robot by Gert Heijenk
 *
 * Spinning. Tracks and fires at the nearest robot it sees
 *
 * Revision information:
 * v0.1  20040910 Just putting sample.Spinbot and sample.TrackFire together
 * v0.2  20040915 Introducing 'scantimeout', better 'onHitRobot' handling
 * v0.3  20040916 Introducing 'wallmove'-statemachine, better firepower on 'easy' targets
 * v0.4  20040919 Introducing 'scan'-statemachine, some tweaks in targetselection and 'onHitRobot'
 * v0.5  20040926 Solved some bugs in wallmove (e.g. not moving), rescanning from time to time.
 *                First time entry in RoboRumble !
 * v0.6  20041011 Added linear and circular targetting, solved 'time'bugs in events. Got it fit in a mini !
 * v0.7  20041014 Small bug in gun solved, some tweaking of bulletpower.
 * v0.8  20041020 Try to do wall-avoidance instead of handling wall-hit, variate turnangle and speed
 * v0.9  20041026 Simple circular movement again, adjusted bulletpower
 * v0.10 20050107 Improved melee-radar, improved onhitrobot-handling, improved scanning and locking opponents
 * v0.11 20050228 Refactored CT-gun.
 * v1.0  20050303 Gunbug solved. Final version. End of Development.
 */
public class GrubbmOne extends AdvancedRobot
{
	private final long SCANTIMEOUT	= 8;	// the number of ticks I received no scanevent (lock is lost)
	private long RESCANTIMEOUT	= 120;		// the number of ticks before a complete rescan
		
	// the states necessary to find and lock targets
	private final int SCANNING	= 1;	// scan all opponents and pick a target
	private final int LOCKING	= 2;	// I determined a target, find it !
	private final int LOCKED	= 3;	// I found the target, trying to keep the lock
	private final int LOCKLOST	= 4;	// no lock for #SCANTIMEOUT ticks, start scanning 

	// wall-avoidance variables
	private final int WALLAVOID = 110;	// the area to avoid near walls
	private boolean restartMove = true;	// true if default movement must be restarted
	private double	newSpeed;			// the speed set by wall-avoidance
	private double	posx, posy;			// my current position (mainloop)
	private double	myHeading;			// my current heading
	private int     direction = 1;		// the direction I am moving (forward or backward)
	private final double	defSpeed = 8;
	
	// scanning and locking variables
	private long	lastscan;		// the time the last robotscanned_event was received
	private long	lastrescan;		// the time the last rescan was done
	private int     lockState = LOCKLOST;	// the initial state of finding and locking targets

	// some info about the locked opponent
	private double	oldHeading;			// the previous heading of the selected opponent
	private double	bearingOpponent;	// the bearing of the selected opponent
	private double  importanceOpponent;	// the importance of the opponent
	private String  nameOpponent;		// the name of the selected opponent

	// statistics counters
//	static	int		bullhit;		// the number of bullets that hit
//	static	int     bullmis;		// the number of bullets that missed
//	static	int     wallhit;		// the number of times I hit the wall
//	static	int		skipturn;		// the number of skipped turns

	// some statistics
	private double	fieldWidth;				// the width of the battlefield
	private double	fieldHeight;			// the heigth of the battlefield
	private Rectangle2D.Double fireField;	// the shootable battlefield

	/**
	 * run: Grubbm_One's default behavior
	 */
	public void run() {

		// Give the robot an appealing look
		setColors( Color.red, Color.yellow, Color.red);

		// First set the statistics
		fieldWidth = getBattleFieldWidth();
		fieldHeight = getBattleFieldHeight();
        fireField = new Rectangle2D.Double( 17, 17, fieldWidth - 34, fieldHeight - 34);

		// Find out howmany opponents are left, one opponent means (semi)radarlock.
		if (1 == getOthers()) RESCANTIMEOUT = 9999;

		// Let gun and radar move independently
		setAdjustGunForRobotTurn( true);
		setAdjustRadarForGunTurn( true);

		while( true ) {
		    // I have seemed to lost my lock, just find any opponent
			if ((lockState == LOCKED) && ((getTime() - lastscan) > SCANTIMEOUT)) {
				lockState = LOCKLOST;
			}

		    // Occasionally find a better opponent ONLY IN MELEE ! !
			if (((getTime() - lastrescan) > RESCANTIMEOUT) && (getGunHeat() > 0.8)) {
				lockState = LOCKLOST;
			}

			// Check if scanning nearly ended
			if ((lockState == SCANNING) && (getRadarTurnRemaining() < 44)) {
				doSomeLocking();
			}
			// I am in need of an(other) opponent, find one !
			if (lockState == LOCKLOST) {
				doSomeScanning();
			}

			// perform some wall-avoidance by playing with the speed
			myHeading = getHeading();
			if (-1 == direction) myHeading = (myHeading + 180) % 360;
			newSpeed = defSpeed;
			posx = getX();			// Remember current position (to avoid getXXX calls)
			posy = getY();
			if (posx < WALLAVOID) {
				if (myHeading > 180) newSpeed = 4;
			}
			if (posx > (fieldWidth - WALLAVOID)) {
				if (myHeading < 180) newSpeed = 4;
			}
			if (posy < WALLAVOID) {
				if ((myHeading > 90) && (myHeading < 270)) newSpeed = 4;
			}
			if (posy > (fieldHeight - WALLAVOID)) {
				if ((myHeading > 270) || (myHeading < 90)) newSpeed = 4;
			}
			setMaxVelocity( newSpeed);

			// restart default moving if requested
			if (restartMove == true) {
				setTurnRight( Double.POSITIVE_INFINITY);
				setAhead( direction * Double.POSITIVE_INFINITY);
				restartMove = false;
			}

			// perform all actions previously set
			execute();
		}
	}

	/**
	 * onScannedRobot: What to do when you see another robot
	 */
	public void onScannedRobot(ScannedRobotEvent e) {

		// prevent a 'lock lost' event
		lastscan = getTime();

		double power;
		double tmpBearing = getHeadingRadians() + e.getBearingRadians();
		double bearingFromGun = Math.toDegrees( Utils.normalRelativeAngle( tmpBearing - getGunHeadingRadians()));
		double importance;
		boolean enemyswitch = false;

		// if scanning, just try to get the best fit opponent !
		if (1 == getOthers()) {
			nameOpponent = e.getName();
			bearingOpponent = bearingFromGun;
			lockState = LOCKED;
		}
		else {
			importance = 200;
			if (e.getDistance() < 500)
				importance = 800 - e.getDistance();
			importance -= Math.abs(bearingFromGun);
			if (e.getEnergy() < 16) {
				importance += 100;
				if (e.getEnergy() < 0.1)
					importance += 75;
			}
			if (e.getVelocity() == 0)
			    importance += 75;
			if ((nameOpponent != null) && (nameOpponent.equals(e.getName()))) {
				importanceOpponent = importance;
			}
			else if (importance > importanceOpponent) {
				nameOpponent = e.getName();
				bearingOpponent = bearingFromGun;
				importanceOpponent = importance;
				enemyswitch = true;
			}
		}
		// if scanning, just try to select the best fit opponent !
		if (lockState == SCANNING)
			return;

		// if not desired target, forget event
		if (nameOpponent.equals(e.getName())) {
			lockState = LOCKED;
		}
		else {
			return;
		}

		// Always turn the radar with a nice widening angle
		setTurnRadarRightRadians( 2.2 * Utils.normalRelativeAngle( tmpBearing - getRadarHeadingRadians()));

		// set the power of the bullet
		if (e.getDistance() > 850)
			power = 0.1;
		else if (e.getDistance() > 700)
		    power = 0.49;
		else if (e.getDistance() > 200)
		    power = 1.9;
		else
			power = 3.0;
		power = Math.min( getEnergy()/5, Math.min( (e.getEnergy()/4) + 0.1, power));	// energy management

		// perform lineair and circular targetting
		long deltahittime;
		Point2D.Double point = new Point2D.Double();
		double myX;
		double myY;
		double head, chead, speed, bspeed;
		double tmpx, tmpy;

		// perform an iteration to find a reasonable accurate expected position
		tmpx = (myX = getX()) + (Math.sin( tmpBearing) % (2 * Math.PI)) * e.getDistance();
		tmpy = (myY = getY()) + (Math.cos( tmpBearing) % (2 * Math.PI)) * e.getDistance();
		head = e.getHeadingRadians();
		chead = head - oldHeading;
		speed = e.getVelocity();
		point.setLocation( tmpx, tmpy);
		deltahittime = 0;
		do {
			tmpx += Math.sin( head) * speed;
			tmpy +=	Math.cos( head) * speed;
			head += chead;
			deltahittime++;
			// if position not in field, adapt bulletpower
			if (!fireField.contains( tmpx, tmpy)) {
				bspeed = point.distance( myX, myY) / deltahittime;
				if (bspeed < 19.7) power = (20 - bspeed) / 3.0;
				else if ((point.distance( myX, myY) / 19.7) > deltahittime + 5) power = 0.0;
				else power = 0.1;
				break;
			}
			point.setLocation( tmpx, tmpy);
		} while ( (int)Math.round( point.distance( myX, myY) / (20 - (3 * power))) > deltahittime);

		// save some info for next time
		oldHeading = e.getHeadingRadians();

		// Turn gun
		bearingFromGun = Utils.normalRelativeAngle(((Math.PI / 2) - Math.atan2( point.y - myY, point.x - myX)) - getGunHeadingRadians());
		setTurnGunRightRadians( bearingFromGun);
		// Do not fire if it is no use or if it disables yourself
		if ((getGunHeat() == 0.0) && (Math.abs( bearingFromGun) < 0.17) && (power > 0.09) && (getEnergy() > 0.1) && (enemyswitch == false))
		{
			// Only fire the gun when less than 10 degrees to turn.
			setFire( power);
		}
	}

	/**
	 * onRobotDeath: What to do when someone else dies
	 */
	public void onRobotDeath(RobotDeathEvent e) {
		// If current target dies, force to find another
		if (nameOpponent.equals( e.getName())) {
			lockState = LOCKLOST;
		}
	}

	/**
	 * onHitRobot:  Bounce off!
	 */	
	public void onHitRobot(HitRobotEvent e) {
		// if I hit someone, just bounce off, ramming is not a good idea when in melee.
		if (e.isMyFault()) {
			restartMove = true;
			direction = -direction;
		}
	}

	/**
	 * onHitWall:  Handle collision with wall.
	 */
	public void onHitWall(HitWallEvent e)
	{
		// 'Slide' off!
//		wallhit++;
		restartMove = true;
	}

	/**
	 * onSkippedTurn:  Handle a skipped turn.
	 */
//	public void onSkippedTurn(SkippedTurnEvent e)
//	{
//		skipturn++;
//		out.println("AAaaaarghhhhh");
//	}

	/**
	 * onWin: Show my private victory dance
	 */
	public void onWin(WinEvent e)
	{
		//Victory dance	
		setTurnGunRight( Double.POSITIVE_INFINITY);
		setTurnRadarLeft( Double.POSITIVE_INFINITY);
		setTurnRight(10);
		ahead(10);
//		out.println("Bullets HIT:" + bullhit);
//		out.println("Bullets mis:" + bullmis);
//		out.println("Percentage :" + (bullhit*100)/(bullhit + bullmis));
//		out.println("Wall hits  :" + wallhit);
//		out.println("SkipTurns  :" + skipturn);
		waitFor(new RadarTurnCompleteCondition(this));
	}

	/**
	 * onDeath: Show some statistics
	 */
//	public void onDeath(DeathEvent e)
//	{
//		// Show some statistics	
//		out.println("Bullets HIT:" + bullhit);
//		out.println("Bullets mis:" + bullmis);
//		out.println("Percentage :" + (bullhit*100)/(bullhit + bullmis));
//		out.println("Wall hits  :" + wallhit);
//		out.println("SkipTurns  :" + skipturn);
//	}

	/**
	 * onBulletHit: Yes, I hit somebody
	 */
//	public void onBulletHit(BulletHitEvent e)
//	{
//		bullhit++;
//	}

	/**
	 * onBulletMissed: Damn, I missed (again)
	 */
//	public void onBulletMissed(BulletMissedEvent e)
//	{
//		bullmis++;
//	}

	/**
	 * doSomeScanning: scan around to find the 'best fit' opponent
	 */
	public void doSomeScanning() {
		lockState = SCANNING;
		importanceOpponent = 0;
		lastrescan = lastscan;
		setTurnRadarRight( 360);
	}

	/**
	 * doSomeLocking: turn gun to 'best fit' opponent
	 */
	public void doSomeLocking() {
		lockState = LOCKING;
		if (bearingOpponent < 0) {
			setTurnRadarRight( Double.NEGATIVE_INFINITY);
		}
		else {
			setTurnRadarRight( Double.POSITIVE_INFINITY);
		}
	}
}
