package kawigi.twin;
import robocode.*;
import java.util.*;
import java.awt.geom.*;
import java.awt.*;
import java.io.*;

/*  *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *
 * Mario - a Twin-Duel bot by Kawigi.                                                                       *
 *  *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *
 * The structure of this code as well as most of the targeting algorithm is from my own Coriantumr.  The    *
 * ideas behind the random orbit code that is implicit in this code is from Jamougha's Raiko.               *
 *  *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *
 * Version 1.0 - First release of the MarioBros TwinDuel team.  The idea is that it's a MinimumRiskMovement *
 *      implementation of Raiko's random orbit algorithm, combined with a GuessFactorTargeting algorithm    *
 *      that is somewhere between FloodMini's and Coriantumr's.  Also includes some team communication, and *
 *      a synchronized victory dance.                                                                       *
 *  *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *
 * This code is released under the terms of the KPL (http://robowiki.net/perl/robowiki?KawigiPublicLicense)	*
 * For more information, visit the RoboWiki (http://robowiki.net/perl/robowiki?MarioBros)                   *
 *  *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   *   */
public class Mario extends TeamRobot
{
	static Point2D myLocation;
	static EnemyInfo currentTarget;
	static boolean flip;
	//having one permanent hash map and one one-round-only hashmap for
	//different types of enemy data seems to be very clean.  I might even
	//do it in the next release of FloodHT...
	static HashMap enemies;
	static HashMap stats = new HashMap();
	
	public void run()
	{
		// Mario colors :-)
		setColors(Color.red, Color.blue, Color.blue);
		setAdjustGunForRobotTurn(true);
		setTurnRadarRight(Double.POSITIVE_INFINITY);
		//meant to be reinitialized every round.  Wouldn't even be static except that they're smaller that way.
		enemies = new HashMap();
		Point2D next = currentTarget = null;
		do
		{
			myLocation = new Point2D.Double(getX(), getY());
			if (currentTarget != null)
			{
				double angle;
				flip = Math.random() > Math.pow(angle = 0.6 * currentTarget.bulletVelocity/myLocation.distance(currentTarget), angle);
				angle = 0;
				//sometimes I'll change my point here, sometimes I won't.  It's a big mystery!
				do
				{
					Point2D p;
					if (inField(p = projectPoint(myLocation, angle, Math.min(170, myLocation.distance(currentTarget)*.5))) && (next == null || findRisk(p) < findRisk(next)))
						next = p;
					angle += .1;
				}
				while (angle < Math.PI*2);
				//Thanks, David, for a quick way of doing this:
				double a1, a2;
				setTurnRightRadians(a1 = Math.atan(Math.tan(a2 = robocode.util.Utils.normalRelativeAngle(angle(next, myLocation) - getHeadingRadians()))));
 				setAhead((Math.abs(getTurnRemainingRadians()) > 1 ? 1 : next.distance(myLocation)) * (a1 == a2 ? 1.0 : -1.0));
			}
			flip = false;
			execute();
		}
		while (true);
	}
	
	private double findRisk(Point2D point)
	{
		// The real purpose to repelling my current location is to encourage me to look ahead a little bit.
		double risk = 4/myLocation.distanceSq(point);
		Collection enemySet;
		Iterator it = (enemySet = enemies.values()).iterator();
		double toangle = angle(myLocation, point);
		do
		{
			EnemyInfo e = (EnemyInfo)it.next();
			//start with an anti-gravity-type value:
			double myenergy, eenergy;
			double thisrisk = Math.max(eenergy = e.energy, myenergy = getEnergy())/point.distanceSq(e);
			
			// This section is supposed to determine if an enemy is likely to be targeting me.  If this is an
			// enemy, and I am the closest enemy to this enemy, or if I'm targeting them, or if they have hit
			// me in the last 200 ticks, they are likely targeting me.
			boolean closest = true;
			Iterator it2 = enemySet.iterator();
			EnemyInfo e2;
			do
				if (e.distance(e2 = (EnemyInfo)it2.next())*.9 > e.distance(point) && e.isTeammate != e2.isTeammate)
					closest = false;
			while (it2.hasNext());
			
			// if they are likely targeting me, multiply by some factor that has to do with "perpendicularity"
			// (I think I remember David Alves calling it that).  Note that the section at the beginning also
			// causes an attraction to my current target if I have an energy advantage on it.
			if (!e.isTeammate && (closest || e.lastHit > getTime()-200 || e == currentTarget))
				thisrisk *= (e == currentTarget && eenergy < myenergy*.7 ? -1 : eenergy < myenergy*1.3 ? 0 : 1) + Math.abs(Math.cos(toangle - angle(e, myLocation)));
			
			risk += thisrisk;
		}
		while (it.hasNext());
		
		// This is the magic of making minimum risk movement into random orbit.  It's slightly modified from a
		// line of code I found commented out in FloodHT, and it seems like a fine replacement for repelling the
		// last point I was at.
		
		// Here's the basic idea:
		// * Find the angle between the angle to this point and the angle I'm currently moving towards.
		// * If that angle is less than 90 degrees, I'm going straight, or at least relatively so.
		//   Otherwise (signified by the cosine being < 0), I would be switching directions to go to this point,
		//   and I go into the if statement.
		// * If I would be switching directions to go to this point, multiply the total risk by a proportion that
		//   is higher if the degree of turn is closer to 180 (actually, turning 90 degrees has its strengths sometimes)
		// * * Note the multiplication by the sign of the risk - because the risk is sometimes an attraction (and thus
		//     is negative), I need to make sure the amount I'm adding here is positive.
		// * * Note the multiplication of (flip ? -8 : 8).  Flip is usually false, but is true when I've randomly decided
		//     that this turn would be a good one to flip directions.  This gives me a more high-level way of sending the
		//     a "pulse" that makes it try and switch orbiting directions.
		// * Note that this has nothing to do with making the direction of movement an orbit, and that it only *suggests*
		//   when to turn around - turning around may not happen when asked and may happen when not asked, if the rest of
		//   the risk function, other bots, or the shape of the battlefield gives me no real choice.  The orbiting happens
		//   because of the perpendicularity function above, which also helps decide how to best be quasi-orbiting any and
		//   all bots in question.
		// * The parts I'm getting for free are distancing and avoidance, as well as making it easier to find my way out
		//   of tight situations.
		
		double curangle = getHeadingRadians();
		if (getDistanceRemaining()+getVelocity() > 0)
			curangle += Math.PI;
		curangle = Math.cos(toangle-curangle);
		// If this direction is different from my current direction by more than 90 degrees...
		if (curangle < -.05)
			risk *= 1 + (flip ? -1 : 1)*Math.signum(risk)*(1-curangle);
		
		return risk;
	}
	
	// Receive messages from my teammate: the only message I should be receiving is updated EnemyInfo objects.
	public void onMessageReceived(MessageEvent event)
	{
		EnemyInfo enemy = (EnemyInfo)event.getMessage();
		EnemyInfo oldEnemy;
		if ((oldEnemy = (EnemyInfo)enemies.get(enemy.name)) != null && oldEnemy.updateTime < enemy.updateTime)
		{
			// This stuff shouldn't come from my teammate:
			enemy.waves = oldEnemy.waves;
			enemy.bulletVelocity = oldEnemy.bulletVelocity;
			enemy.lastHit = oldEnemy.lastHit;
			// Grr, why can't x and y in Point2D.Double be serializable or have non-transient members?
			enemy.setLocation(enemy.savex, enemy.savey);
			if (currentTarget == oldEnemy)
				currentTarget = enemy;
			enemies.put(enemy.name, enemy);
			handleScannedRobot(enemy, false);
		}
	}
	
	// When I get a scan, update the corresponding enemy, call the method to handle the updates, and send the
	// updated enemy to my teammate.
	public void onScannedRobot(ScannedRobotEvent e)
	{
		String name;
		EnemyInfo enemy = (EnemyInfo)enemies.get(name = e.getName());
		if (enemy == null)
			enemies.put(name, enemy = new EnemyInfo());
		enemy.name = name;
		enemy.lastEnergy = enemy.energy;
		enemy.energy = e.getEnergy();
		enemy.lastVelocity = enemy.velocity;
		enemy.velocity = e.getVelocity();
		enemy.heading = e.getHeadingRadians();
		enemy.updateTime = e.getTime();
		enemy.setLocation(projectPoint(myLocation = new Point2D.Double(getX(), getY()), getHeadingRadians()+e.getBearingRadians(), e.getDistance()));
		
		if (!(enemy.isTeammate = isTeammate(enemy.name)))
		{
			handleScannedRobot(enemy, true);
			try
			{
				// I'll lose this data in transmission if I don't cache it :-\
				enemy.savex = enemy.getX();
				enemy.savey = enemy.getY();
				broadcastMessage(enemy);
			}
			catch (IOException ex)
			{
				ex.printStackTrace();
			}
		}
	}
	
	// This is most of the stuff that used to be in onScannedRobot, but now can be updated with an EnemyInfo that
	// came from my teammate via team communication.  Note that I can assume that enemy is really an enemy and not
	// my teammate.
	public void handleScannedRobot(EnemyInfo enemy, boolean mine)
	{
		int[][][][][][] currentStats = (int[][][][][][])stats.get(enemy.name);
		if (currentStats == null)
		{
			// Segmentations: melee/one-on-one, lateral velocity, lateral direction (melee)/acceleration (one-on-one),
			// bft, guessfactors
			// Acceleration segmentation is disabled in melee because it doesn't work reliably with tons of missed
			// scans (sometimes 15 or so at a time while I lock the other way)
			stats.put(enemy.name, currentStats = new int[2][3][3][3][13][31]);
		}
		double distance = enemy.distance(myLocation), absBearing = angle(enemy, myLocation), energy = enemy.energy;
		double ediff;
		if (Math.abs((ediff = enemy.energy-enemy.lastEnergy)-1.55) <= 1.46)
		{
			enemy.bulletVelocity = 20.0-3.0*ediff;
		}
		if (currentTarget == null || distance+energy < myLocation.distance(currentTarget)+currentTarget.energy)
			currentTarget = enemy;
		
		// Variables that are initialized in the minibot-esque code that follows:
		double velocity, latd, latv, bulletv, power;
		
		//problem with this code is that it will default to 1 if the velocity is zero, instead of defaulting the last observed value:
		double direction = ((latv = Math.sin(latd = enemy.heading-absBearing)*(velocity = enemy.velocity)/3) < 0 ? -1 : 1)*Math.asin(8/(bulletv = 20-3*(power = Math.min(Math.min(distance < 200 ? 3 : 2, 1000/distance), Math.min(getEnergy()/6, energy/4)))));
		
		//w00t - classic minibot code!
		int[] current = currentStats[Math.min(getOthers()-1, 1)][(int)Math.abs(latv)][/*getOthers() == 1 ? */Math.signum(velocity) == -Math.signum(enemy.lastVelocity) ? 2 : ((int)Math.signum(Math.rint(Math.abs(velocity)-Math.abs(enemy.lastVelocity)))+1)/* : */][(int)(Math.cos(latd)*Math.signum(velocity)*1.4+1.4)][(int)(distance/bulletv/10)];
		
		ArrayList waves = enemy.waves;
		int i=waves.size();
		while (i > 0)
		{
			i--;
			if (((MeleeBullet)waves.get(i)).updateEnemy(enemy, enemy.updateTime))
				waves.remove(i);
		}
		MeleeBullet wave;
		waves.add(wave = new MeleeBullet());
		wave.startPoint = myLocation;
		wave.startgunheading = absBearing;
		wave.direction = direction;
		wave.lastPoint = new Point2D.Double(enemy.getX(), enemy.getY());
		wave.bulletspeed = bulletv;
		wave.lasttime = enemy.updateTime;
		wave.segment = current;
		
		if (enemy == currentTarget)
		{
			int bestindex = 15;
			//i is zero from before, let's just use it :-)
			double shotBearing = absBearing;
			if (energy > 0)
				do
				{
					//the extension to this if statement is my clever idea to avoid shooting at the wall.  Hope it helps.
					double tempBearing;
					if (inField(projectPoint(myLocation, tempBearing = absBearing + direction*(i/15.0-1), distance*(bulletv/(bulletv+8)))) && current[i] > current[bestindex])
					{
						bestindex = i;
						shotBearing = tempBearing;
					}
					i++;
				}
				while (i < 31);
			
			if ((getOthers() == 1 || getGunHeat() < .8) && mine)
				setTurnRadarLeft(getRadarTurnRemaining());
			setTurnGunRightRadians(robocode.util.Utils.normalRelativeAngle(shotBearing-getGunHeadingRadians()));
			//a good enough constant for both:
			if (Math.max(Math.abs(getGunTurnRemaining()), energy/getEnergy()) < 5)
				setFire(power);
		}
	}
	
	public void onHitByBullet(HitByBulletEvent e)
	{
		EnemyInfo enemy;
		if ((enemy = (EnemyInfo)enemies.get(e.getName())) != null)
		{
			enemy.lastHit = getTime();
			enemy.bulletVelocity = e.getVelocity();
		}
	}
	
	// If a bot dies, remove it - if I was targeting it, forget my current target.
	public void onRobotDeath(RobotDeathEvent e)
	{
		//the EnemyInfo objects are volitile and made for a round at a time, so I can sacrifice them when they die:
		if (enemies.remove(e.getName()) == currentTarget)
			currentTarget = null;
	}
	
	// Fun synchronized victory dance, from an experimental team of mine, hope the codesize limit doesn't
	// force me to remove it:
	public void onWin(WinEvent e)
	{
		setTurnGunRightRadians(Math.atan2(getBattleFieldWidth()/2-getX(), getBattleFieldHeight()/2-getY())-getGunHeadingRadians());
		do
		{
			setAhead(100*Math.sin(getTime()/5.0));
			setTurnRightRadians(robocode.util.Utils.normalRelativeAngle(getTime()*.1-getHeadingRadians()));
			setFire(0);
			execute();
		}
		while (true);
	}
	
	// utility functions, I figure anyone making a goto bot has something like these, and they
	// come in useful in a lot of other random places, too:
	public static Point2D projectPoint(Point2D startPoint, double theta, double dist)
	{
		return new Point2D.Double(startPoint.getX() + dist * Math.sin(theta), startPoint.getY() + dist * Math.cos(theta));
	}
	
	// This strange habit of putting point2 before point1 in this method - and the funny thing is I think
	// I've seen other people do the same thing:
	public static double angle(Point2D point2, Point2D point1)
	{
		return Math.atan2(point2.getX()-point1.getX(), point2.getY()-point1.getY());
	}
	
	// returns true if a point is within a truncated battlefield
	public boolean inField(Point2D p)
	{
		return new Rectangle2D.Double(30, 30, getBattleFieldWidth()-60, getBattleFieldHeight()-60).contains(p);
	}
}
