package pl.Patton.Guns;

import robocode.*;

import java.awt.geom.Point2D;

import pl.Abstract.Gun;
import pl.Enemy.AdvancedEnemy;
import pl.Utilities.*;

/*******************************************************************************
 * A gun that utilizes Segmented Guess-Factor targeting.
 * 
 * Technically, according to the 'most proper definition' of Guess-Factor
 * Targeting, this is not a true GFT gun, but a Segmented Statistical Gun
 * utilizing Visit Count Statistics and Raw Bearing Offsets.
 * 
 * Since: dev
 * 
 * Last changed: 1.54
 * 
 * Created based on:
 * 
 * GFTGun (which was based on pez.tiny.Falcon found at
 * http://robowiki.net/cgi-bin/robowiki?Falcon/WithoutShrinkingTricks and
 * http://www.robocoderepository.com/BotDetail.jsp?id=2009) - removed from
 * GeneralPatton as of 1.52
 * 
 * and pez.micro.Aristocles found at
 * http://www.robocoderepository.com/BotDetail.jsp?id=1923
 ******************************************************************************/
public class SGFTGun extends Gun {
	public static Point2D.Double enemyLoc;

	public static final double MAX_DISTANCE = Point2D
			.distance(0, 0, robot.getBattleFieldWidth(), robot.getBattleFieldHeight());
	// Each distance index is 100 large.
	public static final int DISTANCE_INDEXES = (int) (MAX_DISTANCE / 100);

	// Indexes are 0&+-1, +-2&+-3, +-4&+-5, +-6&+-7, +-8
	public final static int VELOCITY_INDEXES = 5;
	public final static int PREVIOUS_VELOCITY_INDEXES = VELOCITY_INDEXES;
	public final static int MIDDLE_VELOCITY_INDEX = (VELOCITY_INDEXES - 1) / 2;

	// I use 47 bins - one bin every two angles
	public final static int BINS = 47;
	public final static int MIDDLE_BIN = (BINS - 1) / 2;
	// This angle is derived from Math.asin(8D / (20D - (3D * power))),
	// with power being 3.0, the max bullet power. In degrees, it is
	// 46.658241772777615D, the source of the number of bins I use.
	public final static double MAX_ESC_ANGLE = 0.8143399421265254D;
	public final static double BIN_WIDTH = MAX_ESC_ANGLE / MIDDLE_BIN;

	// Segmented GF - 4 dimensional array - this gives me 11,750 bins to work
	// with :P
	public static double[][][][] stats;

	// Keep track of the enemy
	public static AdvancedEnemy enemy;

	public SGFTGun(String name, AdvancedRobot robot) {
		super(name, robot);
		stats = new double[DISTANCE_INDEXES][VELOCITY_INDEXES][PREVIOUS_VELOCITY_INDEXES][BINS];
	}

	public void run(AdvancedEnemy enemy0) {
		// If I did not already have an enemy
		if (enemy == null) {
			// I do now.
			enemy = enemy0;
		}
		// If I have a new enemy
		else if (enemy != enemy0) {
			// Switch to the new one
			enemy = enemy0;
		}
		double absBearing = robot.getHeadingRadians() + enemy.bearing;
		double firePower = enemy.distance < 200 ? 3.0 : 1.72;
		// Make a new wave
		SGFTWave w = new SGFTWave();
		w.firedLocation = new Point2D.Double(robot.getX(), robot.getY());
		enemyLoc = new Point2D.Double(enemy.x, enemy.y);
		// Figure out which direction they're moving in, CW or CCW -
		// Math.sin determines if they're facing CW or CCW, but their
		// velocity might be negative
		w.sign = MiscUtils.sign(enemy.velocity * Math.sin(enemy.heading - absBearing));
		w.bearing = absBearing;
		w.velocity = BulletUtils.bulletVelocity(firePower);
		// Find the segment they're at
		// Divide their distance by the size of each distance segment, this
		// gives me the correct index
		int distanceIndex = (int) (enemy.distance / (MAX_DISTANCE / DISTANCE_INDEXES));
		// Take their velocity, divide it by two, and put it back in the array
		int velocityIndex = (int) Math.abs(enemy.velocity / 2);
		// Same thing, but with their previous velocity
		int previousVelocityIndex = (int) Math.abs(enemy.pVelocity / 2);
		// Set the wave's segment as this
		w.segment = stats[distanceIndex][velocityIndex][previousVelocityIndex];
		robot.setTurnGunRightRadians(AngleUtils.normalizeBearing(
		// Face the gun towards them
				absBearing - robot.getGunHeadingRadians() +
				// and then offset it to the best relative angle
						(w.sign * BIN_WIDTH) * (getMostVisited(w.segment) - MIDDLE_BIN)));
		// Only set the bullet as firing if we actually fired.
		if (robot.getGunHeat() == 0 && robot.getGunTurnRemaining() < 5)
			if (robot.setFireBullet(firePower) != null)
				w.firing = true;
		// Only add the wave if we're still active
		if (robot.getEnergy() >= 0.1)
			robot.addCustomEvent(w);
	}

	/**
	 * Finds the most visited bin
	 * 
	 * @param segment the segment to search through
	 * @return most visited bin
	 */
	public int getMostVisited(double[] segment) {
		// If stats are empty, use head-on
		int mostVisited = MIDDLE_BIN;
		for (int i = 0; i < BINS; i++)
			if (segment[i] > segment[mostVisited])
				mostVisited = i;
		return mostVisited;
	}

	/**
	 * A class for tracking bullet waves.
	 * 
	 * It extends Condition because Robocode has these great things called
	 * Custom Events. Robobocde calls the test() method of it once a tick, every
	 * tick, meaning that it will never miss a turn.
	 */
	public class SGFTWave extends Condition {
		public double[] segment;
		public boolean firing;
		public Point2D.Double firedLocation;
		public double velocity, distanceTraveled, bearing, sign;

		public boolean test() {
			// The bullet travels
			distanceTraveled += velocity;
			double distance = firedLocation.distance(enemyLoc);
			// If the bullet has reached the opponent
			if (distanceTraveled > distance - 18) {
				int bin = (int) Math.round((AngleUtils.normalizeBearing(
				// Find the angle to the opponent and subtract the original
						// angle from it
						AngleUtils.absoluteBearing(firedLocation, enemyLoc)) - bearing)
				// divide it by the size of each bin, but not always positive
						/ (sign * BIN_WIDTH))
				// and of course, we always want to put it back in the array
						+ MIDDLE_BIN;
				// Weight firing and non-firing differently
				if (firing)
					for (int i = 0; i < BINS; i++) {
						// Bin smoothing using inverse power
						segment[i] += 1D / (Math.pow(bin - i, 2) + 1);
					}
				else
					for (int i = 0; i < BINS; i++) {
						// Bin smoothing using inverse power
						segment[i] += 1D / (Math.pow(bin - i, 2) + 2);
					}
				// Since it's done, no point updating anymore
				robot.removeCustomEvent(this);
			}
			return false;
		}
	}

}