package tjk.universe;

import ags.utils.dataStructures.trees.thirdGenKD.*;
import java.util.ArrayList;

import tjk.universe.BracketHist.HistType;
import tjk.utils.FastTrig;
import java.awt.geom.Point2D;

/**
 * WaveStats - a class by tkiesel
 * Copyright (c) 2012 Tom Kiesel (Tkiesel @ Robowiki)
 * 
 * This software is provided 'as-is', without any express or implied warranty. In no event will the authors be held liable for any damages arising from the use of this software.
 * 
 * Permission is granted to anyone to use this software for any purpose, including commercial applications, and to alter it and redistribute it freely, subject to the following restrictions:
 * 
 *     1. The origin of this software must not be misrepresented; you must not claim that you wrote the original software. 
 *        If you use this software in a product, an acknowledgment in the product documentation would be appreciated but is not required.
 * 
 *     2. Altered source versions must be plainly marked as such, and must not be misrepresented as being the original software.
 * 
 *     3. This notice may not be removed or altered from any source distribution.
 * 
 */

public class WaveStats
{
	
	private KdTree<GFBracket> stats;
	
	private WeightedDistanceFunction distanceF;
	
	public WaveStats()
	{
		stats = new KdTree<GFBracket>(Segmentation.DIMENSIONS);
		distanceF = new WeightedManhattanDistanceFunction(Segmentation.WEIGHTS);
	}
	
	public void addWave(Wave w)
	{
		stats.addPoint( w.getSituation(), new GFBracket(w) );
	}

	public void addStartNullHits()
	{
		double[] bracket = {-0.05,0.0505};
		double[] zeroVector = {1.0, 0.0};
		GFBracket GF = new GFBracket(true, bracket, zeroVector);
		double[] situation = new double[Segmentation.DIMENSIONS];
		for ( int i = 0; i < Segmentation.DIMENSIONS; i++  )
		{
			situation[i] = 0.5;
		}
		stats.addPoint( situation, GF );
		//double[] bracket2 = {0.955,1.0};
		//GF = new GFBracket(false, bracket2, zeroVector);
		//stats.addPoint( situation, GF );
	}

	public void addBulletHit(Wave w, Point2D.Double hitLocation)
	{
		double hitDistance = hitLocation.distance(w.getLocation());
		double halfAngleWidth = FastTrig.atan(24.0/hitDistance);
		double middleAngle = FastTrig.atan2(hitLocation.getX()-w.getX(),hitLocation.getY()-w.getY());
		
		double[] bracket = {w.absToGF(middleAngle-halfAngleWidth),w.absToGF(middleAngle+halfAngleWidth)};
		double[] zeroVector = {1.0, 0.0};
		GFBracket GF = new GFBracket(true, bracket, zeroVector);
		stats.addPoint( w.getSituation(), GF );		
	}
	
	// Note, this fetches double the points to iterate over.
	public NearestNeighborIterator<GFBracket> getNearestNeighbors(Wave w, int closestK)
	{
		NearestNeighborIterator<GFBracket> neighbors = stats.getNearestNeighborIterator(w.getSituation(),(int)(1.5*closestK),distanceF);
		if ( !neighbors.hasNext() )
		{
			System.out.println("ERROR: The kdTree returned an empty nearestNeighbor!");
			return null;
		}
		return neighbors;
	}
	
	public ArrayList<ArrayList<GFBracket>> getNearestGFBrackets(Wave w, int closestK, boolean drop)
	{
		NearestNeighborIterator<GFBracket> neighbors = getNearestNeighbors(w,closestK);
		if ( neighbors != null )
		{
			return getNearestGFBrackets(neighbors, w, closestK, drop);
		}
		return null;
	}

	// Return an array[] of these... [nocull,NULL] or [nocull,cull].
	public ArrayList<ArrayList<GFBracket>> getNearestGFBrackets(NearestNeighborIterator<GFBracket> neighbors, Wave w, int closestK, boolean drop)
	{
		
		ArrayList<GFBracket> answer0 = new ArrayList<GFBracket>(closestK);
		ArrayList<GFBracket> answer1 = null;
		if ( drop )
		{
			answer1 = new ArrayList<GFBracket>(closestK);
		}
	
		// Drop entries that fall outside of the battlefield.
		double x0 = w.getX();
		double y0 = w.getY();
		double D = w.START_D;
		double headOn = w.getMCM()[1];
		double cW = w.cW();
		
		while ( neighbors.hasNext()  && ( answer0.size() < closestK || ( drop && answer1.size() < closestK) ) )
		{
			GFBracket b = neighbors.next();
			
			// no cull bracket
			if ( answer0.size() < closestK )
			{
				answer0.add(b);
			}
			
			if ( drop && answer1.size() < closestK )
			{
				double testX = x0 + D*b.vector[0]*FastTrig.sin( headOn + cW*b.vector[1] );
				if ( Double.compare(testX,18.0) < 0 || Double.compare(testX,Universe.bfW()-18.0) > 0 )
				{
					//DEBUG_WAVE_CULL_COUNT++;
				}
				else
				{
					double testY = y0 + D*b.vector[0]*FastTrig.cos( headOn + cW*b.vector[1] );
					if ( Double.compare(testY,18.0) < 0 || Double.compare(testY,Universe.bfH()-18.0) > 0 )
					{
						//DEBUG_WAVE_CULL_COUNT++;
					}
					else
					{
						answer1.add(b);
					}
				}
			}
			
		}
	
		if ( drop && answer1.isEmpty() )
		{
			answer1 = answer0;
		}
		//System.out.println("Useful Bracket! " + answer.size() + " waves, " + DEBUG_WAVE_CULL_COUNT + " culled.");
		
		ArrayList<ArrayList<GFBracket>> answer = new ArrayList<ArrayList<GFBracket>>(2);
		
		answer.add(answer0);
		answer.add(answer1);
		
		return answer;
	}	
	
	
	public BracketHist getBracketHist(HistType type, Wave w, int closestK, double realWeight, boolean drop)
	{
		int listNum = drop ? 1 : 0;
		ArrayList<GFBracket> GF = getNearestGFBrackets(w, closestK, drop).get(listNum);
		if ( GF == null )
		{
			return null;
		}
		return new BracketHist(type, GF,realWeight);
	}
	
	/*
	public BracketHist getBracketHist(HistType type, Wave w, int closestK, double realWeight, boolean drop, double[] halfLife)
	{
		ArrayList<GFBracket> GF = getNearestGFBrackets(w, closestK, drop);
		if ( GF == null )
		{
			return null;
		}
		return new BracketHist(type, GF,realWeight,1.0,true,halfLife);
	}
	*/
	public int getSize()
	{
		return stats.size();
	}
	
}