/*
 * Copyright (c) 2004 Marcin Fuszara
 */  
package fushi.PvP1;
import robocode.*;

/* This class is responsible for robots
 * body movements
 */
public class Body
{
   /* maximum acceptable distance at which the robot is
    * regarded as "at the destination"
    */
   static final double reachEpsilon = 3;
   
   /* maximum distance at which the robot is regarded
    * as "close to the destination";
    * if heading for the destination this is the distance
    * at which the robot would turn the breaks on
    * in the next frame
    */
   static final double closeEpsilon = 32;
   
   /* robot widht/height
    */
   static final double robotSize = 24;
   
   /* this value is used to allow robot move a little towards
    * a wall if blocked by it
    */
   static final double wallCloseEpsilon = 1;
   
   /* maximum time that robot is allowed to be blocked by a wall;
    * after this time the robot will move even if it risks
    * hitting a wall;
    */
   static final int maxBlockedTime = 9;  // enough time to make a sharp turn
  
   // the robot for which this body logic is applicable
   AdvancedRobot robot;
   // arena dimmensions
   double arenaWidth, arenaHeight;
   // current robot destination
   Vector2d destination;
   /* destination clipped to arena borders;
    * if clippedDestination != null then the robot should go first
    * to clippedDestination and upon reach head towards the
    * destination;
    * otherwise it should go directly to destination
    */
   Vector2d clippedDestination;
   /* number of frames for which the robot is blocked;
    * helps avoid deadlocks in corners;
    */
   int blockedTime = 0;
   /* maximum velocity, above which the robot doesn't speed up
    */
   double maximumVelocity = 8;
   
   /* robot simulator used to predict robot positions
    */
   BodyPeer bodySimulator;
   
   public Body( AdvancedRobot robot )
   {
      this.robot = robot;
      // get arena dimmensions
      arenaWidth = robot.getBattleFieldWidth();
      arenaHeight = robot.getBattleFieldHeight();
      // set some destination
      destination = new Vector2d( robot.getX(), robot.getY() );
      
      // create a body simulator
      bodySimulator = new BodyPeer();
   }
   
   /* order the robot to head for a specified destination
    */
   public void goTo( Vector2d dest )
   {
      destination.set( dest );
      Logger.log( "goTo", "Going to: " + dest.toString() );
      clipDestination();
   }
   
   /* set the maximum velocity with which the robot can go
    * to given value
    */
   public void setMaxVelocity( double velocity )
   {
      if ( velocity < 0 )
         maximumVelocity = 0;
      else if ( velocity > 8 )
         maximumVelocity = 8;
      else
         maximumVelocity = velocity;
   }
   
   /* sets the robot turn and speed so
    * that it gets to given destination as fast as possible;
    * this method takes care that the robot doesn't hit the
    * wall and if the required destination is outside the
    * arena boundaries it will make the robot 'slide'
    * along the wall to get as close to the destination
    * as possible;
    * \return a vector describing robot's position in the next frame
    *         if a robot is not blocked by another robot or a wall
    */
   public void run()
   {
      Vector2d robotPos = new Vector2d( robot.getX(), robot.getY() );
      // current destination is one of: destination, clippedDestination
      Vector2d currentDestination = null;
      // don't let the robot to hit the wall by trying to go outside the arena
      if ( clippedDestination == null )
         currentDestination = destination;
      else {
         /* first the robot has to get to clipped destination and upon reach
          * head towards the actual destination along the wall
          */
         // check the clipped destination distance
         double clippedDestDist = Vector2d.distance( robotPos, clippedDestination );
         if ( clippedDestDist <= reachEpsilon ) { // clipped reached; go to destination
            goTo( destination ); // set destination and clip again
            if ( clippedDestination == null )
               currentDestination = destination;
            else
               currentDestination = clippedDestination;
         } else    // have to get to clipped destination first
            currentDestination = clippedDestination;
      }
      
      // get destination relative to robot position
      Vector2d destRelative = new Vector2d( currentDestination );
      destRelative.sub( robotPos );
      
      double destRelativeBearing = destRelative.angle(); // get absolute destination bearing
      destRelativeBearing -= robot.getHeadingRadians(); // compute bearing relative to robot
      destRelativeBearing = Util.normalAbsoluteAngle( destRelativeBearing );
      
      double turnAngle = destRelativeBearing;  // the angle by which i will have to turn
      double direction=1; // 1=forward; -1=backwards
      
      if ( turnAngle > Math.PI/2 && turnAngle < Math.PI + Math.PI/2 ) {  // destination is behind my robot
         turnAngle -= Math.PI;
         direction = -1;
      } // else => destination if somewhere in front of my robot
      
      // turn robot towards destination
      turnAngle = Util.normalRelativeAngle( turnAngle );
      robot.setTurnRightRadians( turnAngle );
      
      // move robot
      double ahead = direction * destinationDistance();
            
      /* determine maximum speed base on bearing to the destination
       * so that the turn has a small radius
       */
      double relativeCos = Math.cos( turnAngle );
      double velocityFactor = Math.abs( relativeCos );
      // use maximum velocity set by the user
      // use velocityFactor^2 to brake harder at tight corners
      double maxVelocity = velocityFactor * velocityFactor * maximumVelocity;
      
      /* if the robot sits near a wall and is starting to move,
       * assure that is doesn't hit a wall;
       */
      if ( blockedTime < maxBlockedTime // don't allow to stay in place for too long
           && willHitWall( direction, ahead ) ) {
         ahead = 0;
         blockedTime++;
      } else {
         blockedTime = 0; // remember that i'm not blocked
      }
      
      robot.setAhead( ahead );
      robot.setMaxVelocity( maxVelocity );
      
      /* run the robot simulator to prepare position and heading predicted
       * for the next frame
       */
      simulate( turnAngle, ahead, maxVelocity );
       
   }
   
   /* the following methods:
    * - getNextPosition
    * - getNextHeading
    * - getNextVelocity
    * may be called after calling the run method;
    * they return appropriate values predicted for the robot
    * in the next frame; these values are valid only
    * if the robot doesn't hit a wall or another robot;
    */
    
   public Vector2d getNextPosition()
   {
      return bodySimulator.getPosition();
   }
   
   public double getNextHeading()
   {
      return bodySimulator.getHeading();
   }
   
   public double getNextVelocity()
   {
      return bodySimulator.getVelocity();
   }
   
   /* compute and return the distance to current destination;
    * if clipped == true then the distance to clipped destination
    * instead of real destination is computed;
    */
   public double destinationDistance()
   {
      Vector2d robotPos = new Vector2d( robot.getX(), robot.getY() );
      robotPos.sub( destination );
      return robotPos.length();
   }
   
   /* returns true if the destination ordered by goTo()
    * is reached; false otherwise;
    */
   public boolean destinationReached()
   {
      // take clipped destination into consideration
      return ( destinationDistance() <= reachEpsilon );
   }
   
   /* returns true if the destination ordered by goTo()
    * is so close that the robot would turn breaks on
    * in the next frame;
    */
   public boolean destinationClose()
   {
      // take clipped destination into consideration
      return ( destinationDistance() <= closeEpsilon );
   };
   
   /* returns true if the robot is going to hit a wall in the
    * next frame going as it is;
    */
   boolean willHitWall( double direction, double ahead )
   {
      // current robot position
      Vector2d robotPos = new Vector2d( robot.getX(), robot.getY() );
            
      // assume maximum velocity
      double distance = 8 * direction;
      if ( Math.abs(distance) > Math.abs(ahead) )
         distance = ahead;
      
      // relative movement
      Vector2d moveRelative = new Vector2d();
      moveRelative.setFromAngleDistance( robot.getHeadingRadians(), distance );
      
      // compute robot pos after movement
      robotPos.add( moveRelative );
      // robot will hit wall only if inside boundaries and not moving towards arena center
      if ( ( robotPos.x < robotSize && moveRelative.x < -wallCloseEpsilon )
         || ( robotPos.x > arenaWidth - robotSize && moveRelative.x > wallCloseEpsilon )
         || ( robotPos.y < robotSize && moveRelative.y < -wallCloseEpsilon )
         || ( robotPos.y > arenaHeight - robotSize && moveRelative.y > wallCloseEpsilon ) ) {
         Logger.log( "block", "Blocked to " + robotPos.toString() );
         Logger.log( "block", " with relative " + moveRelative.toString() );
         return true;
      }
      return false;
   }
   
   /* takes current destination and robot position and
    * sets clippedDestination so that a robot going
    * there will keep the heading towards destination
    * as long as possible;
    * the destination is set so that after reaching
    * clipped destination, the robot will go 
    * along the wall to get to the
    * destination point as close as possible
    */
   void clipDestination()
   {
      Vector2d robotPos = new Vector2d( robot.getX(), robot.getY() );
      
      // prepare minimum and maximum coordinate values
      double minX = robotSize;
      double maxX = arenaWidth - robotSize;
      double minY = robotSize;
      double maxY = arenaHeight - robotSize;
      
      // assume that the robot is in a valid position; clip its coordinates
      if ( robotPos.x < minX )
         robotPos.x = minX;
      else if ( robotPos.x > maxX )
         robotPos.x = maxX;
      if ( robotPos.y < minY )
         robotPos.y = minY;
      else if ( robotPos.y > maxY )
         robotPos.y = maxY;
      
      // find closest wall hit
      Vector2d verticalClipPoint = new Vector2d( destination );
      Vector2d horizontalClipPoint = new Vector2d( destination );
      boolean clippedVertical = false;
      boolean clippedHorizontal = false;
      
      // clip to vertical boundaries
      if ( robotPos.x != destination.x ) { // clip vertically only if not moving vertically
         if ( destination.x < minX ) {
            clippedVertical = true;
            verticalClipPoint.x = minX;
            verticalClipPoint.y += (minX - destination.x) 
                                 * (robotPos.y - destination.y)
                                 / (robotPos.x - destination.x);
         } else if ( destination.x > maxX ) {
            clippedVertical = true;
            verticalClipPoint.x = maxX;
            verticalClipPoint.y += (destination.x - maxX)
                                 * (robotPos.y - destination.y)
                                 / (destination.x - robotPos.x);
         }
      }
      
      Vector2d clipPoint = new Vector2d();      
      // clip to horizontal boundaries
      if ( robotPos.y != destination.y ) { // clip horizontally only if not moving horizontally
         if ( destination.y < minY ) {
            clippedHorizontal = true;
            horizontalClipPoint.y = minY;
            horizontalClipPoint.x += (minY - destination.y)
                                   * (robotPos.x - destination.x)
                                   / (robotPos.y - destination.y);
         } else if ( destination.y > maxY ) {
            clippedHorizontal = true;
            horizontalClipPoint.y = maxY;
            horizontalClipPoint.x += (destination.y - maxY)
                                   * (robotPos.x - destination.x)
                                   / (destination.y - robotPos.y);
         }
      }
      
      if ( clippedHorizontal && clippedVertical ) {
         // find closest clip point
         if ( Vector2d.distance( robotPos, horizontalClipPoint ) 
              < Vector2d.distance( robotPos, verticalClipPoint ) ) {  // horizontal closer
            clippedDestination = horizontalClipPoint;
            destination.y = clippedDestination.y;
         } else {
            clippedDestination = verticalClipPoint;
            destination.x = clippedDestination.x;
         }
      } else if ( clippedHorizontal ) {
         clippedDestination = horizontalClipPoint;
         destination.y = clippedDestination.y;
      } else if ( clippedVertical ) {
         clippedDestination = verticalClipPoint;
         destination.x = clippedDestination.x;
      } else
         clippedDestination = null;
             
      if ( clippedDestination != null ) {
         Logger.log( "clipping", "destination clipped to: " + clippedDestination.toString() );
         Logger.log( "clipping", "destination: " + destination.toString() );
      }
         
   }

   /* given set values for ahead, turn and maximum velocity,
    * simulates the robot for one frame;
    * used to predict robot position and heading in the next frame;
    * \param turnAngle [radians]
    */
   void simulate( double turnAngle, double ahead, double maxVelocity )
   {
      // set important values for the simulator
      bodySimulator.setX( robot.getX() );
      bodySimulator.setY( robot.getY() );
      bodySimulator.setHeading( robot.getHeadingRadians() );
      bodySimulator.setVelocity( robot.getVelocity() );
      bodySimulator.setTurn( turnAngle );
      bodySimulator.setMove( ahead );
      bodySimulator.setMaxVelocity( maxVelocity ); 
      // compute the result
      bodySimulator.simulate();
   }
   
}