/*
 * Copyright (c) 2004 Marcin Fuszara
 */  
package fushi.PvP1;
import robocode.*;

/* This class is responsible for firing bullets
 * and turret movement
 */
public class Gun
{
   /* maximum distance by which a shot bullet can miss
    * the target
    */
   static final double targetEpsilon = 8;

   // the robot for which this body logic is applicable
   AdvancedRobot robot;
   // point on the arena at which the gun should keep facing
   Vector2d targetPosition;
   // if true the target at given position should be shot at
   boolean shootTarget;
   // time at which the bullet has to reach the target (-1 for "as soon as possible")
   long impactTime;
   // power of requested shot
   double firePower; 
   
   public Gun( AdvancedRobot robot )
   {
      this.robot = robot;
      // set some target
      targetPosition = new Vector2d( robot.getBattleFieldWidth()/2, 
                                     robot.getBattleFieldHeight()/2 );
      shootTarget = false;
   }
   
   /* order the gun to keep heading at specific coordinates;
    * the gun will keep turning to face that coordinates as quick
    * as possible;
    */
   public void aimAt( Vector2d target )
   {
      targetPosition.set( target );
   }
   
   /* order the gun to shoot at specified coordinates;
    * the gun fires as soon as possible;
    */
   public void fireAt( Vector2d target, double power )
   {
      targetPosition.set( target );
      shootTarget = true;
      impactTime = -1; // time doesn't matter; fire as soon as possible
      firePower = power;
   }  
   
   /* order the gun to shoot at specified coordinates, so that
    * a bullet fired will appear at given coordinates exactly
    * at specified moment in time;
    * if the gun does not manage to shoot the bullet fast enough
    * (time value is too small), the bullet is shot as soon as possible;
    * to make sure that the time value is not too small use
    * estimateTimeNeeded method to predict how much at least 
    * time will be needed
    */
   public void fireAt( Vector2d target, double power, long time )
   {
      targetPosition.set( target );
      shootTarget = true;
      impactTime = time;
      firePower = power;
   }
   
   /* compute the time in which a bullet with given
    * power can reach given destination;
    * !note that this is minimal time; the actual time can be a few frames longer
    * so keep in mind that the value returned is approximate
    */
   public long estimateTimeNeeded( Vector2d target, double power )
   {
      Vector2d robotPos = new Vector2d( robot.getX(), robot.getY() );
      // estimate bullet flight time
      long bulletTime = bulletTime( robotPos, target, power );
      // compute gun cooling time                              
      double gunCoolingRate = robot.getGunCoolingRate();
      double gunHeat = robot.getGunHeat();
      long gunCoolingTime = (long)Math.ceil( gunHeat/gunCoolingRate );
      
      // compute gun rotation time
      double targetAngle = targetTurnAngle( robotPos, target, robot.getHeadingRadians() );
      long turretTurnTime = (long)Math.ceil( targetAngle/10 );
      
      // preparation time is the time to turn the turret and cool down the gun
      long preparationTime = Math.max( gunCoolingTime, turretTurnTime );
      
      // result is the preparation time and bullet flight time
      return bulletTime + preparationTime;
   }
   
   /* this method should be called per frame;
    * it takes care to fulfill received orders;
    * \param nextRobotPosition provide robot position predicted for the next frame;
    *        it is crucial for precise aiming
    * \param nextRobotHeading provide robot heading in the next frame [radians]
    */
   public void run( Vector2d nextRobotPosition, double nextRobotHeading )
   {
      // if the target has to be shot do it only if the gun is not hot
      if ( shootTarget && robot.getGunHeat() == 0 ) {     
      
         // estimate bullet flight time
         long bulletTime = bulletTime( nextRobotPosition, targetPosition, firePower );
         
         boolean shootNow = true;  // if true, the gun should fire if ready; if false - wait
         if ( impactTime >= 0 ) {    // impact time specified; compute when to shoot the target
            if ( robot.getTime() + bulletTime < impactTime ) // it is to early to shoot
               shootNow = false;
         }
         
         // the gun is ready to fire, go!
         if ( shootNow ) {
            // get current gun bearing to target
            double targetAngle = targetTurnAngle( nextRobotPosition, targetPosition, robot.getHeadingRadians() );
            // get distance to target
            double targetDistance = Vector2d.distance( nextRobotPosition, targetPosition );
            // compute shot error if fired at this angle
            double error = Math.atan2( targetEpsilon, targetDistance );
            
            // shoot only if the error is acceptable or it is too late
            if ( Math.abs( targetAngle ) <= Math.abs( error ) 
                 || robot.getTime() + bulletTime - impactTime > 0 ) {
               robot.setFire( firePower );
               shootTarget = false;  // do not shot unless ordered again
            } 
         }
      }
      
      // turn the turret so that it faces the target in the next frame
      double turnAngle = targetTurnAngle( nextRobotPosition, targetPosition, nextRobotHeading );     
      // turn the gun
      robot.setTurnGunRightRadians( turnAngle );
   }
   
   /* returns true if the previous order has been fulfilled
    * and the gun is ready to receive a new order;
    */
   public boolean isReady()
   {
      return ( !shootTarget );
   }
   
   /* computes the angle by which the gun has to be turned to shoot at given
    * target if the robot is at given position and with given heading
    * with current gun heading
    */
   double targetTurnAngle( Vector2d robotPosition, Vector2d target, double robotHeading )
   {
      // relative position of target
      Vector2d targetRelative = new Vector2d( target );
      targetRelative.sub( robotPosition );
      // absolute angle to target
      double angle = targetRelative.angle();
      // relative angle to target
      double angleRelative = angle - robotHeading;
      // get actual gun heading relative to chassis
      double gunHeadingRelative = robot.getGunHeadingRadians() - robotHeading;
      // angle by which the gun has to be turned
      double turnAngle = Util.normalRelativeAngle( angleRelative - gunHeadingRelative );
      
      return turnAngle;
   }
   
   /* given bullet strength, its source and destination,
    * computes how much time the bullet will need to reach the destination;
    * \return number of frames after which the bullet will pass the destination
    */
   static long bulletTime( Vector2d source, Vector2d destination, double power )
   {
      double bulletSpeed = 20 - 3 * power;
      double distance = Vector2d.distance( source, destination );
      return (long)Math.ceil(distance / bulletSpeed);
   }
}