package nammyung;
import robocode.*;
import java.awt.Color;

/***
 * ModelT: a robot by Sangman Lee
 *
 *   This is an adaptation of SnippetBot.  The firing routine is a bit smarter about
 * not firing at enemies if the predicted position would be out of the battle field
 * bounds.
 *   The movement routine is tweaked in case of a 1-on-1 situation to be a bit more
 * aggressive.  Also, a rudimentary wall-avoiding routine has been implemented.
 *
 * Future plans: Add bullet-dodging.
 *
**/

public class ModelT extends AdvancedRobot
{
	sEnemy target;				// Adapted from SnippetBot's Enemy class
	final double PI = Math.PI;
	int direction = 1;			// Toggle for oscillating movement
	double firePower;			// Bullet power
	int fired;					// Number of shots fired
	int hit;					// Number of shots hit
	
/*	SnippetBot used a linear predictive firing routine based on estimated target position at estimated
	impact time.  We use a slightly different prediction routine, but still use these values for other
	fire-control decisions. */
	long time;					// 1st-approximation of time that bullet will take to hit target
	double x;					// 1st-approximation of target x-coordinate at time of impact
	double y;					// 1st-approximation of target y-coordinate at time of impact
	
/*	Constants */
	final int PERIOD = 23;				// How long before switching directions in movement?
	final double CUSHION = 150;			// How close can we get to the walls in 1-on-1 battle before bouncing?
	final double MIN_CUSHION = 30;		// How close can we get to the walls in multi-bot battle?
										// This needs to be small, or we'll get stuck in the middle.
	final int BOUNCE = 150;				// How far do we 'bounce' off wall boundaries?
	final double SHORT_RANGE = 125;		// We use SnippetBot's old firing routines below this distance	
	final int MAX_TIME_OFFSET = 25;		// Maximum time offset to time of impact.  If the bullet will take longer
										// than this to hit, target's too far / too fast.
	final double MAX_POS_OFFSET = 150;	// Maximum positional offset to predicted target position.  If the target
										// moves more than this until time of impact, that's too far.
	
	public void run() {
		fired = 0;
		hit = 0;
		target = new sEnemy();
		target.distance = 100000;
		setColors(Color.black,Color.red,Color.yellow);

		setAdjustGunForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		turnRadarRightRadians(2*PI);
		
		while(true) {
			doMovement();
			doFirePower();
			doScanner();
	
			/* If target's close, use the old doGun().  If it's far, use the new doGun2() */		
			if (target.distance < SHORT_RANGE) {
				doGun();
			} else {
			/*	doGun2(); */
				doGun();
			}
			
			/* How long, approximately, will it take for bullet to hit? This underestimates. */
			time = (long)(tFun(target,firePower));
			/* Where will the target be after that time? */
			x = target.guessX(time + getTime());
			y = target.guessY(time + getTime());
			
			/* Only fire if gun's cool, we have valid target, and range is close enough according to  firePower() */
			if (getGunHeat() == 0 && target.name != "None" && firePower > 0 && target.distance < 5000){
				/* Only fire if the predicted target position is within the battlefield. */
				if ((x > 10) && (y > 10) && (x < getBattleFieldWidth()-10) && (y < getBattleFieldHeight()-10)) {
					/* Only fire if we feel that the target won't be moving too far or too fast */
					if (time < MAX_TIME_OFFSET && getrange(target.x,target.y,x,y) < MAX_POS_OFFSET) {
						fire(firePower);
						/* Update number of shots fired */
						fired++;
					}
				}
			}
			execute();
		}
	}
	
	public void onBulletHit(BulletHitEvent e) {
		hit++;
	}
	
	public void onDeath(DeathEvent e) {
		out.println("Died. "+getOthers()+" other robots remaining.");
		review();
	}
	
	public void onWin(WinEvent e) {
		stop();
		setTurnRadarLeft(3600);
		turnRight(900);
		review();
	}
	
	void review() {
		out.println("Shots made: "+hit+"/"+fired+" ("+(float)(100*hit/fired)+"%)");
	}

	void doFirePower() {
		if (target.distance < 100) {
			firePower = 3;
		} else if (target.distance < 200) {
			firePower = 2;
		} else if (getEnergy() < 5 && target.energy > getEnergy()) {
			firePower = 0;
		} else if (target.distance < 300) {
			firePower = 1;
		} else if (target.speed == 0 && target.distance < 400) {
			firePower = 1;
		} else if (target.energy < 50) {
			firePower = .5;
		} else firePower = .2;
	}
	
	void doMovement() {
		if (checkWall() < 0) {
			avoidWall(checkWall()); 
		} else {
			if (getTime() % PERIOD == 0)  {
				direction *= -1;
				setAhead(direction*400);
			}
			if (getOthers() == 1) {
				if (target.energy < (getEnergy()/2)) {
					setTurnRightRadians(target.bearing + (PI/2) - (direction*PI/6));
				} else {
					setTurnRightRadians(target.bearing + (PI/2) + (direction*PI/6));
				}
			} else {
				setTurnRightRadians(target.bearing + (PI/2) - (direction*PI/6));
			}
		}	
	}
	
	int checkWall() {
		int parameter = 0;
		double cushion;
		if (getOthers() != 0) {
			cushion = CUSHION / (getOthers());
			if (cushion < 30) cushion = 30;
		} else {
			cushion = CUSHION;
		}

		double xmax = getBattleFieldWidth();
		double ymax = getBattleFieldHeight();
		
		if (getX() < cushion) {
			parameter = -1;
		} else if (getX() > (xmax - cushion)) {
			parameter = -2;
		}
		
		if (getY() < cushion) {
			parameter = parameter - 10;
		} else if (getY() > (ymax - cushion)) {
			parameter = parameter - 20;
		}
		
		return parameter;
	}
	
	void avoidWall(int parameter) {
		double toThis = 0;
		if (parameter % 10 == -1) {
			toThis = 90;
		} else if (parameter % 10 == -2) {
			toThis = -90;
		}
		
		if (parameter / 10 == -1) {
			toThis = toThis / 2;
		} else if (parameter / 10 == -2) {
			toThis = 180 - (toThis / 2);
		}
		setAhead(turnTo(toThis) * BOUNCE);
	}
	
	public void onHitWall(HitWallEvent e) {
		avoidWall(checkWall());
	}
	
	int turnTo(double angle) {
	    double ang;
    	int dir;
	    ang = rad2Deg(NormaliseBearing(deg2Rad(getHeading() - angle)));
	    if (ang > 90) {
	        ang -= 180;
	        dir = -1;
	    }
	    else if (ang < -90) {
	        ang += 180;
	        dir = -1;
	    }
	    else {
	        dir = 1;
	    }
	    setTurnLeft(ang);
	    return dir;
	}	
	
	void doScanner() {
		double radarOffset;
		if (getTime() - target.ctime > 4) {
			radarOffset = 360;
		} else {	
			
			radarOffset = getRadarHeadingRadians() - absbearing(getX(),getY(),target.x,target.y);
			
			if (radarOffset < 0)
				radarOffset -= PI/8;
			else
				radarOffset += PI/8; 
		}
		setTurnRadarLeftRadians(NormaliseBearing(radarOffset));
	}
	
/*	public double closeHit(ScannedRobotEvent e, double strength) {
		double aleph = NormaliseBearing(getHeading() + e.getBearing() + 180 - e.getHeading());
		double vj = e.getVelocity() * Math.sin(deg2Rad(aleph));
		double vb = 20 - 3 * strength;
		double angle = rad2Deg(Math.atan(vj/vb));
		return angle;
	}	/* */
	
/*	public double tFun(double d0, double vb, double vt, double angle) {
		return (d0 / (vb - (vt * Math.cos(deg2Rad(angle)))));
	}   /* */
	
	public double tFun(sEnemy target, double firePower) {
		return (target.distance / ((20 - 3 * firePower) - (target.speed * Math.cos(alpha(target)))));
	}
	
	public double deg2Rad(double deg) {
		return (deg * 2 * PI / 360);
	}
	
	public double rad2Deg(double rad) {
		return (rad * 360 / (2 * PI));
	}	
	
	void doGun() {
		long time = getTime() + (int)(target.distance/(20-(3*firePower)));
		double gunOffset = getGunHeadingRadians() - absbearing(getX(),getY(),target.guessX(time),target.guessY(time));
		setTurnGunLeftRadians(NormaliseBearing(gunOffset));
	}
	
	void doGun2() {
		double aleph = alpha(target);
		double gunOffset = getGunHeadingRadians() - (getHeadingRadians() + target.bearing + Math.asin(target.speed * Math.sin(aleph) / (20 - (3 * firePower))));
		setTurnGunLeftRadians(NormaliseBearing(gunOffset));
	}
	
	double alpha(sEnemy target) {
		return target.head - getHeadingRadians() - target.bearing;
	}		
	
	double NormaliseBearing(double ang) {
		if (ang > PI)
			ang -= 2*PI;
		if (ang < -PI)
			ang += 2*PI;
		return ang;
	}
	
	double NormaliseHeading(double ang) {
		if (ang > 2*PI)
			ang -= 2*PI;
		if (ang < 0)
			ang += 2*PI;
		return ang;
	}
	
	public double getrange( double x1,double y1, double x2,double y2 )
	{
		double xo = x2-x1;
		double yo = y2-y1;
		double h = Math.sqrt( xo*xo + yo*yo );
		return h;	
	}
	
	public double absbearing( double x1,double y1, double x2,double y2 )
	{
		double xo = x2-x1;
		double yo = y2-y1;
		double h = getrange( x1,y1, x2,y2 );
		if( xo > 0 && yo > 0 )
		{
			return Math.asin( xo / h );
		}
		if( xo > 0 && yo < 0 )
		{
			return Math.PI - Math.asin( xo / h );
		}
		if( xo < 0 && yo < 0 )
		{
			return Math.PI + Math.asin( -xo / h );
		}
		if( xo < 0 && yo > 0 )
		{
			return 2.0*Math.PI - Math.asin( -xo / h );
		}
		return 0;
	}


	public void onScannedRobot(ScannedRobotEvent e) {
		if ((e.getDistance() < target.distance)||(target.name == e.getName())) {
			double absbearing_rad = (getHeadingRadians()+e.getBearingRadians())%(2*PI);
			target.name = e.getName();
			target.x = getX()+Math.sin(absbearing_rad)*e.getDistance();
			target.y = getY()+Math.cos(absbearing_rad)*e.getDistance();
			target.bearing = e.getBearingRadians();
			target.head = e.getHeadingRadians();
			target.ctime = getTime();
			target.speed = e.getVelocity();
			target.distance = e.getDistance();
			target.energy = e.getEnergy();
		}
	}

	public void onRobotDeath(RobotDeathEvent e) {
		if (e.getName() == target.name)
			target.name = "None";
			target.distance = 10000;
	}

}
class sEnemy {
	String name;
	public double bearing;
	public double head;
	public long ctime;
	public double speed;
	public double x,y;
	public double distance;
	public double energy;
	public double oldEnergy;
	public double guessX(long when)
	{
		long diff = when - ctime;
		return x+Math.sin(head)*speed*diff;
	}
	public double guessY(long when)
	{
		long diff = when - ctime;
		return y+Math.cos(head)*speed*diff;
	}
}	
