package tzu.intel;

import tzu.util.*;
import java.io.*;


/**
 * Defines a likely course of action for an enemy robot
 * when fired at.
 *
 * Heading and speed are compressed to fit within one
 * byte each.  Some accuracy is lost this way, but we
 * only need a "rough" idea where the enemy robot is
 * going.  Also, there is a limit on how much data
 * can be stored to file in the Robocode game.
 *
 * Heading is relative to the enemy's bearing to you
 * and is always positive.  To determine the absolute
 * enemy heading, add it to the bearing and mod 360.
 */
public final class Reaction implements Constants {


    private byte relativeHeading;
    private byte speed;


    /**
     * Create a new reaction using the given relative heading
     * and absolute speed.
     */
    public Reaction(double newRelativeHeading, double newSpeed) {

        relativeHeading = (byte)(Math.round(newRelativeHeading / 1.5) - 128);
        if (newSpeed > MAX_SPEED) newSpeed = MAX_SPEED;
        speed = (byte)(Math.round(newSpeed * 31) - 128);
    }


    /**
     * Create a new reaction using the given relative heading
     * and absolute speed compressed to byte format.
     */
    public Reaction(byte newRelativeHeading, byte newSpeed) {

        relativeHeading = newRelativeHeading;
        speed = newSpeed;
    }


    /** Returns the relative heading. */
    public double getRelativeHeading() {
        return (((double)relativeHeading + 128) * 1.5);
    }


    /** Returns the absolute heading given the target's current bearing. */
    public double getHeading(double targetBearing) {

        return ((targetBearing + (((double)
                relativeHeading + 128) * 1.5)) % A_360);
    }


    /** Return the absolute speed. */
    public double getSpeed() {
        return (((double)speed + 128) / 31);
    }


    /** Return the relative heading in compressed byte format. */
    public byte getByteRelativeHeading() {
        return relativeHeading;
    }


    /** Return the absolute speed in compressed byte format. */
    public byte getByteSpeed() {
        return speed;
    }


    /** Set the relative heading in compressed byte format. */
    public void setByteRelativeHeading(byte newRelativeHeading) {
        relativeHeading = newRelativeHeading;
    }


    /** Set the absolute speed in compressed byte format. */
    public void setByteSpeed(byte newSpeed) {
        speed = newSpeed;
    }
}
