package hirataatsushi;
import java.util.*;
import java.awt.geom.*;
import robocode.*;


//̃NXraybot04Ƃ܂

/**
 * Provides movement for an AbstractRobot.
 */
public final class Movement {




    /** Maximum robot speed. */
    static final double SPEED_MAX               = 8;

    /** Speed used when circling an opponent. */
    static final double SPEED_SPIN_MODE         = 5;

    /** Speed used when a wall collision is imminent. */
    static final double SPEED_NEAR_WALL         = 2;

    /** Maximum distance this robot will approach an enemy bot. */
    static final double DISTANCE_MAX_APPROACH   = 150;

    /** Preferred distance between this robot and an enemy bot. */
    static final double DISTANCE_TARGET_BUFFER  = 500;

    /** Distance this robot will move when circling an enemy bot. */
    static final double DISTANCE_CIRCLE         = 100;

    /** Used to control distance moved when pursuing an enemy bot. */
    static final double DISTANCE_PURSUE_RATIO   = 0.75;

    /** Controls degree of wiggle generated by wiggle() method. */
    static final double WIGGLE_ANGLE            = Math.PI*2/9; // means 40 degrees


    private static final int    FORWARD         = +1;
    private static final int    REVERSE         = -1;




    private static double   currSpeed           = 0;
    private static double   prevSpeed           = 0;
    private static double   prevX               = 0;
    private static double   prevY               = 0;
    private static long     prevTime            = 0;
    private static int      direction           = FORWARD;
    private static Random   random              = new Random();




    private AdvancedRobot r = null;

	private BattleFieldManager field = BattleFieldManager.getInstance();
	private TargetManager manager = TargetManager.getInstance();



    /**
     * Initialize the Movement object.
     *
     * @param yourRobot     A reference to your AdvancedRobot.
     */
    public Movement(AdvancedRobot yourRobot) {

        r = yourRobot;
    }


    public void initialize() {

        currSpeed       = 0;
        prevSpeed       = 0;
        prevX           = r.getX();
        prevY           = r.getY();
        prevTime        = 0;
    }


    /**
     * Change this robot's direction of movement from forward to
     * reverse or vise versa.
     */
    public void changeDirection() {

        if (direction * r.getDistanceRemaining() >= 0) {
            direction *= Math.pow(-1, r.getTime() % 3);
        }
    }

    /**
     * Save data that will allow us to estimate our current speed and
     * change in speed.
     */
    public void recordSpeed() {

        double distance = Math.sqrt(
                          Math.pow(prevX - r.getX(), 2) +
                          Math.pow(prevY - r.getY(), 2));

        prevSpeed = currSpeed;
		System.out.print("prevSpeed ");System.out.println(prevSpeed);
        currSpeed = distance / (r.getTime() - prevTime);
        System.out.print("currSpeed ");System.out.println(currSpeed);
		prevX     = r.getX();
		System.out.print("prevX ");System.out.println(prevX);
        prevY     = r.getY();
		System.out.print("prevY ");System.out.println(prevY);
		System.out.print("currTime ");System.out.println(prevTime);
        prevTime  = r.getTime();
		System.out.print("prevTime ");System.out.println(prevTime);
    }

    /**
     * We are in a one to one battle and the enemy just shot at us.
     * Reverse direction or take a sharp turn.
     *
     * @param target    The enemy robot whose bullets we
     *                  are trying to dodge.
     */
    public void evadeFireMove(Enemy target) {
		System.out.println("Movement.evadeFireMove() begin");
        if (target != null) {
		System.out.println("  target is not null.");
		System.out.println(target.getName());
            double turnRemaining = r.getTurnRemainingRadians(); //* -1;
		System.out.print  ("  turnRemaining ");System.out.println(turnRemaining);  
            double angleToTarget = Helper.normalizeBearingRadians(target.getBearing());
		System.out.print  ("  original angleToTarget ");System.out.println(angleToTarget);
            if (angleToTarget > Math.PI/2) {
                angleToTarget = angleToTarget - Math.PI;
            } else if (angleToTarget < -Math.PI/2) {
                angleToTarget = Math.PI + angleToTarget;
            }
		System.out.print  ("  changed  angleToTarget ");System.out.println(angleToTarget); 
            if (
				Math.abs(angleToTarget) > Math.PI/9 &&
                currSpeed > 3 && currSpeed >= prevSpeed
				) {
                changeDirection();
                setMove(400);
		System.out.println("  changeDirection & setMove(400)");
            } else {
                if (angleToTarget < 0) {
                    setTurn(Math.PI/4);
		System.out.println("  setTurn(Math.PI/4)");
                } else {
                    setTurn(-Math.PI/4);
		System.out.println("  setTurn(-Math.PI/4)");
                }
				setMove(400);
            }
        }
		//r.execute();
		CurrentBotMode.currMode = BotMode.INITIAL;
		System.out.println("Movement.evadeFireMove() end");
    }



    /**
     * Chase the enemy.
     *
     * @param target    The enemy bot to chase.
     */
    public void pursueMove(Enemy target) {


        if (target != null) {
            double robotTurn;

            if (field.nearWall(r.getX(), r.getY())) {
                robotTurn = Helper.normalizeBearingRadians(target.getBearing());
            } else {
                robotTurn = Helper.normalizeBearingRadians(target.getBearing() + wiggle());
            }

            if (r.getTurnRemainingRadians() == 0) setTurn(robotTurn);
            setMove(target.getDistance() * DISTANCE_PURSUE_RATIO);
            setSpeed(SPEED_MAX);
        }
    }


    /**
     * Go around the enemy.
     *
     * @param target    The enemy bot to circle around.
     */
    public void circleMove(Enemy target) {

        if (target != null) {
            double robotTurn;
            if ( field.headedForWall( r.getX(), r.getY(), r.getHeading() ) ) changeDirection();
            robotTurn = Helper.normalizeBearingRadians(target.getBearing());

            robotTurn += (robotTurn > 0 ? -Math.PI*4/9 : +Math.PI*4/9);

            setTurn(robotTurn);
            setMove(DISTANCE_CIRCLE);
            setSpeed(SPEED_SPIN_MODE);
        }
    }


    /**
     * Get closer to the enemy, then go around them in circles.
     *
     * @param target    The enemy robot to chase, then circle.
     */
    public void attackMove(Enemy target) {

        if (target != null) {
            if (target.getDistance() < DISTANCE_MAX_APPROACH) {
                circleMove(target);
            } else {
                pursueMove(target);
            }
        }
    }
	

    /**
     * Move away from other bots and walls.
     */
    public void antiGravityMove() {

        AntiGravity ag = new AntiGravity( manager.getIterator(), field, r.getX(), r.getY());

        if (ag.getForce() > 0) {
            setTurn(Helper.normalizeBearingRadians(ag.getDirection()));
            setMove(ag.getForce());
            setSpeed(SPEED_MAX);

			System.out.print("AntiGravity : setTurn");
			System.out.println(Helper.normalizeBearingRadians(ag.getDirection()));
			System.out.print("AntiGravity : setMove");
			System.out.println(ag.getForce());
			
        }
    }


    /**
     * Ram the enemy.
     *
     * @param target    The enemy robot to be rammed.
     */
    public void ramMove(Enemy target) {

        if (target != null) {
            setTurn(Helper.normalizeBearingRadians(target.getBearing()));
            setMove(target.getDistance());
            setSpeed(SPEED_MAX);
        }
    }


    /**
     * Move away from the current target.  Pick a point a safe
     * distance from the current target and wiggle towards it.
     *
     * @param target    The enemy robot to keep away from.
     */
    public void retreatMove(Enemy target) {

        Point2D.Double p = field.pickBestPosition(target);

        if (r.getTurnRemainingRadians() == 0) {


            double robotTurn =
                Helper.atan(p.y - r.getY(), p.x - r.getX());

            robotTurn = Helper.normalizeBearingRadians(robotTurn) + wiggle();

            setTurn(robotTurn);
        }

        if (r.getDistanceRemaining() == 0) {

            setMove(400);
        }
        setSpeed(SPEED_MAX);

    }


    /**
     * Sets the robot to move towards the given x,y coordinate.
     *
     * @param x     The x coordinate to move to.
     * @param y     The y coordinate to move to.
     */
    public void moveTo(double x, double y) {

        x -= r.getX();
        y -= r.getY();

        setTurn(Helper.normalizeBearingRadians(Helper.atan(y, x)));
        setMove(Math.sqrt(x * x + y * y));
        setSpeed(SPEED_MAX);
    }


    /**
     * Move to the center of the battlefield.
     */
    public void centerMove() {

        if (Math.abs(field.midX - r.getX()) > 50 ||
            Math.abs(field.midY - r.getY()) > 50) {
            if (r.getTurnRemaining() == 0) {

                double robotTurn =
                    Helper.atan((field.midY - r.getY()),
                                   (field.midX - r.getX()));
                r.turnRight(Helper.normalizeBearingRadians(robotTurn));
            }
            setMove(Math.sqrt(Math.pow(field.midX - r.getX(), 2) +
                                 Math.pow(field.midY - r.getY(), 2)));

            setSpeed(SPEED_MAX);
        } else {

            //r.atCenter = true;
        }
    }


    /**
     * Set the robot to move the specified distance forward or
     * reverse depending on current robot direction.
     *
     * @param distance  Distance robot will travel.
     */
    public void setMove(double distance) {

        r.setAhead(distance * direction);
		r.execute();
    }


    /**
     * Set the robot to turn the specified number of degrees.
     *
     * @param angle     +/- degrees.
     */
    public void setTurn(double angle) {

        r.setTurnRightRadians(angle);
    }


    /**
     * Set the maximum speed our bot will travel at.  If we
     * are about to collide with a wall, speed will be limited
     * to avoid damage.  Call setSpeed AFTER turning and
     * movement have been set.
     *
     * @param newSpeed      New maximum speed this robot
     *                      will travel at.
     */
    public void setSpeed(double newSpeed) {

        if (  field.headedForWall( r.getX(), r.getY(), r.getHeading() ) ) {
            r.setMaxVelocity(SPEED_NEAR_WALL);
        } else {
            r.setMaxVelocity(newSpeed);
        }
    }

    /**
     * Returns the current speed this robot is travelling at.
     *
     * @return  The current speed.
     */
    public double getSpeed() {

        return currSpeed;
    }


    /**
     * This method returns the shortest turn required to get the
     * front or rear of this robot pointing towards the newHeading
     * and also sets the robot direction to FORWARD or REVERSE
     * as required.
     *
     * Robots can drive in forward and reverse at the same speed.
     * If this robot is going to move to a specific point, sometimes
     * it's faster to turn the rear of the robot towards that point
     * and drive in reverse then turn about and drive forward.
     *
     * @param   newHeading Direction robot will head in (in degrees).
     * @return  Shortest turn to align robot to newHeading.
     */
    public double shortestTurn(double newHeading) {

        double turn = Helper.normalizeBearingRadians(newHeading);

         if (Math.abs(turn) > 100) {

            changeDirection();
            turn += (turn > Math.PI*100/180 ? -Math.PI : +Math.PI);
        }
        return turn;
    }


    /**
     * Returns the heading that our robot is driving in.
     *
     * @return Heading in degrees that our robot is driving in.
     */
    public double getRealHeading() {

        return (r.getDistanceRemaining() < 0
             ? (r.getHeading() + 180) % 360
             :  r.getHeading());
    }


    /**
     * Returns a angle between plus/minus WIGGLE_ANGLE:  This
     * can be added to robot turns to impart randomness to our
     * robot's movement.
     *
     * @return Random angle in degrees.
     */
    private static double wiggle() {

        return (WIGGLE_ANGLE -
            (WIGGLE_ANGLE * 2 * random.nextDouble()));
    }
}
