package nat.move;

import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.HashMap;

import robocode.RobotStatus;
import robocode.ScannedRobotEvent;
import robocode.StatusEvent;
import robocode.util.Utils;

import nat.base.EventListener;
import nat.base.Movement;
import nat.base.Time;
import nat.base.actors.MovementActor;

public class HawkOnFireMove extends EventListener implements Movement {
	private static HashMap<String, MicroEnemy> enemies = new HashMap<String, MicroEnemy>();
	private static MicroEnemy target;
	private static Point2D.Double nextDestination;
	private static Point2D.Double lastPosition;
	private static Point2D.Double myPos;
	private static double myEnergy;

	private Time time;
	private RobotStatus status;
	
	private boolean isFirst = true;

	@Override
	public void doMovement(MovementActor actor) {
		if (isFirst) {
			nextDestination = lastPosition = myPos = new Point2D.Double(status.getX(), status.getY());
			isFirst = false;
		}
		
		myPos = new Point2D.Double(status.getX(), status.getY());
		myEnergy = status.getEnergy();

		// wait until you have scanned all other bots. this should take around 7
		// to 9 ticks.
		if (target != null && target.live && time.getTime() > 9) {
			doMove(actor);
		}
	}

	private final void doMove(MovementActor actor) {
		double distanceToTarget = myPos.distance(target.pos);
		double distanceToNextDestination = myPos.distance(nextDestination);

		// search a new destination if I reached this one
		if (distanceToNextDestination < 15) {

			// there should be better formulas then this one but it is basically
			// here to increase OneOnOne performance. with more bots
			// addLast will mostly be 1
			double addLast = 1 - Math
					.rint(Math.pow(Math.random(), time.getOthers()));

			Rectangle2D.Double battleField = new Rectangle2D.Double(30, 30,
					actor.getRobot().getBattleFieldWidth() - 60, actor
							.getRobot().getBattleFieldHeight() - 60);
			Point2D.Double testPoint;
			int i = 0;

			do {
				// calculate the testPoint somewhere around the current
				// position. 100 + 200*Math.random() proved to be good if there
				// are
				// around 10 bots in a 1000x1000 field. but this needs to be
				// limited this to distanceToTarget*0.8. this way the bot wont
				// run into the target (should mostly be the closest bot)
				testPoint = calcPoint(myPos, Math.min(distanceToTarget * 0.8,
						100 + 200 * Math.random()), 2 * Math.PI * Math.random());
				if (battleField.contains(testPoint)
						&& evaluate(testPoint, addLast) < evaluate(
								nextDestination, addLast)) {
					nextDestination = testPoint;
				}
			} while (i++ < 200);

			lastPosition = myPos;

		} else {

			// just the normal goTo stuff
			double angle = calcAngle(nextDestination, myPos)
					- status.getHeadingRadians();
			double direction = 1;

			if (Math.cos(angle) < 0) {
				angle += Math.PI;
				direction = -1;
			}

			actor.move(distanceToNextDestination * direction);
			actor.turn(angle = Utils.normalRelativeAngle(angle));
			// hitting walls isn't a good idea, but HawkOnFire still does it
			// pretty often
			actor.setSpeed(Math.abs(angle) > 1 ? 0 : 8d);

		}
	}

	private static final double evaluate(Point2D.Double p, double addLast) {
		// this is basically here that the bot uses more space on the
		// battlefield. In melee it is dangerous to stay somewhere too long.
		double eval = addLast * 0.08 / p.distanceSq(lastPosition);

		for (MicroEnemy en : enemies.values()) {
			// this is the heart of HawkOnFire. So I try to explain what I
			// wanted to do:
			// - Math.min(en.energy/myEnergy,2) is multiplied because
			// en.energy/myEnergy is an indicator how dangerous an enemy is
			// - Math.abs(Math.cos(calcAngle(myPos, p) - calcAngle(en.pos, p)))
			// is bigger if the moving direction isn't good in relation
			// to a certain bot. it would be more natural to use
			// Math.abs(Math.cos(calcAngle(p, myPos) - calcAngle(en.pos,
			// myPos)))
			// but this wasn't going to give me good results
			// - 1 / p.distanceSq(en.pos) is just the normal anti gravity thing
			if (en.live) {
				eval += Math.min(en.energy / myEnergy, 2)
						* (1 + Math.abs(Math.cos(calcAngle(myPos, p)
								- calcAngle(en.pos, p))))
						/ p.distanceSq(en.pos);
			}
		}
		return eval;
	}

	@Override
	public void onScannedRobot(ScannedRobotEvent e) {
		MicroEnemy en = enemies.get(e.getName());

		if (en == null) {
			en = new MicroEnemy();
			enemies.put(e.getName(), en);
		}

		en.energy = e.getEnergy();
		en.live = true;
		en.pos = calcPoint(myPos, e.getDistance(), status.getHeadingRadians()
				+ e.getBearingRadians());

		if (target == null || !target.live || e.getDistance() < myPos.distance(target.pos)) {
			target = en;
		}
	}

	private final static Point2D.Double calcPoint(Point2D.Double p,
			double dist, double ang) {
		return new Point2D.Double(p.x + dist * Math.sin(ang), p.y + dist
				* Math.cos(ang));
	}

	private final static double calcAngle(Point2D.Double p2, Point2D.Double p1) {
		return Math.atan2(p2.x - p1.x, p2.y - p1.y);
	}

	@Override
	public void onStatus(StatusEvent e, Time time) {
		status = e.getStatus();
		this.time = time;
	}

	private final class MicroEnemy {
		public Point2D.Double pos;
		public double energy;
		public boolean live;
	}
}
