package kc.micro;

import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import robocode.AdvancedRobot;
import robocode.RobotDeathEvent;
import robocode.Rules;
import robocode.ScannedRobotEvent;
import robocode.StatusEvent;
import robocode.util.Utils;

/**
* Figment, by Kevin Clark (Kev).
* Micro-sized melee bot that aims similar to Mirage and moves similar to MeleeSeed (by Simonton).
* See https://robowiki.net/wiki/Figment for more information.
*/
public class Figment extends AdvancedRobot {
  // Constants
  private static final double FIELD_WIDTH = 1000.0;
  private static final double FIELD_HEIGHT = 1000.0;
  private static final double AIM_WALL_MARGIN = 17.5;
  private static final int MOVE_WALL_MARGIN = 25;
  private static final int FIRE_ANGLES = 1000;
  private static final int TABLE_SIZE = 126;
  private static final int OPPONENT_HASHES = 256;

  // Aiming data
  private static int[][][] markovTransitionTable = new int[OPPONENT_HASHES][579][TABLE_SIZE + 1];

  // Opponent info
  private static String targetName;
  private static double targetDistance;
  private static double targetVelocity;
  private static double targetHeading;
  private static int targetMarkovState;

  // Our info
  private static double myX;
  private static double myY;
  private static int moveDir = 1;

  @Override
  public void run() {
    setAdjustGunForRobotTurn(true);
    onRobotDeath(null);
  }

  @Override
  public void onStatus(StatusEvent e) {
    // Corner movement inspired by MeleeSeed
    int x = MOVE_WALL_MARGIN + 30 + (int)(targetDistance / 2.5);
    int y = MOVE_WALL_MARGIN;

    if (getDistanceRemaining() == 0) {
      moveDir = -moveDir;
    }
    if (moveDir > 0) {
      y = x;
      x = MOVE_WALL_MARGIN;
    }

    if ((myX = getX()) > FIELD_WIDTH / 2) {
      x = (int)FIELD_WIDTH - x;
    }
    if ((myY = getY()) > FIELD_HEIGHT / 2) {
      y = (int)FIELD_HEIGHT - y;
    }

    double turn;
    setTurnRightRadians(Math.tan(turn = absoluteBearing(x, y) - getHeadingRadians()));
    setAhead(Math.cos(turn) * Point2D.distance(myX, myY, x, y));
  }

  @Override
  public void onScannedRobot(ScannedRobotEvent e) {
    int id;
    int[] aimBins = new int[FIRE_ANGLES];
    String name;
    id = (name = e.getName()).hashCode() & (OPPONENT_HASHES - 1);
    double opponentVelocity;
    double distance;
    double absoluteBearing = getHeadingRadians() + e.getBearingRadians();

    // Check if we should target this opponent
    if ((distance = e.getDistance()) < targetDistance || targetName == name) {
      targetName = name;

      // Radar lock
      if (getGunHeat() < 1) {
        setTurnRadarRightRadians(Double.POSITIVE_INFINITY * Utils.normalRelativeAngle(
          absoluteBearing - getRadarHeadingRadians()));
      }

      // Fire
      double bulletPower;
      if ((bulletPower = Math.log10(getEnergy()) * 325 / (targetDistance = distance)) > 0 &&
          getGunTurnRemaining() == 0) {
        setFire(bulletPower);
      }

      // Aiming
      int state = (int)Math.signum(
          (opponentVelocity = e.getVelocity()) - targetVelocity) + 1 +  // acceleration
          ((int)(8.5 + opponentVelocity) << 2) +  // velocity
          (((int)(-2 * Utils.normalRelativeAngle(  // delta heading
              targetHeading - (targetHeading = e.getHeadingRadians())) /
              Rules.getTurnRateRadians(targetVelocity) + 2.5)) << 7);
      try {
        int[][] table = markovTransitionTable[id];
        int bestBin = 0;
        int i = 0;
        do {
          double predictedX = myX + Math.sin(absoluteBearing) * distance;
          double predictedY = myY + Math.cos(absoluteBearing) * distance;
          id = state;
          int ticks = 1;
          double h = targetHeading;
          double v = opponentVelocity;
          int weight = 100;
          do {
            // Sample a random next state and play forward the movement
            int tableSize;
            if ((tableSize = Math.min(TABLE_SIZE, table[id][TABLE_SIZE])) != 0) {
              id = table[id][(int)(Math.random() * tableSize)];
            } else {
              weight = 5;
            }
            if (!new Rectangle2D.Double(AIM_WALL_MARGIN, AIM_WALL_MARGIN, FIELD_WIDTH -
                (2 * AIM_WALL_MARGIN), FIELD_HEIGHT - (2 * AIM_WALL_MARGIN)).contains(
                  predictedX += Math.sin(h += ((id >> 7) - 2) * Rules.getTurnRateRadians(v) / 2) *
                    (v = ((id >> 2) & 31) - 8),
                  predictedY += Math.cos(h) * v)) {
              weight = 1;
            }
          } while(++ticks * Rules.getBulletSpeed(bulletPower) < Point2D.distance(
              myX, myY, predictedX, predictedY));

          // Update kernel densities and check if new best bin is found
          id = -4;
          //int width;
          //id = -(width = (int)(FIRE_ANGLES * (18 / distance) / (2 * Math.PI)));
          int b;
          do {
            if ((aimBins[b = ((int)(FIRE_ANGLES * absoluteBearing(predictedX, predictedY) /
                (2 * Math.PI)) + id + FIRE_ANGLES) % FIRE_ANGLES] +=
                weight + 1 / (Math.abs(id) + 1)) > aimBins[bestBin]) {
              bestBin = b;
            }
          } while (++id <= 4);
        } while (++i <= 127);

        setTurnGunRightRadians(Utils.normalRelativeAngle(
            2 * Math.PI * bestBin / FIRE_ANGLES - getGunHeadingRadians()));

        // Update aiming data
        if (getGunHeat() < 0.7) {
          (aimBins = table[targetMarkovState])[aimBins[TABLE_SIZE]++] = state;
        }
      } catch(Exception ex) {}

      // Update opponent info
      targetMarkovState = state;
      targetVelocity = opponentVelocity;
    }
  }

  @Override
  public void onRobotDeath(RobotDeathEvent e) {
    setTurnRadarRightRadians(targetDistance = Double.POSITIVE_INFINITY);
  }

  /*@Override
  public void onHitRobot(HitRobotEvent e) {
    targetName = e.getName();
    targetDistance = 0;
  }*/

  private static double absoluteBearing(double targetX, double targetY) {
    return Math.atan2(targetX - myX, targetY - myY);
  }
}