package gadsky;

import robocode.*;

import java.awt.*;
import java.util.ArrayList;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;


public final class Gadsky extends AdvancedRobot {
	private static final double PI2 = Math.PI / 2.0;
	private static final double PI6 = Math.PI / 6.0;
	private static final double MIN_MOVEMENT = 8.0;
	private static final int TOP_SIZE = 50;

	private static final List<Me> myTrack = new ArrayList<Me>(10000);
	public Me my = null;

	private void refreshMe() {
		if (my == null || my.time < getTime()) {
			my = new Me(this);
			myTrack.add(my);
		}
	}


	private static final List<Enemy> enemyTrack = new ArrayList<Enemy>(10000);
	private Enemy enemy = null;

	private void refreshEnemy(ScannedRobotEvent e) {
		refreshMe();

		if (enemy == null || enemy.time < my.time) {
			enemy = new Enemy(this, e, enemy);
			enemyTrack.add(enemy);
		}
	}


	private Vector2D destination = null;

	private void moveToDestination() {
		refreshMe();

		destination = destination != null ? destination : getDestination();

		Vector2D offset = destination.sub(my.position);
		if (offset.length() < MIN_MOVEMENT) {
			destination = null;
			return;
		}

		double offsetAngle = my.bodyDirection.signedAngle(offset);

		if (Math.abs(offsetAngle) < PI2) {
			setTurnLeftRadians(offsetAngle);
			if (Math.abs(offsetAngle) > PI6) return;
			setAhead(offset.length());
		}
		else {
			offsetAngle = Tools.narrow(Math.PI + offsetAngle);
			setTurnLeftRadians(offsetAngle);
			if (Math.abs(offsetAngle) > PI6) return;
			setBack(offset.length());
		}
	}


	private void configure() {
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		setAdjustRadarForRobotTurn(true);
		this.setColors(
			new Color(0.0f, 0.1f, 0.0f),
			new Color(0.0f, 0.2f, 0.0f),
			new Color(0.0f, 0.0f, 0.0f)
		);
	}


	public final void run() {
		configure();

		while (true) {
			refreshMe();

			if (destination == null) destination = getDestination();
			moveToDestination();

			checkRadarLocking();
			execute();
		}
	}


	private int ticksWithoutLock = 1;

	private void checkRadarLocking() {
		if (ticksWithoutLock > 0) setTurnRadarRight(Double.POSITIVE_INFINITY);
		ticksWithoutLock++;
	}


	private double signModifier = Tools.randDouble11();
	private double distanceModifier = Tools.randDouble11() * 50.0;

	private Vector2D getDestination() {
		if (Tools.coin(2)) signModifier = Tools.randDouble11();
		if (Tools.coin(2)) distanceModifier = Tools.randDouble11() * 50.0;

		if (enemy != null) {
			double maxDistance = 0.6 * enemy.distance;

			for (int i = 0; i < 100; i++) {
				double angle = PI2 + Tools.randDouble11() * PI6;
				Vector2D way = enemy.offset.rotated(angle).normalized();
				double sign = Tools.randDouble11() > signModifier ? 1.0 : -1.0;

				double distance =
					distanceModifier +
					Tools.randDouble01() * (maxDistance - distanceModifier) * sign;

				Vector2D offset = way.mul(distance);
				Vector2D destination = my.position.add(offset);

				if (isInsideField(destination)) return destination;
			}
		}

		int x = Tools.randInt(20, (int) getBattleFieldWidth() - 40);
		int y = Tools.randInt(20, (int) getBattleFieldHeight() - 40);
		Vector2D destination = new Vector2D(x, y);

		Vector2D delta = destination.sub(my.position);
		if (delta.length() > 200.0) delta = delta.mul(200.0 / delta.length());
		return my.position.add(delta);
	}


	private boolean isInsideField(Vector2D point) {
		final double minX = getWidth() / 2.0;
		final double minY = getHeight() / 2.0;
		final double maxX = getBattleFieldWidth() - getWidth() / 2.0;
		final double maxY = getBattleFieldHeight() - getHeight() / 2.0;
		return point.x >= minX && point.y >= minY && point.x <= maxX && point.y <= maxY;
	}


	public final void onScannedRobot(ScannedRobotEvent e) {
		refreshMe();
		detectGunFire(e);
		refreshEnemy(e);
		setRadarRotation();
		setGunFire();
	}


	public final void onBulletHit(BulletHitEvent event) {
		enemy.energy = event.getEnergy();
	}


	private void detectGunFire(ScannedRobotEvent e) {
		if (enemy == null) return;

		double energyDrop = enemy.energy - e.getEnergy();
		if (0.099 <= energyDrop && energyDrop <= 3.001) {
			onGunFireDetected(energyDrop);
		}
	}

	private final List<Shot> enemyShots = new ArrayList<Shot>(1000);

	private void onGunFireDetected(double bulletPower) {
		Shot shot = new Shot(enemy.time, enemy.position, bulletPower, my.position);
		enemyShots.add(shot);
	}


	public final void onHitByBullet(HitByBulletEvent e) {
		resolveBulletHit(e.getBullet(), e.getTime());
		destination = getDestination();
	}


	public final void onBulletHitBullet(BulletHitBulletEvent e) {
		resolveBulletHit(e.getHitBullet(), e.getTime() - 1);
	}


	private void resolveBulletHit(Bullet bullet, long time) {
		Vector2D hit = new Vector2D(bullet.getX(), bullet.getY());
		double angle = Tools.normalize(bullet.getHeadingRadians());

		for (int i = enemyShots.size() - 1; 0 <= i && i >= enemyShots.size() - 10; i--) {
			Shot shot = enemyShots.get(i);
			shot.resolve(time, hit, angle, bullet.getVelocity());
		}
	}


	private void setGunFire() {
		if (enemy.energy <= 0.6 && Tools.randDouble11() > 0.5) destination = enemy.position;

		if (getGunHeat() > getGunCoolingRate()) return;

		double bulletPower = getBulletPower();
		double bulletSpeed = Rules.getBulletSpeed(bulletPower);
		Vector2D shot = predictShotByAngularDensity(bulletSpeed);
		double gunDeltaAngle = my.gunDirection.signedAngle(shot);
		setTurnGunLeftRadians(gunDeltaAngle);

		if (Math.abs(gunDeltaAngle) > 1.2 * Rules.GUN_TURN_RATE_RADIANS) return;

		if (enemy.energy > 0.6 && bulletPower >= 0.1) setFire(bulletPower);
	}


	private void setRadarRotation() {
		Vector2D nextDeltaToEnemy = enemy.offset.add(enemy.speed).sub(my.speed);
		double radarDeltaAngle = my.radarDirection.signedAngle(nextDeltaToEnemy);
		setTurnRadarLeftRadians(radarDeltaAngle);
		ticksWithoutLock = 0;
	}


	private Vector2D predictShotBySpeed(double bulletSpeed) {
		double t = enemy.offset.length() / bulletSpeed;
		Vector2D shot = enemy.offset.add(enemy.speed.mul(t));
		for (int k = 0; k < 10; k++) {
			Vector2D prev = shot;
			t = shot.length() / bulletSpeed;
			shot = enemy.offset.add(enemy.speed.mul(t));
			if (shot.sub(prev).length() < 1.0) break;
		}

		return insideShot(shot);
	}


	private Vector2D predictShotByAngularDensity(double bulletSpeed) {
		collectPredictions(bulletSpeed);
		if (predictions.size() < TOP_SIZE) return predictShotBySpeed(bulletSpeed);

		double maxError = 1.0;
		for (Prediction prediction : predictions) {
			maxError = Math.max(maxError, prediction.error);
		}

		final int STEP_COUNT = 100;
		final double step = 2.0 * Math.PI / STEP_COUNT;
		final double[] density = new double[STEP_COUNT + 1];

		for (Prediction prediction : predictions) {
			double angle = Vector2D.EX.signedAngle(prediction.shot);
			int index = (int) Math.round(angle / step);
			if (index < 0) index = STEP_COUNT + index;
			density[index] += 1.0 - Math.pow(prediction.error / maxError, 2.0);
		}

		int maxDensityIndex = 0;
		double maxDensity = 0.0;
		for (int i = 0; i < STEP_COUNT + 1; i++) {
			if (density[i] > maxDensity) {
				maxDensityIndex = i;
				maxDensity = density[i];
			}
		}

		double maxDensityAngle = Tools.narrow(maxDensityIndex * step);
		return Vector2D.EX.rotated(maxDensityAngle);
	}


	private Vector2D predictShotByBestPattern(double bulletSpeed) {
		collectPredictions(bulletSpeed);
		if (predictions.isEmpty()) return predictShotBySpeed(bulletSpeed);

		return predictions.first().shot;
	}


	private final long predictionsTime = -1;
	private final SortedSet<Prediction> predictions = new TreeSet<Prediction>();

	private void collectPredictions(double bulletSpeed) {
		final double T = enemy.offset.length() / bulletSpeed;
		final int PATTERN_LENGTH = 7;
		final int PATTERN_AREA = 5000;
		final int PREDICTION_OFFSET = (int) Math.round(T) + 10;
		final int TRACK_SIZE = enemyTrack.size();

		if (predictionsTime != my.time) {
			predictions.clear();

			int startFrom = enemyTrack.size() - PREDICTION_OFFSET;
			int endBy = enemyTrack.size() - PATTERN_AREA;
			endBy = Math.max(endBy, PATTERN_LENGTH);

			for (int i = startFrom; startFrom >= PATTERN_LENGTH && i >= endBy; i--) {
				double error = 0.0;

				for (int j = 0; j < PATTERN_LENGTH; j++) {
					Enemy current = enemyTrack.get(TRACK_SIZE - j - 1);
					Enemy history = enemyTrack.get(i - j);

					double bodyAngleFactor = current.bodyAngleChange - history.bodyAngleChange;
					double velocityFactor = current.velocity - history.velocity;
					double accelerationFactor = current.acceleration - history.acceleration;
					double distanceFactor = current.distance - history.distance;
					double iWillFireInFactor = my.iWillFireIn - current.iWillFireIn;

					error += bodyAngleFactor * bodyAngleFactor * 2.0;
					error += velocityFactor * velocityFactor;
					error += accelerationFactor * accelerationFactor;
					error += distanceFactor * distanceFactor / 1000.0;
					error += iWillFireInFactor * iWillFireInFactor;
				}

				if (predictions.size() == TOP_SIZE && predictions.last().error < error) continue;

				Vector2D shot = enemy.offset.add(predictEnemyOffsetByTrack(T, i));
				shot = insideShot(shot);
				for (int k = 0; k < 5; k++) {
					Vector2D prev = shot;
					double t = shot.length() / bulletSpeed + 1;
					shot = insideShot(enemy.offset.add(predictEnemyOffsetByTrack(t, i)));
					if (shot.sub(prev).length() < 1.0) break;
				}

				Vector2D enemyPosition = my.position.add(shot);
				predictions.add(new Prediction(error, shot, enemyPosition));
				if (predictions.size() > TOP_SIZE) predictions.remove(predictions.last());
			}
		}
	}

	private Vector2D inside(Vector2D position) {
		final double minX = getWidth() / 2.0;
		final double minY = getHeight() / 2.0;
		final double maxX = getBattleFieldWidth() - getWidth() / 2.0;
		final double maxY = getBattleFieldHeight() - getHeight() / 2.0;

		return new Vector2D (
			Tools.in(position.x, minX, maxX),
			Tools.in(position.y, minY, maxY)
		);
	}


	private Vector2D insideShot(Vector2D shot) {
		Vector2D randevu = my.position.add(shot);
		return inside(randevu).sub(my.position);
	}


	private Vector2D predictEnemyOffsetByTrack(double t, int index) {
		Enemy start = enemyTrack.get(index);
		double rotation = enemy.bodyAngle - start.bodyAngle;

		int turn = (int) t;
		int endIndex = Math.min(index + turn, enemyTrack.size() - 1);
		Enemy end = enemyTrack.get(endIndex);

		if (end.time < start.time) return start.speed.rotated(rotation).mul(t);

		double overTurn = t - turn;
		Vector2D position = end.position.add(end.speed.mul(overTurn));
		Vector2D offset = position.sub(start.position);

		return offset.rotated(rotation);
	}


	private double getBulletPower() {
		double powerByEnemy = getBulletPowerByEnemy();
		double powerByDistance = getBulletPowerByDistance();
		double powerByMe = getBulletPowerByMe();
		return Tools.min3(powerByEnemy, powerByDistance, powerByMe);
	}

	private double getBulletPowerByDistance() {
		if (enemy.distance > 400) return 1.5;
		if (enemy.distance > 200) return 2.0;
		return 3.0;
	}

	private double getBulletPowerByMe() {
		return my.energy - 0.1;
	}

	private double getBulletPowerByEnemy() {
		if (enemy.energy <= 4.3) return (enemy.energy - 0.3) / 4.0;
		return (enemy.energy + 1.7) / 6.0;
	}

	public final void onHitByRobot(HitRobotEvent e) {
		destination = getDestination();
	}


	public final void onPaint(Graphics2D g) {
		double maxError = 1.0;
		for (Prediction prediction : predictions) {
			maxError = Math.max(maxError, prediction.error);
		}

		for (Prediction prediction : predictions) {
			double color = 1.0 - prediction.error / maxError;
			g.setColor(new Color((float) color, 0, 0));
			g.drawOval(
				(int) prediction.position.x - 1,
				(int) prediction.position.y - 1,
				3,
				3
			);
		}

		for (int i = enemyShots.size() - 1; 0 <= i && i >= enemyShots.size() - 7; i--) {
			Shot shot = enemyShots.get(i);
			g.setColor(Color.yellow);
			g.drawOval(
				(int) shot.source.x - 2,
				(int) shot.source.y - 2,
				5,
				5
			);

			long ticks = my.time - shot.time;
			double radius = shot.bulletSpeed * ticks;
			g.drawOval(
				(int) (shot.source.x - radius),
				(int) (shot.source.y - radius),
				(int) (2.0 * radius),
				(int) (2.0 * radius)
			);

			if (shot.resolved) {
				g.setColor(Color.green);
				Vector2D hit = shot.source.add(Vector2D.EX.rotated(shot.heading).mul(radius));

				g.drawLine(
					(int) shot.source.x,
					(int) shot.source.y,
					(int) hit.x,
					(int) hit.y
				);
			}
		}

	}


}
