package rc.yoda.plugin.guns;

import robocode.*;
import rc.yoda.utils.*;
import java.util.ArrayList;
import java.awt.geom.Point2D;
import java.awt.Graphics2D;
import java.awt.Color;

/** 
 * Electrum - by Robert Codd (Gorded) 
 *
 * This code is here by released under the RoboWiki Public Code Licence (RWPCL),
 * datailed on: http://robowiki.net/?RWPCL
 * (Basically it means you must keep the code public if you base any bot on it)
 *
 * Electrum.java : v1.0 -- 2007/05/12
 */


/**
 * Electrum a Guess Factor targeting gun
 *
 * @author Robert Codd
 * @version v1.0
 */
public class Electrum extends Gun
{
	/**
	 * The number of stat bins in this Guess Factor gun
	 */
	private static final int BINS = 47;

	/**
	 * The bin that represents GF 0
	 */
	private static final int MIDDLE_BIN = (BINS - 1) / 2;

	/**
	 * Non-segmented array of Stats for each bin plus the depth of the stats
	 */
	private static double[] minimalWaveStats = new double[BINS + 1];

	protected static double[][][] fastWaveStats = 
		new double[Index.DISTANCE][Index.LATERAL_VELOCITY][BINS + 1];

	protected static double[][][][][] waveStats = 
		new double[Index.DISTANCE][Index.LATERAL_VELOCITY][Index.ACCELERATION][Index.BULLET_POWER][BINS + 1];

	protected static double[][][][][][][] ultimateWaveStats = 
		new double[Index.DISTANCE][Index.LATERAL_VELOCITY][Index.ACCELERATION][Index.BULLET_POWER]
		                                                                       [Index.DELTA_HEADING][Index.NEAR_WALL][BINS + 1];		

	/**
	 * The location of the robot this gun is virtually mounted on
	 */
	private Point2D.Double robotsLocation;

	/**
	 * The location of the enemy to target
	 */
	private Point2D.Double enemyLocation;

	/**
	 * A list of the current wave on the battle field
	 */ 
	private ArrayList _Waves = new ArrayList();

	/**
	 * The current angle this gun wants to shoot at
	 */
	private static double fireAngle = 0;

	/**
	 * The current bullet power this gun wants to shoot
	 */
	private static double bulletPower = 3;

	/** 
	 * The last scanned enemy later velocity
	 */
	private double lastEnemyLateralVelocity = 0;

	private double lastHeading = 0;

	/**
	 * The direction the enemy was previously travelling
	 */
	private int previousDirection = 1;

	/**
	 * Class Constructor specifying the robot this 
	 * gun is virtually mounted on
	 */	
	public Electrum(AdvancedRobot robot) {
		super(robot);
	}

	/**
	 * Event method called by Robocode when this robot's scanner
	 * passes over another robot
	 * 
	 * @param ScannedRobotEvent information about the scanned robot
	 */
	public void onScannedRobot(ScannedRobotEvent e) {
		robotsLocation = new Point2D.Double(robot.getX(), robot.getY());

		double enemyDistance = e.getDistance(),
		enemyHeading = e.getHeadingRadians(),
		enemyBearing = robot.getHeadingRadians() + e.getBearingRadians(),
		enemyLateralVelocity = e.getVelocity() * Math.sin(e.getHeadingRadians() - enemyBearing);

		int direction = enemyLateralVelocity > 0 ? 1 : (enemyLateralVelocity < 0 ? -1 : previousDirection);
		previousDirection = direction;			

		enemyLocation = YUtils.project(robotsLocation, enemyBearing, enemyDistance);
		bulletPower = Math.min(YUtils.getMinBulletPower(e.getEnergy()), YUtils.limit(0.1, 500 / enemyDistance, 3.0));		

		updateWaves();

		Wave wave = new Wave();

		int[] segments = { Index.distanceIndex(enemyDistance),
				Index.lateralVelocityIndex(enemyLateralVelocity),
				Index.accelerationIndex(enemyLateralVelocity - lastEnemyLateralVelocity),
				Index.bulletPowerIndex(bulletPower),
				Index.deltaHeadingIndex(enemyHeading - lastHeading),
				Index.nearWallIndex(enemyLocation) };

		wave.fireLocation = (Point2D.Double)robotsLocation.clone(); 
		wave.bulletVelocity = Laws.getBulletSpeed(bulletPower);
		wave.distanceTraveled = Laws.getBulletSpeed(bulletPower);		
		wave.directAngle = enemyBearing;
		wave.fireTime = robot.getTime();
		wave.direction = direction;
		wave.segments = segments;

		_Waves.add(wave);

		fireAngle = offsetAngle(wave);

		lastHeading = enemyHeading;
		lastEnemyLateralVelocity = enemyLateralVelocity;
	}

	/**
	 * Event method called by Robocode when this robot is
	 * allowed to draw debugging graphics to the screen
	 *
	 * @param Graphics2D graphics that provides drawing method for painting
	 */
	public void onPaint(Graphics2D g) {				
		/*	for (int count = 0; count < _Waves.size(); count++) 
		{
			Wave wave = (Wave)_Waves.get(count);
			double[] stats = getWaveStats(wave);

			double max = 0;
	        for (int bin = 0; bin < BINS; bin++) { max = (stats[bin] > max ? stats[bin] : max); }

			int color = 255 - (255 / Math.max(1, _Waves.size() - 1) * count);			
			g.setColor(new Color(color, color, color));			
			g.drawOval((int)(wave.fireLocation.x - wave.distanceTraveled), (int)(wave.fireLocation.y - wave.distanceTraveled), (int)(wave.distanceTraveled * 2) - 1, (int)(wave.distanceTraveled * 2) - 1);
            for (int bin = 0; bin < BINS; bin++) 
			{
                double size = 3;
                Point2D.Double bullet = projectBin(wave, bin);
				g.setColor(new Color(Color.HSBtoRGB((float)((1 - (stats[bin] / max)) * 240) / 360,1,1)));
                g.fillOval((int)(bullet.x - size),(int)(bullet.y - size),(int)(size * 2),(int)(size * 2));
            }
        } */
	}	

	/**
	 * Returns the power of the bullet this gun wants to 
	 * fire caculated in onScannedRobot
	 * 
	 * @return double power of the bullet this Gun wants to shoot
	 */
	public double getBulletPower() {
		return bulletPower;
	}

	/**
	 * Returns the angle to fire this gun wants to 
	 * fire caculated in onScannedRobot
	 * 
	 * @return double angle at which this Gun wants to shoot
	 */
	public double getFireAngle() {
		return fireAngle;
	}

	/**
	 * Initiates needed aspects of Electrum
	 */
	private void init() {
		if (robot.getRoundNum() == 0) { minimalWaveStats[BINS]++; }
		_Waves.clear();
		_Waves.trimToSize();
	}

	/**
	 * Advances all waves in the air by one tick
	 */
	private void updateWaves() {
		for (int count = 0; count < _Waves.size(); count++) 
		{
			Wave wave = (Wave)_Waves.get(count);

			wave.distanceTraveled = (robot.getTime() - wave.fireTime + 1) * wave.bulletVelocity;
			if (wave.distanceTraveled > enemyLocation.distance(wave.fireLocation)) 
			{
				_Waves.remove(count--);
				logHit(wave, enemyLocation);
			}
		}
	}

	/**
	 * Logs a hit on the specified wave at the specified Location
	 *
	 * @param Wave the wave the hit 
	 * @param Point2D.Double the location at which the wave hit
	 */
	private void logHit(Wave wave, Point2D.Double location) {
		int index = getFactorIndex(wave, location);

		double[] firstStats = minimalWaveStats,
		secondStats = fastWaveStats[wave.segments[0]][wave.segments[1]],
		thirdStats = waveStats[wave.segments[0]][wave.segments[1]][wave.segments[2]][wave.segments[3]],
		fourthStats = ultimateWaveStats[wave.segments[0]][wave.segments[1]][wave.segments[2]][wave.segments[3]][wave.segments[4]][wave.segments[5]];

		firstStats[BINS]++;
		secondStats[BINS]++;
		thirdStats[BINS]++;
		fourthStats[BINS]++;

		double weight = wave.bulletVelocity * (wave.fireLocation.distance(location) / 75);		
		for (int count = 0; count < BINS; count++) 
		{
			firstStats[count] = YUtils.rollingAverage(firstStats[count], Math.pow(0.5, Math.abs(index - count)), 
					Math.min(firstStats[BINS], 10), weight);	

			secondStats[count] = YUtils.rollingAverage(secondStats[count], Math.pow(0.5, Math.abs(index - count)), 
					Math.min(secondStats[BINS], 10), weight);					

			thirdStats[count] = YUtils.rollingAverage(thirdStats[count], Math.pow(0.5, Math.abs(index - count)), 
					Math.min(thirdStats[BINS], 10), weight);	

			fourthStats[count] = YUtils.rollingAverage(fourthStats[count], Math.pow(0.5, Math.abs(index - count)), 
					Math.min(fourthStats[BINS], 10), weight);				
		}

		minimalWaveStats = firstStats;
		fastWaveStats[wave.segments[0]][wave.segments[1]] = secondStats;
		waveStats[wave.segments[0]][wave.segments[1]][wave.segments[2]][wave.segments[3]] = thirdStats;
		ultimateWaveStats[wave.segments[0]][wave.segments[1]][wave.segments[2]][wave.segments[3]][wave.segments[4]][wave.segments[5]] = fourthStats;
	}

	private double[] getWaveStats(Wave wave) {
		if (ultimateWaveStats[wave.segments[0]][wave.segments[1]][wave.segments[2]][wave.segments[3]][wave.segments[4]][wave.segments[5]][BINS] > 0) {
			return ultimateWaveStats[wave.segments[0]][wave.segments[1]][wave.segments[2]][wave.segments[3]][wave.segments[4]][wave.segments[5]];
		}
		else if (waveStats[wave.segments[0]][wave.segments[1]][wave.segments[2]][wave.segments[3]][BINS] > 0) {
			return waveStats[wave.segments[0]][wave.segments[1]][wave.segments[2]][wave.segments[3]];
		}
		else if (fastWaveStats[wave.segments[0]][wave.segments[1]][BINS] > 0) {
			return fastWaveStats[wave.segments[0]][wave.segments[1]];
		}
		return minimalWaveStats;
	}

	/**
	 * Finds the bin with highest hit count
	 *
	 * @return the bin number with the highest hit count
	 */
	private int bestFactor(Wave wave) {
		int bestIndex = MIDDLE_BIN;
		double[] stats = getWaveStats(wave);
		for (int count = 0; count < BINS; count++) { bestIndex = (stats[count] > stats[bestIndex] ? count : bestIndex); }
		return bestIndex;
	}

	/**
	 * Transforms a bin number into its represented angle
	 *
	 * @return double the angle represented by the best factor bin
	 */
	private double offsetAngle(Wave wave) {
		return (bestFactor(wave) - MIDDLE_BIN) * (YUtils.maxEscapeAngle(Laws.getBulletSpeed(bulletPower)) * previousDirection / MIDDLE_BIN);
	}

	/**
	 * Calculate the position of the bullet represented by the
	 * enemy wave and the bin
	 * 
	 * @param the enemy wave the bullet is on
	 * @param the bin representation of the bullet
	 * @return the current location of the bullet
	 */
	private Point2D.Double projectBin(Wave wave, int bin) {
		return YUtils.project(wave.fireLocation,(bin - MIDDLE_BIN) * (YUtils.maxEscapeAngle(wave.bulletVelocity) * wave.direction
				/ MIDDLE_BIN) + wave.directAngle, wave.distanceTraveled);	
	}

	/**
	 * Caculates the bin representation of the targetLocation on the specified Wave
	 * 
	 * @param Wave the wave to caculate angle from the target
	 * @param Point2D.Double the location to calculate angle from the wave source
	 * @return the bin representation of the angle from the wave source
	 */
	private static int getFactorIndex(Wave wave, Point2D.Double targetLocation) {
		double factor = YUtils.normalRelativeAngle(YUtils.absoluteBearing(wave.fireLocation, targetLocation) - wave.directAngle)
		/ YUtils.maxEscapeAngle(wave.bulletVelocity) * wave.direction;
		return (int)YUtils.limit(0,(factor + 1) * MIDDLE_BIN, BINS - 1);
	}

	static class Index {
		public static final int DISTANCE = 10; 
		public static final int LATERAL_VELOCITY = 9;
		public static final int ACCELERATION = 3;
		public static final int BULLET_POWER = 3;
		public static final int DELTA_HEADING = 5;	
		public static final int NEAR_WALL = 3;

		public static int distanceIndex(double distance) {
			return (int) (distance / 150);
		}

		public static int lateralVelocityIndex(double lateralVelocity) {
			return (int) Math.abs(lateralVelocity);
		}

		public static int accelerationIndex(double acceleration) {
			return (acceleration > 0 ? 2 : (acceleration < 0 ? 0 : 1));
		}

		public static int bulletPowerIndex(double power) {
			//	return (int) (power / 0.5);
			return (int) Math.min(2,power / 1);
		}

		public static int deltaHeadingIndex(double deltaHeading) {
			return (int) ((deltaHeading + 10) / 5);
		}

		public static int nearWallIndex(Point2D.Double location) {
			if ((location.x >= YUtils.battleFieldWidth - 150 && location.y >= YUtils.battleFieldHeight - 150) || (location.x <= 150 && location.y <= 150)
					|| (location.x <= 150 && location.y >= YUtils.battleFieldHeight - 150) || (location.x >= YUtils.battleFieldWidth - 150 && location.y <= 150)) {
				return 2;
			}
			else if (location.x <= 150 || location.y <= 150 || location.x >= YUtils.battleFieldWidth - 150 || location.y >= YUtils.battleFieldHeight - 150) {
				return 1;
			}				
			return 0;
		}
	}
}