package ags.muse.recon.waves;

import ags.util.Range;
import ags.util.newtree.KdTree;
import ags.util.points.AbsolutePoint;
import ags.util.points.RelativePoint;

import java.awt.Graphics2D;
import java.awt.Color;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.WeakHashMap;

import robocode.Bullet;
import robocode.BulletHitBulletEvent;
import robocode.Event;
import robocode.HitByBulletEvent;
import robocode.util.Utils;

/**
 * Watches the enemies target!
 * Stores stats in a statically defined HashMap (saves between rounds)
 * with the robot name as the key.
 * 
 * @author Alexander Schultz
 */
public class EnemyTargetingWatcher {
    private static final Map<String, KdTree<Double>> stats = new HashMap<String, KdTree<Double>>();

    private final EWaveDetector ewd;
    private boolean waveDangersOutdated = false;
    private Map<EWave, double[]> waveDangers;

    public EnemyTargetingWatcher(EWaveDetector ewd) {
       this.ewd = ewd;
       this.waveDangersOutdated = false;
       this.waveDangers = new WeakHashMap<EWave, double[]>();
    }

    public void update(Iterable<Event> events) {
        for (Event e : events) {
            if (e instanceof HitByBulletEvent) {
                handleBulletCollide((HitByBulletEvent) e);
            } else if (e instanceof BulletHitBulletEvent) {
                handleBulletCollide((BulletHitBulletEvent) e);
            }
        }

        for (EWave wave : ewd.getWaves()) {
            if (waveDangersOutdated || !waveDangers.containsKey(wave)) {
                waveDangers.put(wave, getDangerFactors(wave));
            }
        }
        
        // Heat wave handling would go here if existing

        waveDangersOutdated = false;
    }

    public double getDanger(EWave wave, Range gfRange)
    {
        // Get danger
        double[] waveDanger = waveDangers.get(wave);
        if (waveDanger == null)
        {
            return 0;
        }

        // Get extended range 1.1x the normal range
        double rangeCenter = gfRange.getCenter();
        double rangeSize = gfRange.getSize();
        Range gfRange11 = new Range(rangeCenter - rangeSize * 0.55, rangeCenter + rangeSize * 0.55);

        // Calculate danger
        double danger = 0;
        for (int i=0; i < waveDanger.length; i+=2)
        {
            double gf          = waveDanger[i+0];
            double probability = waveDanger[i+1];
            danger += probability * (
                gfRange.intersects(gf) ? 1.0 : (
                    gfRange11.intersects(gf) ? 0.1 : 0.0)
                );
        }

        return danger;
    }

    // Returns a list of weighted dangerous guessfactors
    private double[] getDangerFactors(EWave wave) {
        String sourcename = wave.getSource().getName();
        if (wave.isMelee() || !stats.containsKey(sourcename))
        {
            double[] list = new double[4];
            list[0] = 0;                  // GF = HOT
            list[1] = 0.52;               // 52%
            list[2] = wave.getLinearGF(); // GF = linear
            list[3] = 0.48;               // 48%
            return list;
        }
        
        double[] p = getSegmentation(wave);
        List<KdTree.Entry<Double>> l = stats.get(sourcename).nearestNeighbor(p, 20, false);
        
        double ptotal = 0;
        
        int arrayLen = l.size();
        boolean isFull = (arrayLen >= 20);
        if (!isFull) {
            // Make room for hardcoded default values
            arrayLen += 2;
        }

        double[] list = new double[2*arrayLen];
        for (int i=0; i<l.size(); i++) {
            list[i*2+0] = l.get(i).value;
            
            double pweight = 1.0/(1+Math.sqrt(l.get(i).distance));
            ptotal += pweight;
            list[i*2+1] = pweight;
        }

        if (!isFull) {
            double avgP = ptotal / l.size();
            int idx = l.size()*2;
            list[idx + 0] = 0;
            list[idx + 1] = 1.04 * avgP;
            ptotal += 1.04 * avgP;
            list[idx + 2] = wave.getLinearGF();
            list[idx + 3] = 0.96 * avgP;
            ptotal += 0.96 * avgP;
        }

        for (int i=l.size(); i<arrayLen; i++) {
            list[i*2+1] /= ptotal;
        }
        
        // TODO: Add interpolation-focusing!
        
        return list;
    }

    // JGAP SEGMENTATION RESEARCH:
    // Lat:0.49533494812047046 Adv:0.47888435173735894 Dist:0.024915706369010034 Time:8.64993773160576E-4
    // USED BELOW
    private static final int dimensions = 2;
    private static final double[] segmentweights = {0.49533494812047046, 0.47888435173735894, 0.024915706369010034, 8.64993773160576E-4};
    private double[] getSegmentation(EWave wave) {
        double[] segmentdata = new double[dimensions];
        RelativePoint targetVel = wave.getSelfHistory().getVelocity();
        double lateralVelocity = (targetVel.magnitude * Math.sin(targetVel.direction - wave.getZeroAngle()));
        double advancingVelocity = (targetVel.magnitude * -Math.cos(targetVel.direction - wave.getZeroAngle()));
        // double distance = self.getLocation().distance(wave.getOrigin());
        
        segmentdata[0] = Math.abs(lateralVelocity);
        segmentdata[1] = advancingVelocity;
        //segmentdata[2] = distance;
        //segmentdata[3] = ((double)(stateholder.getTotalTime()-stateholder.getTime())+self.getTime());
        
        // Apply segment weights!
        for (int x = 0; x < dimensions; x++) {
            segmentdata[x] *= segmentweights[x];
        }
        
        return segmentdata;
    }

    private void hitAt(EWave wave, AbsolutePoint bulletpos, double power, boolean hit) {
        if ((power <= 0.1) && !hit)
        {
            // 0.1 power bullet-hit-bullet is likely attempted bullet shielding
            // so ignore it for purposes of learning surfing
            return;
        }

        double guessfactor = wave.angleToGF(wave.getOrigin().angleTo(bulletpos));
        
        String sourcename = wave.getSource().getName();
        double[] p = getSegmentation(wave);
        if (!stats.containsKey(sourcename))
        {
            stats.put(sourcename, new KdTree.SqrEuclid<Double>(dimensions, null));
        }
        stats.get(sourcename).addPoint(p, guessfactor);
        
        // The wave dangers may need updating now
        waveDangersOutdated = true;
    }

    // Handle a bullet hitting us
    private void handleBulletCollide(HitByBulletEvent hit) {
        handleBulletCollide(hit.getBullet(), true);
    }
    
    // Handle a bullet hitting one of ours
    private void handleBulletCollide(BulletHitBulletEvent hit) {
        handleBulletCollide(hit.getHitBullet(), false);
    }

    // Handle a bullet hitting something
    private void handleBulletCollide(Bullet bullet, boolean hit) {
        EWave wave = findMatchingWave(bullet);
        // If there's no match? Oh well then, nothing to do with this hit
        if (wave != null) {
            AbsolutePoint bulletpos = AbsolutePoint.fromXY(bullet.getX(), bullet.getY());
            
            // Deal with the bullet hitting us at a given point!
            hitAt(wave, bulletpos, bullet.getPower(), hit);
            
            // Remove the old wave
            ewd.remove(wave);
        } else {
            // Warn that we were hit by a bullet we didn't know about!
            System.out.println("Warning: Hit not matched!");
        }
    }

    private EWave findMatchingWave(Bullet bullet) {
        AbsolutePoint bulletpos = AbsolutePoint.fromXY(bullet.getX(), bullet.getY());
        List<EWave> waves = ewd.getWaves();
        
        for (EWave wave: waves) { 
            // If it's not from the right bot, it's wrong
            if (!wave.getSource().getName().equals(bullet.getName()))
                continue;
            // Not our wave if power is wrong
            if (Math.abs(wave.getBulletPower() - bullet.getPower()) > 0.01)
                continue;
            // If the wave is not over the bullet, it's wrong
            if (Math.abs(wave.getRadius() - wave.getVelocity() - bulletpos.distance(wave.getOrigin())) > 22.0)
                continue;
            // If the bullet heading doesn't match, it's wrong
            double wHeading = wave.getOrigin().angleTo(bulletpos);
            double bHeading = bullet.getHeadingRadians();
            if (Math.abs(Utils.normalRelativeAngle(wHeading - bHeading)) > 0.00001)
                continue;
            
            return wave;
        }
        return null;
    }

    private final Color dangerColor = new Color(255, 255, 0, 255);
    public void paint(Graphics2D g) {

        double maxProb = 0.0;
        for (EWave wave: ewd.getWaves()) { 
            double[] waveDanger = waveDangers.get(wave);
            if (waveDanger != null)
            {
                for (int i=0; i < waveDanger.length; i+=2)
                {
                    double probability = waveDanger[i+1];
                    if (probability > maxProb)
                    {
                        maxProb = probability;
                    }
                }
            }
        }

        for (EWave wave: ewd.getWaves()) { 
            double[] waveDanger = waveDangers.get(wave);
            if (waveDanger != null)
            {
                for (int i=0; i < waveDanger.length; i+=2)
                {
                    double gf          = waveDanger[i+0];
                    double probability = waveDanger[i+1];
                    AbsolutePoint p = wave.getOrigin().project(wave.gfToAngle(gf), wave.getRadius());
                    int x = (int)Math.round(p.x);
                    int y = (int)Math.round(p.y);
                    g.setColor(new Color(
                        dangerColor.getRed(),
                        dangerColor.getGreen(),
                        dangerColor.getBlue(),
                        (int)(255*probability/maxProb))
                        );
                    g.drawLine(x-2, y, x+2, y);
                    g.drawLine(x, y-2, x, y+2);
                }
            }
        }
    }
}