/*
 * Class.java
 *
 * Created on April 7, 2002, 12:35 AM
 */

package tobe.util;

/**
 * BearingVector - for doing calculations
 * This is like a polar vector, except sin and cos are switched.
 * It has a point of origin, a distance and a bearing
 * @author  tobe
 */
public class BearingVector implements java.io.Serializable {

        private double r = 1;
        private double theta = 0;
        private double origoX = 0;
        private double origoY = 0;
        
        public BearingVector() {}
        public BearingVector(double bearing) { setPolar(bearing); }
        public BearingVector(double bearing, double distance) { setPolar(bearing, distance); }
        public BearingVector(double bearing, double distance, double fromX, double fromY) { setPolar(bearing, distance, fromX, fromY); }
		public BearingVector(BearingVector bv) { setVector(bv); }
		/** sets vector to the difference of the arguments */
		public BearingVector(BearingVector to, BearingVector from) { setDifference(to, from); }
        
        /** sets a unit vector (distance = 1) */
        public void setPolar(double bearing) {
            setDistance(1);
            setBearing(bearing);
            setOrigin(0,0);
        }
        
        public void setPolar(double bearing, double distance) {
            setDistance(distance);
            setBearing(bearing);
            setOrigin(0,0);
        }
        
        public void setPolar(double bearing, double distance, double fromX, double fromY) {
            setDistance(distance);
            setBearing(bearing);
            setOrigin(fromX,fromY);
        }
        
        /** sets the vector as the difference between two points.
         * The first is where you go to, the second where you start,
         * think of a mail To: followed by From:
         */
        public void setPoints(double toX, double toY, double fromX, double fromY) {
            if(toX == fromX && toY == fromY) {
                setDistance(0);
                setBearing(0);
            } else {
                setDistance(Math.sqrt(Math.pow(toX-fromX,2)+Math.pow(toY-fromY,2)));
                setBearing(Math.atan2(toX - fromX, toY - fromY));
            }
            setOrigin(fromX, fromY);
        }
        
        /** sets the vector to be a copy of the argument */
        public void setVector(BearingVector v) {
            setPolar(v.getBearing(), v.getDistance(), v.getFromX(), v.getFromY());
        }
        
        /** @deprecated use setSum
		 * this vector is set = v + m*w */
        public void setVector(BearingVector v, BearingVector w, double m) {
            setSum(v,w,m);
        }
        
        /** this vector is set = v + m*w */
		public void setSum(BearingVector v, BearingVector w, double m) {
			double b = w.getBearing();//have to do this, because v or w might be this vector
			double d = w.getDistance();
            setVector(v);
            add(b,d*m);
        }
        
        /** this vector is set = k*v + m*w */
		public void setSum(BearingVector v, double k, BearingVector w, double m) {
			double bv = v.getBearing();//have to do this, because v or w might be this vector
			double dv = v.getDistance();
			double bw = w.getBearing();
			double dw = w.getDistance();
            setPolar(0,0);
			add(bv,dv*k);
            add(bw,dw*m);
        }

		/** this vector is set diff = to - from. Consistent with to.setSum(from, diff, 1.0) */
		public void setDifference(BearingVector to, BearingVector from) {
			setPoints(to.getToX(), to.getToY(), from.getToX(), from.getToY());
		}
        
        public double getBearing() { theta=normalizeAngle(theta); return theta; }
        public double getAbsoluteBearing() {
            if( r < 0 ) return normalizeAngle(theta+Math.PI);
            else return getBearing();
        }
        public void setBearing(double bearing) { theta = bearing;}
        public void setAbsoluteBearing(double bearing) {
            if( r < 0 ) theta = bearing + Math.PI;
            else theta = bearing;
        }
        public double getDistance() { return r; }
        public double getAbsoluteDistance() { return Math.abs(r); }
        public void setDistance(double distance) { r = distance; }
        public void setDistanceToX(double toX) {
            r *= (toX - getFromX())/(getToX() - getFromX());
        }
        public void setDistanceToY(double toY) {
            r *= (toY - getFromY())/(getToY() - getFromY());
        }
        public double getFromX() { return origoX; }
        public double getFromY() { return origoY; }
        public double getToX() { return origoX+r*Math.sin(theta); }
        public double getToY() { return origoY+r*Math.cos(theta); }
		public double getExtendedX(double m) {
			double d = getDistance();
			setDistance(m*d);
			double x = getToX();
			setDistance(d);
			return x;
		}
		public double getExtendedY(double m) {
			double d = getDistance();
			setDistance(m*d);
			double y = getToY();
			setDistance(d);
			return y;
		}
        /** translate the vector */
        public void setOrigin(double fromX, double fromY) {
            origoX = fromX;
            origoY = fromY;
        }
        /** keep origin, but point to new position */
        public void pointTo(double toX, double toY) {
            setPoints(toX, toY, getFromX(), getFromY());
        }
        /** point to same point, but from a different origin */
        public void changeOrigin(double fromX, double fromY) {
            setPoints(getToX(), getToY(), fromX, fromY);
        }
        
        /** sets this vector to point to the point reached by
         * first moving to the endpoint of this vector,
         * translating the argument vector to the end point of
         * this vector, then moving along its bearing
         * for m distances */
        public void add(BearingVector v, double m) {
            add(v.getBearing(), m*v.getDistance());
        }
        
        /** sets this vector to point to the point reached by
         * moving from the endpoint of this vector
         * distance along bearing
         */
        public void add(double bearing, double distance) {
            double x = getFromX();
            double y = getFromY();
            setOrigin(getToX(), getToY());
            setBearing(bearing);
            setDistance(distance);
            changeOrigin(x, y);
        }
        
        /** point to the same point but change the origin
         */
        public void addToOrigin(BearingVector v, double m) {
            addToOrigin(v.getBearing(), m*v.getDistance());
        }
        
        /** point to the same point but change the origin
         */
        public void addToOrigin(double bearing, double distance) {
            double x = getToX();
            double y = getToY();
            setBearing(bearing);
            setDistance(distance);
            setPoints(x, y, getToX(), getToY());
        }

		/** set the endpoint to the closest point on another vector */
		public void dropPerpendicular(BearingVector line) {
			pointTo(line.getFromX(), line.getToX());
			double diff = normalizeAngle(getBearing() - line.getBearing());
			setDistance(getDistance()*Math.sin(diff));
			setBearing(line.getBearing()+Math.PI/2.0);
		}
        
        /** Utility routine to take an angle in radians and returns its normal value between -PI and PI*/
        public static double normalizeAngle(double a) {
            while(a>Math.PI) a-=2*Math.PI;
            while(a<-Math.PI) a+=2*Math.PI;
            return a;
        }
    
}
