/*
 * Written by Kinsen Choy
 */

package kinsen.melee.Guns;
import kinsen.melee.Bullet;
import kinsen.melee.Details;
import kinsen.melee.Utils;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.util.ArrayList;

/*
 * SimpleGuns: The class that handles all the different types of guns.
 * 		It also handles the virtual bullets.
 */
public class SimpleGuns extends Gun
{
	// Contains the classes that store gun hit rate
	private ArrayList gunData = new ArrayList();
	// Contains the classes that store the virtual gun hit rate
	// Contains gun classes to calculate firing parameters
	private ArrayList gunFunctions = new ArrayList();
	private static final byte NUMBER_GUNS = 9;

	// Virtual guns data
	private ArrayList virtualBullets = new ArrayList();
	private ArrayList virtualBulletTypes = new ArrayList();
	// New bullets to be added if bullet is fired
	private ArrayList newVirtualBullets = new ArrayList();
	private ArrayList newVirtualBulletTypes = new ArrayList();

	// My data
	private double myX;
	private double myY;
	private long myTime;

	// Data that is returned
	private byte selectedGun = -1;

	// These two constants synchronized with GunStatistics
	// Minimum percent gun has to achieve or it will continue trying others
	private static final int MINIMUM_CHANGE_PERCENT = 25;
	// Maximum amount of tries for each gun
	private static final int MAXIMUM_GUN_TRIES = 40;

	// Guess factor gun
	private GuessFactor guessFactorGun;
	private ArrayList maximumAngles = new ArrayList();
	private ArrayList currentAngles = new ArrayList();

	private static final int BULLET_SIZE = 8;
	private static final int GF_GUN = 7;
	private static final int PATTERN_GUN = 8;

	/*
	 * Guns
	 * 0: Head On
	 * 1: Circular
	 * 2: Linear
	 * 3: Half-Linear
	 * 4: Stop and Go
	 * 5: Oscillator
	 * 6: Averaged Circular
	 * 7: Guess Factor
	 * 8: Pattern Matching
	 */

	// Creates new instances of the gun classes and resets hit rates
	public SimpleGuns()
	{
		// Remove any exisiting percentage classes
		gunData.clear();
		// Add gun hit rate classes
		for (int i = 0; i < NUMBER_GUNS; i++)
		{
			gunData.add(new GunStatistics());
		}

		// Remove any existing gun classes
		gunFunctions.clear();
		// Add gun classes
		gunFunctions.add(new HeadOn());
		gunFunctions.add(new Circular());
		gunFunctions.add(new Linear());
		gunFunctions.add(new HalfLinear());
		gunFunctions.add(new StopAndGo());
		gunFunctions.add(new Oscillator());
		gunFunctions.add(new AveragedCircular());
		guessFactorGun = new GuessFactor();
		gunFunctions.add(guessFactorGun);
		gunFunctions.add(new PatternMatching());
	}

	/*
	 * recordData: Records data for each gun
	 */
	public void recordData(Details enemy, Details me, long time)
	{
		myX = me.getX();
		myY = me.getY();
		myTime = time;
		for (int i = 0; i < NUMBER_GUNS; i++)
		{
			((Gun) gunFunctions.get(i)).recordData(enemy, me, time);
		}

		int gfBulletIndex = 0;

		// Check virtual bullets
		int i = 0;
		while (i < virtualBullets.size())
		{
			Bullet currentBullet = (Bullet) virtualBullets.get(i);
			double currentX = currentBullet.getCurrentX(myTime);
			double currentY = currentBullet.getCurrentY(myTime);
			double currentDistance = Utils.distance(enemy.getX(), enemy.getY(), currentX, currentY);
			double newDistance = Utils.distance(enemy.getX(), enemy.getY(),
									currentBullet.getCurrentX(myTime + 1), currentBullet.getCurrentY(myTime + 1));

			int gunType = ((Integer) virtualBulletTypes.get(i)).intValue();

			// Virtual bullet hit
			if (currentDistance < 30)
			{
				((GunStatistics) gunData.get(gunType)).recordBullet(true);
				// If guess factor then record
				if (gunType == GF_GUN)
					recordGuessFactor(currentBullet, gfBulletIndex, enemy, time);
				virtualBullets.remove(i);
				virtualBulletTypes.remove(i);
			}
			// Virtual bullet missed
			else if ((currentDistance < newDistance && currentDistance > 40) || Utils.outOfBounds(currentX, currentY))
			{
				((GunStatistics) gunData.get(gunType)).recordBullet(false);
				// If guess factor then record
				if (gunType == GF_GUN)
					recordGuessFactor(currentBullet, gfBulletIndex, enemy, time);
				virtualBullets.remove(i);
				virtualBulletTypes.remove(i);
			}
			else
			{
				i++;
				if (gunType == GF_GUN)
					gfBulletIndex++;
			}
		}
	}

	/*
	 * recordGuessFactor: Records guess factor number
	 */
	private void recordGuessFactor(Bullet gfBullet, int gfBulletIndex, Details enemy, long time)
	{
		double gfIndex = Utils.angleFromPoint(gfBullet.getStartX(), gfBullet.getStartY(), enemy.getX(), enemy.getY());
		gfIndex -= ((Double) currentAngles.get(gfBulletIndex)).doubleValue();
		gfIndex /= ((Double) maximumAngles.get(gfBulletIndex)).doubleValue();
		gfIndex = Math.max(Math.min(gfIndex, 1), -1);
		guessFactorGun.recordGuessFactor(gfIndex);
		currentAngles.remove(gfBulletIndex);
		maximumAngles.remove(gfBulletIndex);
	}

	/*
	 * calculateGun: Calculates location to fire at using the best gun
	 */
	public void calculateGun(double initialFirePower)
	{
		calculateGun(initialFirePower, true);
	}

	public void calculateGun(double initialFirePower, boolean canChangePower)
	{
		// Gets best gun
		byte currentGunIndex = chooseGun();

		// Debug: Only use one gun
//		currentGunIndex = 8;

		// Calculate main gun
		Gun currentGun = (Gun) gunFunctions.get(currentGunIndex);
		currentGun.calculateGun(initialFirePower, canChangePower);
		fireAt = currentGun.getFireAt();
		firePower = currentGun.getFirePower();

		// Reset new virtual bullets
		newVirtualBullets.clear();
		newVirtualBulletTypes.clear();

		for (int i = 0; i < NUMBER_GUNS; i++)
		{
			Gun virtualGun = (Gun) gunFunctions.get(i);
			// Calculates location and fire power
			virtualGun.calculateGun(firePower, false);

			Point virtualFireAt = virtualGun.getFireAt();
			double bulletHeading = Utils.angleFromPoint(myX, myY, virtualFireAt.getX(), virtualFireAt.getY());
			newVirtualBullets.add(new Bullet(myX, myY, bulletHeading, firePower, myTime - 1));
			newVirtualBulletTypes.add(new Integer(i));
		}

		// Record selected gun
		selectedGun = currentGunIndex;
	}

	/*
	 * getGunType: Returns the type of gun used
	 */
	public byte getGunType()
	{
		return selectedGun;
	}

	/*
	 * firedBullet: Whether virtual bullets should be added
	 */
	public void firedBullet(boolean firedBullet)
	{
		if (firedBullet)
		{
			virtualBullets.addAll(newVirtualBullets);
			virtualBulletTypes.addAll(newVirtualBulletTypes);
			currentAngles.add(new Double(guessFactorGun.getCurrentAngle()));
			maximumAngles.add(new Double(guessFactorGun.getMaximumAngle()));
		}
	}

	/*
	 * recordBullet: Records whether a fired bullet hit or missed
	 */
	public void recordBullet(byte gunType, boolean hit)
	{
		((GunStatistics) gunData.get(gunType)).recordBullet(hit);
	}

	/*
	 * chooseGun: Returns the best gun to use
	 */
	private byte chooseGun()
	{
		// Decide on best gun by hit percentage and frequency of usage
		double bestPercent = -1;
		double leastSuccess = Double.POSITIVE_INFINITY;
		// Two possible choices for gun to use
		byte index = -1;
		byte secondIndex = -1;
		for (byte i = 0; i < NUMBER_GUNS; i++)
		{
			// Get hit percent
			GunStatistics data = (GunStatistics) gunData.get(i);
			double gunPercent = data.getPercentage();
			// Use this gun if the hit percentage is better
			if (gunPercent > bestPercent)
			{
				bestPercent = gunPercent;
				index = i;
			}

			double usePercent = gunPercent;
			if (gunPercent == 0)
				usePercent = 5 / MAXIMUM_GUN_TRIES;

			// For other choice base on hit percentage but also number of times the gun was used
			if ((100 - usePercent) * data.getTotal() < leastSuccess)
			{
				leastSuccess = (100 - usePercent) * data.getTotal();
				secondIndex = i;
			}
		}

		// If the best hit-percentage is less than a threshold then use the second choice
		if (bestPercent < MINIMUM_CHANGE_PERCENT)
			index = secondIndex;
		return index;
	}

	/*
	 * newRound: Resets some round-sensitive data
	 */
	public void newRound()
	{
		virtualBullets.clear();
		virtualBulletTypes.clear();
		maximumAngles.clear();
		currentAngles.clear();
	}

	public void debug(Graphics2D g)
	{
		for (int i = 0; i < virtualBullets.size(); i++)
		{
			Bullet currentVirtual = (Bullet) virtualBullets.get(i);
			int gunType = ((Integer) virtualBulletTypes.get(i)).intValue();
			if (gunType == GF_GUN)
				g.setColor(Color.yellow);
			else if (gunType == PATTERN_GUN)
				g.setColor(Color.red);
			else
				g.setColor(Color.cyan);
			g.drawOval((int) currentVirtual.getCurrentX(myTime) - BULLET_SIZE / 2, (int) currentVirtual.getCurrentY(myTime) - BULLET_SIZE / 2, BULLET_SIZE, BULLET_SIZE);
		}

		((PatternMatching) gunFunctions.get(8)).debug(g);
	}
}