package nat.ether.wavesurfing;

import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import nat.ether.utils.*;


public final class MovementPredictor {
	public final static class PredictionStatus extends Point2D.Double {
		private static final long serialVersionUID = 4116202515905711057L;
		public final double heading, velocity;
		public final long time;

		public PredictionStatus(double x, double y, double h, double v, long t) {
			super(x, y);
			heading = h;
			velocity = v;
			time = t;
		}
	}

	public final static PredictionStatus predict(PredictionStatus status, double goAngle, double maxVelocity, double distanceRemaining, Rectangle2D field) {
		double x = status.x;
		double y = status.y;
		double heading = status.heading;
		double velocity = status.velocity;

		// goAngle here is absolute, change to relative bearing
		goAngle -= heading;

		// Normalize angle
		goAngle = M.normalRelativeAngle(goAngle);

		// If angle is at back, consider change direction
		if (M.abs(goAngle) > M.PI / 2) {
			goAngle += M.PI;
			distanceRemaining *= -1;
		}

		// Normalize angle
		goAngle = M.normalRelativeAngle(goAngle);

		// Max turning rate, taken from Rules class
		double maxTurning = M.getTurnRate(velocity);
		heading += M.limit(-maxTurning, goAngle, maxTurning);

		// Get next velocity
		velocity = getVelocity(velocity, maxVelocity, distanceRemaining);

		// Calculate new location
		x += FastTrig.sin(heading) * velocity;
		y += FastTrig.cos(heading) * velocity;

		// check for wall hitting
		if (field != null) {
			boolean hit = false;
			if (x < field.getMinX()) {
				x = field.getMinX();
				hit = true;
			}
			if (y < field.getMinY()) {
				y = field.getMinY();
				hit = true;
			}
			if (x > field.getMaxX()) {
				x = field.getMaxX();
				hit = true;
			}
			if (y > field.getMaxY()) {
				y = field.getMaxY();
				hit = true;
			}
			if (hit)
				velocity = 0;
		}

		// return the prediction status
		return new PredictionStatus(x, y, heading, velocity, status.time + 1l);
	}
	
	public final static double getHeading(double goAngle, double heading) {
		// goAngle here is absolute, change to relative bearing
			goAngle -= heading;

			// Normalize angle
			goAngle = M.normalRelativeAngle(goAngle);

			// If angle is at back, consider change direction
			if (M.abs(goAngle) > M.PI / 2) {
				goAngle += M.PI;
			}

			// Normalize angle
			goAngle = M.normalRelativeAngle(goAngle);
			
			return goAngle;
	}
	
	private final static double getVelocity(double currentVelocity,
			double maxVelocity, double distanceRemaining) {

		// If the distance is negative, then change it to be positive
		// and change the sign of the input velocity and the result
		if (distanceRemaining < 0)
			return -getVelocity(-currentVelocity, maxVelocity, -distanceRemaining);

		final double goalVel;
		if (distanceRemaining == Double.POSITIVE_INFINITY)
			goalVel = maxVelocity;
		else
			goalVel = M.min(getMaxVelocity(distanceRemaining), maxVelocity);

		if (currentVelocity >= 0)
			return M.limit(currentVelocity - M.DECELERATION, goalVel,
					currentVelocity + M.ACCELERATION);

		return M.limit(currentVelocity - M.ACCELERATION, goalVel,
				currentVelocity + maxDeceleration(-currentVelocity));
	}

	private static final double getMaxVelocity(double distance) {
		final double decelTime = M.max(1, M.ceil((M
				.sqrt((4 * 2 / M.DECELERATION) * distance + 1) - 1) / 2));
		final double decelDist = (decelTime / 2.0) * (decelTime - 1)
				* M.DECELERATION;

		return ((decelTime - 1) * M.DECELERATION)
				+ ((distance - decelDist) / decelTime);
	}

	private static final double maxDeceleration(double speed) {
		double decelTime = speed / M.DECELERATION;
		double accelTime = (1 - decelTime);

		return M.min(1, decelTime) * M.DECELERATION + M.max(0, accelTime) * M.ACCELERATION;
	}
}
