package ar.horizon.components.gun;

import static java.lang.Math.*;
import static ar.horizon.util.Util.*;

import java.awt.geom.*;
import java.util.*;

import robocode.BulletHitBulletEvent;

import ar.horizon.*;
import ar.horizon.stats.*;
import ar.horizon.util.*;
import ar.horizon.util.graphics.ColoredShape;
import ar.horizon.util.graphics.DrawingUtil;

/**
 * @author Aaron Rotenberg
 */
public class SolarFlareGun extends Component {
	/**
	 * Determines how strong to fire at a distance.
	 */
	private static final double FIREPOWER_MULTIPLIER = 1.7 * 500;
	/**
	 * Determines how much bullet power is constrained by the remaining energy
	 * of us and the opponent. A higher number means a tighter constraint.
	 */
	private static final double FIREPOWER_ENERGY_CONSTRAINT = 10.0;
	private static final double WALL_STICK = 160.0;
	private static final double IMAGINARY_BULLETS_WEIGHT = 0.5;
	/**
	 * Indicates whether this component attempts to paint graphical debugging
	 * info.
	 */
	private static final boolean GRAPHICAL_DEBUGGING = true;

	private final boolean fireFullPower;

	private static final StatBuffer<FireRecording, Double> gunStats =
			new DynamicClusteringStatBuffer<FireRecording, Double>(
					new SolarFlareBufferParams());

	private final List<Wave> myWaves = new ArrayList<Wave>();
	/**
	 * Fire exactly one full-power shot against disabled opponents, for added
	 * awesome factor. =D
	 */
	private boolean epicDisabledOpponentShotFired = false;

	public SolarFlareGun(final boolean fireFullPower) {
		this.fireFullPower = fireFullPower;
	}

	@Override
	public void onTickEnd() {
		// Make sure we've scanned the opponent at least once--and that he's
		// around so that we can fire at him!
		if (enemyRecording != null && !roundEnded) {
			boolean weAreDisabled = (myRecording.getEnergy() < 0.0001);
			boolean opponentIsRamming = (myRecording.getEnemyDistance() < 150);
			boolean opponentIsDisabled = (enemyRecording.getEnergy() < 0.0001);
			boolean lowEnergyDodgeMode = robot.getEnergy() < 0.6;

			// If we're really low on energy (and not in the Targeting
			// Challenge), we don't fire and just try to dodge. If we get lucky,
			// maybe the opponent will shoot until he disables himself.
			if (!weAreDisabled
					&& (fireFullPower || opponentIsDisabled || !lowEnergyDodgeMode)) {
				double bulletPower;
				if (fireFullPower || opponentIsRamming || opponentIsDisabled) {
					// In these cases, we always fire with the maximum possible
					// bullet power. We still include the minimum energy bound
					// so that our precise prediction is accurate.
					bulletPower = min(robot.getEnergy(), MAX_BULLET_POWER);
				} else {
					// If none of the above apply, we choose our firepower based
					// on distance and constrained by robot energy.
					double baseBulletPower =
							FIREPOWER_MULTIPLIER
									/ myRecording.getEnemyDistance();
					bulletPower =
							limit(MIN_BULLET_POWER, min(myRecording.getEnergy()
									/ FIREPOWER_ENERGY_CONSTRAINT, min(
									enemyRecording.getEnergy()
											/ FIREPOWER_ENERGY_CONSTRAINT,
									baseBulletPower)), MAX_BULLET_POWER);
				}

				// Each tick, move all the current waves and add a new wave.
				// ("!opponentIsDisabled" because we don't want to log data if
				// the opponent is disabled).
				updateWaves(!opponentIsDisabled);
				Wave newWave = addWave(bulletPower, robot.getGunHeat() == 0);

				if (opponentIsDisabled) {
					// If the opponent is disabled, fire head on.
					robot.aimHeadOn();
					// Wait until we're lined up to ensure a perfect shot.
					if (!epicDisabledOpponentShotFired
							&& robot.getGunHeat() == 0.0
							&& abs(angleDifference(
									myRecording.getEnemyAbsoluteBearing(),
									robot.getGunHeadingRadians())) < 0.05) {
						robot.debugPrintLine("YOU MUST DIE!");
						robot.setFire(bulletPower);
						epicDisabledOpponentShotFired = true;
					}
				} else {
					// Set this to false here, in case the disabled opponent's
					// bullet hits us and he comes back to life.
					epicDisabledOpponentShotFired = false;

					// Don't bother calculating where to aim the gun until
					// shortly before we're ready to fire. This is worth a good
					// bit of speed. We always aim if graphical debugging is
					// turned on, to make things prettier.
					if (!GRAPHICAL_DEBUGGING
							&& (opponentIsRamming || robot.getGunHeat() >= 3 * robot.getGunCoolingRate())) {
						// Aim head-on when not preparing to fire, to reduce the
						// distance the gun has to turn. Also aim head-on if the
						// opponent is ramming us.
						robot.aimHeadOn();
					} else {
						// Here's the meat of the gun: the guess factor array
						// generated from the stat buffer.

						// We want to compare against a version of the current
						// enemy recording in which we ARE firing on this tick,
						// even if we aren't.
						FireRecording comparisonRecording =
								new FireRecording(enemyRecording);
						comparisonRecording.setRealBullet(true);

						GuessFactorArray guessFactorArray =
								new GuessFactorArray(
										gunStats.getNearestNeighbors(comparisonRecording),
										IMAGINARY_BULLETS_WEIGHT, false,
										myRecording.getTotalTime());
						debugDrawGuessFactorDots(guessFactorArray,
								newWave.getMaxEscapeAngleClockwise(),
								newWave.getMaxEscapeAngleCounterclockwise());
						robot.aim(getBearingOffsetFromGuessFactor(
								GuessFactorArray.getGuessFactorFromIndex(guessFactorArray.getBestAngleIndex()),
								enemyRecording.getLateralDirection(),
								newWave.getMaxEscapeAngleClockwise(),
								newWave.getMaxEscapeAngleCounterclockwise()));
					}

					if (robot.getGunHeat() == 0) {
						robot.setFire(bulletPower);
					}
				}
			}
		}
	}

	/**
	 * Advances all the gun waves currently on the battlefield. If one hits the
	 * opponent, it is removed and the guess factor it hit at is recorded.
	 */
	private void updateWaves(boolean logHits) {
		for (int i = 0; i < myWaves.size(); i++) {
			Wave wave = myWaves.get(i);

			wave.advance(robot.getTime());
			if (wave.isHit(enemyRecording.getLocation())) {
				if (logHits) {
					gunStats.add(new Pair<FireRecording, Double>(
							wave.getFireRecording(),
							wave.getGuessFactor(enemyRecording.getLocation())));
				}

				myWaves.remove(i);
				i--;
			}
		}
	}

	/**
	 * Creates a new wave from current robot and opponent info, and adds it to
	 * the wave list.
	 */
	private Wave addWave(double bulletPower, boolean realBullet) {
		FireRecording fireRecording = new FireRecording(enemyRecording);
		fireRecording.setRealBullet(realBullet);
		// TODO: Refactor this class so that we can pass the GF array in here
		// and use it for virtual guns calculations.
		Wave newWave = new Wave(fireRecording, robot.getTime(), bulletPower);

		// Precise predict the opponent's maximum escape angles.
		PredictedState escapePointCounterclockwise =
				predictPosition(new PredictedState(enemyRecording), newWave,
						newWave.getFirerRecording().getLocation(), -1,
						EnemyGoAngleGenerator.INSTANCE, WALL_STICK,
						fieldRectangle).getLast();
		PredictedState escapePointClockwise =
				predictPosition(new PredictedState(enemyRecording), newWave,
						newWave.getFirerRecording().getLocation(), 1,
						EnemyGoAngleGenerator.INSTANCE, WALL_STICK,
						fieldRectangle).getLast();

		double maxEscapeAngleCounterclockwise =
				angleDifference(myRecording.getEnemyAbsoluteBearing(),
						absoluteBearing(myRecording.getLocation(),
								escapePointCounterclockwise.getLocation()));
		double maxEscapeAngleClockwise =
				angleDifference(absoluteBearing(myRecording.getLocation(),
						escapePointClockwise.getLocation()),
						myRecording.getEnemyAbsoluteBearing());
		newWave.setMaxEscapeAngleCounterclockwise(maxEscapeAngleCounterclockwise);
		newWave.setMaxEscapeAngleClockwise(maxEscapeAngleClockwise);

		myWaves.add(newWave);
		return newWave;
	}

	private static class EnemyGoAngleGenerator implements GoAngleGenerator {
		public static final EnemyGoAngleGenerator INSTANCE =
				new EnemyGoAngleGenerator();

		private EnemyGoAngleGenerator() {
		}

		public double getGoAngle(Point2D.Double myPosition,
				Point2D.Double enemyPosition, int direction) {
			int nonzeroDirection = sign(direction);
			return absoluteBearing(enemyPosition, myPosition)
					+ (nonzeroDirection * PI / 2);
		}
	}

	private void debugDrawGuessFactorDots(GuessFactorArray guessFactorArray,
			double maxEscapeAngleClockwise,
			double maxEscapeAngleCounterclockwise) {
		if (GRAPHICAL_DEBUGGING) {
			for (ColoredShape shape : DrawingUtil.drawGuessFactorArray(
					guessFactorArray, myRecording.getLocation(),
					myRecording.getEnemyDistance(),
					myRecording.getEnemyAbsoluteBearing(),
					enemyRecording.getLateralDirection(),
					maxEscapeAngleClockwise, maxEscapeAngleCounterclockwise)) {
				robot.debugFillShape(shape);
			}
		}
	}

	@Override
	public void onBulletHitBullet(BulletHitBulletEvent e) {
		// Yes, believe it or not, I've seen this happen.
		epicDisabledOpponentShotFired = false;
	}
}
