package jwst.DAD.utils;

import java.awt.geom.*;

import jwst.DAD.*;
import jwst.DAD.Movement.EnemyWave;
import robocode.*;

/**************************************************************************************
 * 
 * A utilities class for use by any other extending class
 * 
 * @author Jared Wong
 * @author Shawn Thomas
 * @period Period 2
 *
 * Credit: Mr. Peck
 * 
 *************************************************************************************/
public class DADUtils
{
    //this is used to get the distance of the enemy to the closest wall
    //for the guess factor targeting segments(will be explained later)
    private static double BattleFieldWidth;
    private static double BattleFieldHeight;

    /**
     * sets the battlefield width to a variable d
     * @param d - the battle field width
     */
    public static void setBattleFieldWidth(double d)
    {
        BattleFieldWidth = d;
    }

    /**
     * sets the battlefield height to a variable d
     * @param d - the battle field height
     */
    public static void setBattleFieldHeight(double d)
    {
        BattleFieldHeight = d;
    }

    /**
     * passes the method the location of any robot in the battlefield in terms
     * of a point2D. It gets the lateral distance and compares it to the vertical
     * distance and then returns the distance to the closest wall.
     * @param loc - the location of any robot in the battle field in terms of a point2D
     * @return the distance of the robot to the closest wall
     */
    public static double distanceFromWall(Point2D.Double loc)
    {
        double latDis = loc.getX() > BattleFieldWidth/2 ? BattleFieldWidth - loc.getX() : loc.getX();
        double vertDis = loc.getY() > BattleFieldHeight/2 ? BattleFieldHeight - loc.getY() : loc.getY();
        return Math.min( latDis, vertDis );
    }

    /**
     * Normalizes given angle to between 0 and TWO_PI.
     * @param angle - angle to be normalized between 0 and 2PI
     * @return the normalized angle between 0 and 2PI
     */
    public static double normalizeHeading(double angle) 
    {
        return (angle % (Math.PI * 2) + (Math.PI * 2)) % (Math.PI * 2);
    }

    /**
     * Normalizes given angle to between -PI (inclusive) to PI (exclusive).
     * @param angle - angle to be normalized between -PI (inclusive) to PI (exclusive)
     * @return the normalized angle between -PI (inclusive) to PI (exclusive)
     */
    public static double normalizeBearing(double angle) 
    {
        return normalizeHeading(angle + Math.PI) - Math.PI;
    }

    /**
     * 
     * @param a - it takes the angle and if it is less than 0 it returns -1
     * else it returns 1, basically gets the sign of the angle, negative or
     * positive
     * @return -1 or 1 whichever is the sign of the parameter a
     */
    public static int sign(double a) 
    {
        return a < 0 ? -1 : 1;
    }

    /**
     * basically using polar coordinate system to x and y
     *
     * @param sourceLocation location to start at
     * @param angle of the projected point from north
     * @param length /distance of projected point
     * @return the location at which is projected
     */
    public static Point2D.Double project(Point2D.Double sourceLocation,
        double angle, double length) 
    {
        return new Point2D.Double(sourceLocation.x + Math.sin(angle) * length,
            sourceLocation.y + Math.cos(angle) * length);
    }

    /**
     * Find the absolute bearing from us to our enemy
     * 
     * @param source - my (our) location
     * @param target - the enemy robot
     * @return the absolute bearing from us to our enemy
     */
    public static double absoluteBearing(Point2D.Double source, Point2D.Double target) 
    {
        return Math.atan2(target.x - source.x, target.y - source.y);
    }

    /**
     * 
     * @param min value
     * @param max value
     * @param value to be limited
     * @return limit the value between the min and max
     */
    public static double limit(double min, double max, double value) 
    {
        return Math.max(min, Math.min(value, max));
    }

    /**
     * 
     * @param power - the power that we want our bullets to fire, as stated by
     * the robocode physics
     * 
     * @return the velocity
     */
    public static double bulletVelocity(double power) 
    {
        return (20.0 - (3.0*power));
    }

    /**
     * Calculates using the maximum possible escape angle uses the law of sines. 
     * Solves for the maximum angle that the enemy robot can move considering that
     * it moves AT MOST with a velocity of 8 pixels/turn.
     * @param velocity
     * @return the maximum escape angle
     */
    public static double maxEscapeAngle(double velocity) 
    {
        return Math.asin(8.0/velocity);
    }

    /*Predicts the future position we will be at the next time we get an
    * intercept
    * 
    * CREDIT: rozu & Albert*/
    
    /**
     * A method mainly to be used by Wave Surfing
     * 
     * @param w - the enemy wave
     * @param sign - the direction in which the robot is generally going in relation to another
     * @param robot - AdvancedRobot to be predicted
     * @param WALL_STICK - for wall smoothing purposes
     * @param fieldRect - a rectangle representing the robocode field
     * 
     * @return the predicted location
     */
    public static Point2D.Double predictFutureLocation(EnemyWave w, int sign, AdvancedRobot robot, 
        double WALL_STICK, Rectangle2D.Double fieldRect) 
    {
        // The current number of ticks into the future is 0, meaning now
        int futureTicks = 0;

        // Set the future location as the current one because future time is 0
        Point2D.Double predictedLocation = new Point2D.Double(robot.getX(), robot.getY());
        // Set the future velocity as the current one because future time is 0
        double predictedVelocity = robot.getVelocity();
        // Set the future heading as the current one because future time is 0
        double predictedHeading = robot.getHeadingRadians();
        // Stuff to calculate later
        double maxTurnRate, turnAngle, moveDir;

        boolean intercepted = false;

        do //while loop
        {
            // calculate angle to turn based on our current predicted location
            turnAngle = 
                wallSmooth(predictedLocation, 
                    DADUtils.absoluteBearing(w.getFiredLocation(), predictedLocation) + (sign * (Math.PI / 2)), 
                    sign, WALL_STICK, fieldRect) - predictedHeading;

            // Initialize to move right
            moveDir = 1;

            // If we're turning backwards, let's turn and move the other way
            if (Math.cos(turnAngle) < 0) 
            {
                turnAngle += Math.PI;
                moveDir = -1;
            }

            // Normalize the angle
            turnAngle = DADUtils.normalizeBearing(turnAngle);

            // Max turn rate is built into robocode
            // the formula 10 - .75 * |velocity| calculates max turn rate in 
            // degrees per turn
            // therefore I want to multiply by pi/180 to get radians per turn
            maxTurnRate = Math.PI / 180D * (40D - 3D * Math.abs(predictedVelocity)) / 4D;

            // Add the angle we will turn at to the predicted heading
            predictedHeading = DADUtils.normalizeBearing(predictedHeading
                + DADUtils.limit(-maxTurnRate, maxTurnRate, turnAngle));

            // If predictedVelocity and moveDir have different signs, break,
            // otherwise, accelerate
            predictedVelocity += (predictedVelocity * moveDir < 0 ? 2 * moveDir : moveDir);
            // Limit the velocity to possible velocities
            predictedVelocity = DADUtils.limit(-8, 8, predictedVelocity);

            // calculate the new predicted location
            predictedLocation = DADUtils.project(predictedLocation, predictedHeading, predictedVelocity);

            futureTicks++;

            // If our future location's distance from the wave's fired location
            // is less than the distance it will have traveled when we get
            // there, we have intercepted it
            if (predictedLocation.distance(w.getFiredLocation()) < w.getDistanceTraveled()
                            + (futureTicks * w.getVelocity()) + w.getVelocity())
                intercepted = true;
        }
        // Keep predicting until we find an intercept or we have predicted 500
        // ticks ahead
        while (!intercepted && futureTicks < 500);

        return predictedLocation;
    }

    /**
     * Smooth the walls
     * For Example:
     * 
     * I am heading straight towards a wall.
     * 
     * Do I:
     *  <p>
     *   1) Stop abruptly and then turn around or back out
     *   </p>
     *   <p>
     *   or
     *   </p>
     *   <p>
     *   2) Turn gradually as I approach the wall and keep up my speed
     *   </p>
     *   <p>
     * Definitely TWO
     * </p>
     *
     * 
     * @param botLocation - our robots location
     * @param angle - that you ideally want to turn at
     * @param orientation - that you are facing
     * @param WALL_STICK - a stick to determine how far in front of you to check
     * @param fieldRect - a field representing the robocode playing area
     * @return the angle in which you SHOULD turn considering the walls
     */
    public static double wallSmooth(Point2D.Double botLocation, double angle, int orientation, double WALL_STICK, Rectangle2D.Double fieldRect) 
    {
        while (!fieldRect.contains(DADUtils.project(botLocation, angle, WALL_STICK))) 
        {
            angle += orientation*0.05;// just use some arbitrarily small number
            // its in radians you want to check in small
            // increments. 0.05 equates to 2.8638 degrees, about
        }
        return angle;
    }

}