/*
 * davidalves.net presents:
 * ________                   __    __           __      _____    __                 
 * \______ \   __ __   ____  |  |  |__|  _______/  |_   /     \  |__|  ____  _______   ____  
 *  |    |  \ |  |  \_/ __ \ |  |  |  | /  ___/\   __\ /  \ /  \ |  |_/ ___\ \_  __ \ /  _ \ 
 *  |    `   \|  |  /\  ___/ |  |__|  | \___ \  |  |  /    Y    \|  |\  \___  |  | \/(  <_> )
 * /_______  /|____/  \___  >|____/|__|/____  > |__|  \____|__  /|__| \___  > |__|    \____/ 
 *         \/             \/                \/                \/          \/         
 * 
 * Features:
 *  * Virtual bullets. (The first microbot to fit them into 749 bytes!)
 *  * Duelist-style movement.
 *  * Every codesize-reducing trick in the book. :-)
 * 
 * Changes in version 1.2:
 *  * Now uses Conditions for virtual bullets. Saves on codesize since I don't have to manage the Vector of VBs.
 *  * Segments data on everything I can fit, instead of just distance like 1.x
 * 
 * Changes in version 1.11:
 *  *  Fixed a bug where the virtual bullet system would not work against a
 *      target that had never moved. (e.g. TrackFire or SittingDuck)
 * 
 * Changes in version 1.1:
 *  * Removed random(double min, double max) function and changed movement to select destination points from
 *     the entire field, not just the part near us. Movement is now controlled by a timer rather than waiting
 *     for our current move to end.
 *  * Fixed a bug in virtual bullets. Now instead of using the current distance from the enemy for virtual
 *     bullet collision detection, we use the distance at the time the bullet was fired. I was aware of the
 *     issue before but didn't have space to fix it. :-P
 *  * Now fires 101 virtual bullets instead of 21. Not really necessary but since the loop is used for movement
 *     as well, when the new movement needed more iterations, I had to increase the number of virtual bullets,
 *     too. :-P
 * 
 * 
 */

package davidalves.net;

import java.awt.Color;
import java.awt.geom.Point2D;

import robocode.*;
import robocode.util.Utils;

public class DuelistMicro extends AdvancedRobot {
	/* **********************        Constants      ********************** */

	static final int DISTANCE_SEGMENTS = 8;
	static final int TIME_SEGMENTS = 3;
	static final int VELOCITY_SEGMENTS = 3;
	static final int GUESSFACTORS = 25;
	static final int MIDDLE_GUESSFACTOR = (GUESSFACTORS - 1) / 2;

	/* **********************        Declarations      ********************** */

	static double targetDirection;
	static double[][][][] data =
		new double[DISTANCE_SEGMENTS][TIME_SEGMENTS][VELOCITY_SEGMENTS][GUESSFACTORS];
	static double timer;
	static int time;
	static double enemyX, enemyY;
	static final double radiansPerGF = (Math.PI * .3) / (MIDDLE_GUESSFACTOR);

	/* **********************          Run            ********************** */
	public void run() {

		setColors(Color.darkGray, Color.darkGray, Color.cyan);
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForGunTurn(true);

		do {
			turnRadarRightRadians(timer = targetDirection = 1);
		} while (true);

	}

	/* **********************    Event Handlers      ********************** */

	public void onScannedRobot(ScannedRobotEvent e) {
		double distance,
			absoluteBearing,
			bestX,
			bestY,
			destinationX,
			destinationY;
		double myX, myY, attackAngle;
		double bestAttackAngle = Double.NEGATIVE_INFINITY;

		//Not very clean, but this is the most efficient way I've found to set x, y, distance, and bearing
		enemyX =
			(bestX = myX = getX())
				+ Math.sin(
					absoluteBearing =
						e.getBearingRadians() + getHeadingRadians())
					* (distance = e.getDistance());
		enemyY =
			(bestY = myY = getY()) + Math.cos(absoluteBearing) * (distance);

		//out.println("Target is moving" + (targetDirection > 0?"clockwise":"counter-clockwise"));

		/* ***********************     Segmentation       ********************** */

		int distanceIndex = (int) (distance / 160);

		int velocityIndex;

		if ((velocityIndex = (int) (Math.abs(e.getVelocity()) / 2.75)) != 0) {
			targetDirection =
				e.getVelocity()
					* Math.sin(e.getHeadingRadians() - absoluteBearing);
			targetDirection /= Math.abs(targetDirection);
		}

		if (velocityIndex == 0)
			time = 0;

		int moveIndex =
			(int) Math.min(
				TIME_SEGMENTS * 10 * (time++) / distance,
				TIME_SEGMENTS - 1);

		int bestGuessFactor = MIDDLE_GUESSFACTOR;

		int i = GUESSFACTORS;
		do {
			i--;
			if (data[distanceIndex][moveIndex][velocityIndex][i]
				> data[distanceIndex][moveIndex][velocityIndex][bestGuessFactor])
				bestGuessFactor = i;
		} while (i > 0);

		if (e.getEnergy() == 0)
			targetDirection = 0;

		setTurnGunRightRadians(

		//Absolute angle to target
		Math.sin(absoluteBearing - getGunHeadingRadians()

		//Gun offset angle
		+targetDirection
			* radiansPerGF
			* (bestGuessFactor - MIDDLE_GUESSFACTOR)));

		/* ***********************     Spin Radar       ********************** */

		/*
		 * By using this code, the radar sweeps much farther than needed, so we can avoid having to call
		 * setAdjustRadarForGunTurn(true), saving 5 bytes. There's nothing special about 3.0, any value
		 * greater than 2.0 should work.
		 */

		setTurnRadarRightRadians(
			Math.sin(absoluteBearing - getRadarHeadingRadians()));

		/* ***********************     Fire Cannon       ********************** */

		/*
		 * Using && means that the second part of the test, setFireBullet() != null, won't get
		 * called unless the first part evaluates to true. This means we'll only fire if we have
		 * more than 3 energy. We don't have room to check if we're firing at a low energy opponent
		 * and lower the firepower accordingly. :-(
		 */
		BulletWave bw;

		(bw = new BulletWave()).distance = distanceIndex;
		bw.agDistance = moveIndex;
		bw.velocity = velocityIndex;
		bw.angle = absoluteBearing;
		bw.originX = myX;
		bw.originY = myY;
		bw.targetDirection = targetDirection;
		if (getEnergy() > Math.min(4,e.getEnergy())) {

			bw.actualBullet =
				setFireBullet(Math.min(getEnergy() / 8.0, e.getEnergy() / 4.0));
		}
		addCustomEvent(bw);

		i = 100;
		do {
			//Much smaller than the old method, though it requires more iterations to get a good destination
			//Thanks to DrLoco for the suggestion.

			/*
			 * Check if driving to the new random destination will make us drive closer to perpendicular
			 * than the current best destination. If so, change the best destination to the current one.
			 */

			if ((attackAngle =
				Math.abs(
					(Math
						.tan(
							Math.atan2(
								(destinationX =
									50.0
										+ Math.random()
											* (getBattleFieldWidth() - 100.0))
									- myX,
								(destinationY =
									50.0
										+ Math.random()
											* (getBattleFieldHeight() - 100.0))
									- myY)
								- absoluteBearing))))
				> bestAttackAngle) {
				bestX = destinationX;
				bestY = destinationY;
				bestAttackAngle = attackAngle;
			}

		} while (i-- > 0);

		/* ***********************      Drive Tank        ********************** */
		//if(Math.random() == Double.POSITIVE_INFINITY){
		if (timer-- < 0) {
			//destinationY now represents the angle to turn to. Re-using variables saves space! :-)
			//destinationX now represents the angle to turn to, normalized to between -90 and 90 degrees
			
			setTurnRightRadians(
				destinationX =
					Math.atan(
						Math.tan(
							destinationY =
								Utils.normalRelativeAngle(
									((Math.atan2(bestX - myX, bestY - myY)
										- getHeadingRadians()))))));

			setAhead(
				(destinationX == destinationY ? 1 : -1)
					* (Point2D.distance(myX, myY, bestX, bestY)));
			
			//Set the movement timer to a random number between 0 
			//and the time a medium power bullet would reach us.
			timer = Math.random() * distance / 13.0;
			
		}
		execute();
	}

	class BulletWave extends Condition {

		public Bullet actualBullet;
		public double originX,
			originY,
			distanceTraveled,
			angle,
			targetDirection;

		public int distance, agDistance, velocity;

		public boolean test() {
			try {
				//If the BulletWave has gone far enough to hit the enemy, calculate which bin it goes in.
				if ((distanceTraveled += actualBullet.getVelocity())
					> Point2D.distance(enemyX, enemyY, originX, originY)) {
					data[this.distance][this.agDistance][this.velocity][(int)
						((Utils
							.normalRelativeAngle(
								Math.atan2(enemyX - originX, enemyY - originY)
									- angle))
							/ (this.targetDirection * radiansPerGF))
						+ MIDDLE_GUESSFACTOR]++;
					actualBullet = null;
				}
			} catch (Exception e) {
				removeCustomEvent(this);
			}
			return false;
		}

	}
}
