/*
 * PatternAnalyzer - simple pattern analyzer for BlotBot
 * Copyright (C) 2002  Joachim Hofer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * You can contact the author via email (qohnil@johoop.de) or write to
 * Joachim Hofer, Feldstr. 12, D-91052 Erlangen, Germany.
 */
package qohnil.blot;

import qohnil.util.ObjectBuffer;
import qohnil.util.BotMath;
import qohnil.util.Polar;
import qohnil.util.Logger;

import java.util.ArrayList;

/**
 * TODO:
 *      normalize sign of velocity/turnVelocity???
 *
 */
public class PatternAnalyzer {
    ArrayList historyList = new ArrayList(100);
    int sampleSize = 0;
    int maxCapacity = 0;

    int sampleIdx = -1;
    int historyIdx = -1;
    int convertID = 0;

    double error = Double.POSITIVE_INFINITY;

    public PatternAnalyzer(int sampleSize, int maxCapacity) {
        this.sampleSize = sampleSize;
        this.maxCapacity = maxCapacity;
    }

    public void addHistory(ObjectBuffer history) {
        historyList.add(history);
    }

    public void setSampleSize(int sampleSize) {
        this.sampleSize = sampleSize;
    }

    public void setMaxCapacity(int maxCapacity) {
        this.maxCapacity = maxCapacity;
    }

    /**
     * Compares the currently 16 last moves to historical data
     * and returns the index of the best historical matching situation
     *
     * @param maxLookAhead
     *      the maximum time we need to look into the future for the
     *      prediction
     */
    public void findSimilarIdx(int maxLookAhead) {

        historyIdx = -1;
        sampleIdx = -1;

        if (historyList.size() == 0) {
            return;
        }

        ObjectBuffer currentHistory
                = (ObjectBuffer)historyList.get(historyList.size() - 1);

        if (currentHistory.size() < sampleSize) {
            return;
        }

        // prepare current sample
        MovableData[] current = new MovableData[sampleSize];
        for (int i = 0; i < sampleSize; i++) {
            current[i] = (MovableData)currentHistory.get(i);
        }

        boolean matchFound = false;
        error = Double.POSITIVE_INFINITY;

        int patternCount = 0;
        // from latest to oldest history list
        for (int h = historyList.size() - 1; h >= 0; h--) {
            ObjectBuffer history = (ObjectBuffer)historyList.get(h);

            // while we can get complete samples from the history do
            int currentIdx = maxLookAhead;
            while (currentIdx + sampleSize + 1 < history.size() && !matchFound) {

                // System.out.println("comparing 0 to " + currentIdx);
//                double[] diffSum = new double[4]; // initialized to zero anyway
                double diffSum = 0.0;
                // from back (past) of history to front (present)
                for (int i = sampleSize - 1; i >= 0; i--) {
                    // the historical pattern to compare with
                    MovableData pattern = (MovableData)history.get(currentIdx + i);
                    if (pattern == null) {
                        System.out.println("FATAL: pattern == null: " + currentIdx + i + " (" + history.size()
                                + ", " + sampleSize +  ", " + i + ")");
                    }
                    diffSum += Math.abs(pattern.velocity - current[i].velocity);
                    diffSum += Math.toDegrees(Math.abs(pattern.turnVelocity - current[i].turnVelocity)) * 0.8 * 0.125;
/*
                    double[] diffSumD = new double[4];
                    diffSumD[0] = Math.abs(pattern.velocity - current[i].velocity);
                    diffSumD[1] = Math.abs(pattern.velocity + current[i].velocity);
                    diffSumD[2] = Math.abs(pattern.turnVelocity - current[i].turnVelocity) * 0.8;
                    diffSumD[3] = Math.abs(pattern.turnVelocity + current[i].turnVelocity) * 0.8;
                    diffSum[0] += diffSumD[0] + diffSumD[2];
                    diffSum[1] += diffSumD[1] + diffSumD[2];
                    diffSum[2] += diffSumD[0] + diffSumD[3];
                    diffSum[3] += diffSumD[1] + diffSumD[3];
*/
                }
                if (diffSum < error) {
                    error = diffSum;
                    sampleIdx = currentIdx;
                    historyIdx = h;
                    if (error <= 0.005 * 18 * sampleSize) {
                        System.out.println("found good enough match at " + patternCount);
                        matchFound = true;
                        break;
                    }
                }

/*
                for (int i = 0; i < 4; i++) {
                    if (diffSum[i] < error) {
                        // System.out.println("better diffSum = " + diffSum);
                        error = diffSum[i];
                        sampleIdx = currentIdx;
                        historyIdx = h;
                        convertID = i;
                        if (error <= 0.003) {
                            System.out.println("found good enough match at " + patternCount);
                            matchFound = true;
                            break;
                        }
                    }
                }
*/
                // examine next sample
                currentIdx++;

                patternCount++;
                if (patternCount > maxCapacity) {
                    matchFound = true;
                    break;
                }
            }
            if (matchFound) {
                break;
            }
        }
        error /= sampleSize * 18;
//        System.out.println("PATTERN: error = " + error);
    }

    public Polar getPrediction(int lookAheadTime) {
        if (historyIdx == -1 || sampleIdx == -1) {
            Logger.error("INVALID PREDICTION: " + historyIdx + ", " + sampleIdx);
            return null;
        }

        ObjectBuffer history = (ObjectBuffer)historyList.get(historyIdx);

        if (lookAheadTime > sampleIdx) {
            Logger.error("INVALID PREDICTION: lookAhead > sampleIdx: " + lookAheadTime + ", " + sampleIdx);
            return null;
        }

        // return reaction velocity and turnVelocity
        MovableData previous = (MovableData)history.get(sampleIdx);
        MovableData reaction = (MovableData)history.get(sampleIdx - lookAheadTime);

        /*
        if (previous.velocity == 0.0) {
            double err = 0.0;
            for (int i = 0; i < 16; i++) {
                MovableData current = (MovableData)((ObjectBuffer)historyList.get(historyList.size() - 1)).get(i);
                MovableData prev = (MovableData)history.get(sampleIdx + i);
                System.out.println(i + " [" + current.time + " <-> " + prev.time + "]: " + current.velocity + " <-> " + prev.velocity
                        + ", " + current.turnVelocity + " <-> " + prev.turnVelocity);
                err += Math.abs(current.velocity - prev.velocity);
                err += Math.abs(current.turnVelocity - prev.turnVelocity);
            }
            err /= 16 * 16;
            System.out.println("err = " + err);
        }
        */

        // compute bearing and distance
        Polar p = BotMath.coordsToPolar(reaction.coords, previous.coords);
        double angle = p.getBearing() - previous.heading;
        double velocity = p.getDistance();

        switch(convertID) {
            case 0:
                break;

            case 1:
                velocity = -velocity;
                angle = Math.PI + angle;
                break;

            case 2:
                angle = -angle;
                break;

            case 3:
                velocity = -velocity;
                angle = Math.PI - angle;
                break;
        }
        /*
        System.out.println("pattern predicted angle: "
                + Math.round(Math.toDegrees(BotMath.normalizeRelativeAngle(angle)))
                + ", direction: " + Math.round(velocity)
                + ", previous.velocity = " + previous.velocity
                + ", reaction.velocity = " + reaction.velocity
                + ", sampleIdx = " + sampleIdx + ", error = " + error);
                */

        // create angle relative to bot heading
        Polar p1 = new Polar(BotMath.normalizeRelativeAngle(angle), velocity);
        if (p1 == null) {
            Logger.error("FATAL: PatternAnalyzer.getPrediction() cannot allocate Polar!");
        }
        return p1;
    }

    public double getError() {
        return error;
    }

    public int getNumPatterns() {
        int numPatterns = 0;
        for (int i = 0; i < historyList.size(); i++) {
            numPatterns += ((ObjectBuffer)historyList.get(i)).size();
        }
        return numPatterns;
    }
}
