/*
 * Copyright (c) 2004 Marcin Fuszara
 */  
package fushi.PvP1;
import robocode.*;

/* This method uses pattern matching to predict enemy movement
 */
public class PatternMatchingTargeting extends Targeting
{
   // length of pattern when doing pattern matching
   static final int patternLength = 16;
   // maximum number of iterations when predicting enemy movement
   static final int maxIterations = 64;
   // minimum match rank to shoot at target (0..1)
   static final double minMatchRank = 0.5;
   // length of pattern log at which the aim is reliable
   static final double reliableLogLength = 256;
   
   // robot for which this movement is used
   AdvancedRobot robot;
   // robot's gun
   Gun gun;
   // enemy to shoot at
   Enemy enemy;
   
   // simulator used to predict enemy movement
   EnemySimulator enemySimulator = new EnemySimulator();

   /* at construction provide robot, its gun and enemy;
    * \param averageDepth number of records from which the robot data is averaged
    */
   public PatternMatchingTargeting( AdvancedRobot robot, Gun gun, Enemy enemy )
   {
      this.robot = robot;
      this.gun = gun;
      this.enemy = enemy;
   }
   
   /* prepare orders
    */
   public void run()
   {
      if ( gun.isReady() && robot.getGunHeat() == 0 ) {
         PatternMatcher patternMatcher = enemy.patternMatcher;
         // try to find a match
         PatternMatcher.Match match = patternMatcher.findMatch( patternLength );
         if ( match == null )   // could not find a match
            return;
            
         Logger.log( "pattar", "Found a match at " + Integer.toString( match.index )
                              + " with rank " + Double.toString( match.rank ) );
                              
         if ( match.rank < minMatchRank ) // do not shoot at too unreliable target
            return;
         
         Enemy.Data enemyData = enemy.getData( 0 );
         if ( enemyData == null ) // no data available
            return;
            
         // get latest enemy position
         Vector2d enemyPosition = enemyData.position;
         
         // compute distance to enemy robot
         Vector2d robotPosition = new Vector2d( robot.getX(), robot.getY() );
         double distance = Vector2d.distance( enemyPosition, robotPosition );
         
         /* compute aim confidence as a function of match rank
          * and length of pattern log
          */
         double confidence = patternMatcher.getLogLength() / reliableLogLength;
         confidence = Util.clipDouble( confidence, 0, 1 );
         confidence *= match.rank;
         // use selector to determine fire power
         double firePower = FirePowerSelector.determineFirePower( 
            robot.getEnergy(), distance, confidence, false );
         
         if ( firePower < 0 )  // shoot only if it is ok to shoot
            return;
            
         // simulate enemy
         enemySimulator.set( enemyPosition, enemyData.heading );
         // get index of first pattern log record from which simulation will start
         int index = match.index + patternLength;
         PatternMatcher.PatternData patternData = patternMatcher.getData( index );
         ++index;
         if ( patternData == null )  // no data
            return;
         // enemy data comes from the last frame, so i need first to predict current enemy position
         enemySimulator.frame( patternData.speed, patternData.headingDelta );
         
         // iteratively select the target
         int iteration = 0;
         do {
            patternData = patternMatcher.getData( index );
            ++index;
            if ( patternData == null )  // no more data => cannot predict enemy position
               return;
            // simulate a frame
            enemySimulator.frame( patternData.speed, patternData.headingDelta );
            iteration++;
            // update enemy position
            enemyPosition = enemySimulator.getPosition();
         } while ( gun.estimateTimeNeeded( enemyPosition, firePower ) + 1 > iteration 
                 && iteration < maxIterations );

         if ( iteration >= maxIterations )
            return;   // could not find enemy position in the future
            
         // position found; check if it is in the arena   
         if ( !pointInArena( enemyPosition ) )
            return;     // do not shoot outside the arena
            
         // order to shoot
         gun.fireAt( enemyPosition, firePower, robot.getTime() + iteration );
      } else if ( gun.isReady() ) {
         Enemy.Data enemyData = enemy.getData( 0 );
         if ( enemyData == null ) // no data available
            return;
         gun.aimAt( enemyData.position );
      }
   }
   
   /* checks if given point is in the arena;
    */
   boolean pointInArena( Vector2d point )
   {
      double arenaWidth = robot.getBattleFieldWidth();
      double arenaHeight = robot.getBattleFieldHeight();
      
      if ( point.x < 0 || point.x > arenaWidth 
          || point.y < 0 || point.y > arenaHeight )
         return false;
      return true;
   }
   
}