package tzu.gun;

import tzu.util.*;
import tzu.intel.*;

/**
 * Provides estimated aiming using iteration.
 * This technique takes into account turning rate and de/acceleration.
 * NOTE: CHANGES TO THIS CLASS MAY ALSO APPLY TO PatternAim.
 */
public class EstimateAim implements Constants {

    /**
     * Given the current motion of a target and a bullet, uses iteration
     * to calculate a FireSolution that will allow the bullet to hit the
     * target, provided the target continues to follow the same motion.
     *
     * @return FireSolution or null if no reasonable solution found.
     */
    public static FireSolution calculate(Motion target, Motion bullet) {

        double x    = target.getX();
        double y    = target.getY();
        double s    = target.getSpeed();
        double a    = target.getChgSpeed();
        double h    = target.getHeading();
        double t    = target.getChgHeading();
        double bx   = bullet.getX();
        double by   = bullet.getY();
        double bs   = bullet.getSpeed();

        double minX = BattleField.getMinX();
        double minY = BattleField.getMinY();
        double maxX = BattleField.getMaxX();
        double maxY = BattleField.getMaxY();

        long time   = 0;
        long bTime  = (long)(BotMath.calcDistance(x,y, bx, by)/bs);

        while (s != 0.0  && time < bTime  &&
               x >= minX && y >= minY    &&
               x <= maxX && y <= maxY)   {

            time++;
            if (t != 0.0) {
                h += BotMath.calcTurnRate(s) * ((t < 0) ? -1 : +1);
            }
            s = BotMath.limit(s + a, 0, MAX_SPEED);
            x = BotMath.calcX(x, h, s);
            y = BotMath.calcY(y, h, s);
            bTime = (long)(BotMath.calcDistance(x,y, bx, by)/bs);
        }

        x = BotMath.limit(x, minX, maxX);
        y = BotMath.limit(y, minY, maxY);
        bTime = (long)(BotMath.calcDistance(x, y, bx, by)/bs);

        return new FireSolution(
            target.getBot(),
            BotMath.calcHeading(bx, by, x, y),
            FireSolution.AIM_ESTIMATED,
            x, y, bTime);
    }
}