/*
 * Created by IntelliJ IDEA.
 * User: Joachim
 * Date: 21.06.2002
 * Time: 17:31:30
 */
package qohnil.util;

public class MultimodeStatistics {
    long[] numTotal;
    long[] numHits;
    int numModes;
    boolean invert;

    /**
     * Initializes the statistics
     * @param numModes
     *      how many modes we should keep statistics for
     * @param invert
     *      true if you want to measure success with inverse hit ratios
     */
    public MultimodeStatistics(int numModes, boolean invert) {
        this.numModes = numModes;
        this.invert = invert;
        reinit();
    }

    public void reinit() {
        numTotal = new long[numModes];
        numHits = new long[numModes];
    }

    public void add(int mode) {
        numTotal[mode]++;
    }

    public void add() {
        for (int i = 0; i < numModes; i++) {
            numTotal[i]++;
        }
    }

    public void hit(int mode) {
        numHits[mode]++;
    }

    public double getHitRatio(int mode) {
        if (numTotal[mode] < 30) {
            // assume perfect if no data available
            if (invert) {
                return 0.0;
            } else {
                return 1.0;
            }
        }
        return (double)numHits[mode] / numTotal[mode];
    }

    public long getNumTotal(int mode) {
        return numTotal[mode];
    }

    public long getNumHit(int mode) {
        return numHits[mode];
    }

    public int getBestMode() {
        double bestRatio = Double.NEGATIVE_INFINITY;
        int bestMode = -1;
        for (int i = 0; i < numModes; i++) {
            double hitRatio = getHitRatio(i);
            if (hitRatio > bestRatio) {
                bestRatio = hitRatio;
                bestMode = i;
            }
        }

        return bestMode;
    }

    /**
     * Decides which mode to use probabilistically.
     * Does not always use the best mode, just most often!
     * @param power
     *      how much you want to emphasize the best modes
     */
    public int decideMode(int power) {

        // compute all hit ratios (and their total sum
        double sum = 0.0;
        double[] hitRatio = new double[numModes];
        for (int i = 0; i < numModes; i++) {
            if (invert) {
                hitRatio[i] = 1 - getHitRatio(i);
            } else {
                hitRatio[i] = getHitRatio(i);
            }

            // take hit ratio to power of "power"
            double temp = 1.0;
            for (int j = 0; j < power; j++) {
                temp *= hitRatio[i];
            }
            hitRatio[i] = temp;
            sum += hitRatio[i];
        }

        // now decide probabilistically
        double rnd = BotMath.random.nextDouble() * sum;
        double accumulate = 0.0;
        for (int i = 0; i < numModes; i++) {
            accumulate += hitRatio[i];
            if (rnd <= accumulate) {
                return i;
            }
        }
        return (numModes - 1);
    }
}
