package dam;

import robocode.*;
import dam.targetting.*;

import java.awt.Color;
import java.util.*;
import java.io.*;
import java.util.zip.*;

import dam.movement.*;
import dam.util.BotMath;

/**
* MogBot. CatBot's harder brother.
* Version 2.4. It's the pattern targetting that sets him apart.
* Version 2.8. Now he learns when to use regular targetting.
*/
public class MogBot extends MasterBot
{			
	int direction = 1;				   
	Random rand = new Random();
	int hits = 0;
	private int radarDirection = 1;
	int radarCounter = 0;
	MovementStrategy move;
	boolean locked = false;
	private long jiggleTime = 0;
	private long classicTime = 0;
	private long moggyTime = 0;
	private long lastChangeTime = 0;
	private Target lastTarget = null;

	public void run() {
		move = new MoggyMovement();
		move.setBot(this);
		// load the data from the data file mog.dat
		loadSettings();
		
		setColors(Color.orange,Color.blue,Color.yellow);
		setAdjustGunForRobotTurn(true);
		addCustomEvent(new RadarTurnCompleteCondition(this));
		addCustomEvent(new Condition("MoveDone"){
			public boolean test(){
				return isMoveDone();
			}
		});
		addCustomEvent(new Condition("NearWall"){
			public boolean test(){
				return isCloseToWall();
			}
		});
		addCustomEvent(new TurnCompleteCondition(this));
		setAdjustRadarForGunTurn(true);
		setTurnRadarRight(360);
		move.doMove();
		out.println("There are " + getAllTargets().size() + " targets.");
		while(true) {
			doGun();
			execute();
		}
	}
	
	/**
	 * Calls saveSettings to maintain the data on targetting the baddies
	 * between rounds. 
	 */
	public void onDeath(DeathEvent evt)
	{
		if(allTargets.size() == 1){
			double d = allTargets.get(0).getFearFactor();
			if(d > 50) d = 0;
			else d = 500;
			allTargets.get(0).setFearFactor(d);
		}
		SaveThread t = new SaveThread();
		t.start();
	}
	
	public void loadSettings(){
		if(getRoundNum() < 1){ // don't load the settings on the first round
			return;
		}
		try{
			FileInputStream r = new FileInputStream(getDataFile("mogbot.dat"));
			ZipInputStream zis = new ZipInputStream(r);
			zis.getNextEntry();
			ObjectInputStream is = new ObjectInputStream(zis);
			allTargets = (TargetList) is.readObject();
			r.close();
			for(int i = 0; i < allTargets.size(); i++)
			{
				Target t = allTargets.get(i);
				t.setBot(this);
			}
		}
		catch(Exception ex){
			out.println(ex);
		}
		
	}
	
	public class SaveThread extends Thread
	{
		public void run(){
			saveSettings();
		}
	}
	
	
	public void saveSettings(){
		try{
			RobocodeFileOutputStream fo = new RobocodeFileOutputStream(getDataFile("mogbot.dat"));
			ZipOutputStream zos = new ZipOutputStream(fo);
			zos.putNextEntry(new ZipEntry("data.dat"));
			ObjectOutputStream os = new ObjectOutputStream(zos);
			os.writeObject(allTargets);
			zos.closeEntry();
			zos.finish();
			fo.close();
		}
		catch(Exception ex){
			ex.printStackTrace(out);
		}
	}
	
	public void selectTarget(Target t){
		super.selectTarget(t);
		System.out.println("Accuracy: " + target.getAccuracy());
		System.out.println("Strategy: " + target.getStrategy().getClass().getName());
		System.out.println("Fear factor: " + target.getFearFactor());
	}
	
	private String padString(String str, int sz)
	{
		while(str.length() < sz) str += " ";
		return str;
	}
	
	public boolean isMoveDone()
	{
		return move.isMoveDone();
	}
	
	public boolean isCloseToWall(){
		return distanceToWall() < 36;
	}
	
	public void onSkippedTurn(SkippedTurnEvent e)
	{
		out.println("Skipped a turn.");
	}
	
	public void onCustomEvent(CustomEvent e)
	{
		if(e.getCondition() instanceof RadarTurnCompleteCondition)
		{
			sweepRadar();
		}
		else if(e.getCondition() instanceof TurnCompleteCondition)
		{
			move.onTurnComplete();
		}
		else if(e.getCondition().getName().equals("MoveDone"))
		{
			move.onMoveComplete();
		}
		else if(e.getCondition().getName().equals("NearWall"))
		{
			move.onNearWall();
		}
		
	}
	
	
	// constrain the turn to be one of the cardinal directions, and at least 22 degs
	private double constrainTurn(double angle, double degrees)
	{
		double cusp = Math.toRadians(degrees);
		double base = angle % cusp;
		angle = angle - base;
		if((angle >= 0) && (angle < cusp)) angle = cusp;
		else if((angle < 0) && (angle > -(cusp))) angle = -(cusp);
		return angle;
	}
	
	// constrain the turn to be one of the cardinal directions, and at least 22 degs
	private double constrainTurnAbs(double angle, double degrees)
	{
		angle = getHeadingRadians() + angle;
		double cusp = Math.toRadians(degrees);
		double base = angle % cusp;
		angle = angle - base;
		int which = BotMath.sign(angle);
		return getHeadingRadians() - angle;
	}

	
	
	/**
	* Each time we are hit, we will change our strategy, if it is a duel.
	* Call the super class's onHitByBullet to update the HitPoint
	*/
	public void onHitByBullet(HitByBulletEvent e){
		super.onHitByBullet(e);
		changeMovementStrategy();
		Target t = allTargets.get(e.getName());
		if(t != null){
			if(getOthers() > 1) t.addFearFactor(e.getPower() * 20);
			t.addScore(e.getPower());
		}
		
	}
	
	/**
	* Note that we typically will not do this unless we are
	* in a duel. We could alter this so we do it if there are
	* only two other bots, but it may be risky
	*/
	
	
	
	
	
	
	public void changeMovementStrategy()
	{
		long ctime = getTime() - lastChangeTime;
		if(getOthers() < 2){
			if(move instanceof ClassicMovement){
				classicTime += ctime;
				move = new JiggleMove();
			}
			else if(move instanceof JiggleMove){
				jiggleTime += ctime;
				move = new MoggyMovement();
			}
			else if(move instanceof MoggyMovement){
				moggyTime += ctime;
				move = new ClassicMovement();
			}
			
			lastChangeTime = getTime();
			move.setBot(this);
		}
	
	}
	
	
	/**
	 * Reduce our fear of this bot a little if we hit him
	 */
	public void onBulletHit(BulletHitEvent e)
	{
		super.onBulletHit(e);
		Target t = allTargets.get(e.getName());
		if(t != null){
			if(getOthers() > 1) t.addFearFactor(e.getBullet().getPower() * -20);
			t.addScore(e.getBullet().getPower() * -1);
		}
	}
	
	
	public void onBulletMissed(BulletMissedEvent e)
	{
		super.onBulletMissed(e);
		if(getHits() < 12) // changeMovementStrategy
		{
			changeMovementStrategy();
			setHits(0);
		}
	}
	
			
	public double distanceToWall()
	{
		double t_dist = Math.min(getX(),getY());
		double l_dist = Math.min(getBattleFieldWidth() - getX(), getBattleFieldHeight() - getY());
		return Math.min(t_dist, l_dist);
	}
	
		
	void setTurnToRadians(double angle)
	{
		angle = normaliseBearing(angle);
		if(angle > PI/2){
			angle -= PI;
			direction = -1;
		}
		else if(angle < -(PI/2))
		{
			angle += PI;
			direction = -1;
		}
		else{
			direction = 1;
		}
		setTurnRightRadians(angle);
	}
	
	
	



	/**
	 * Method doGun.
	 */
	public void doGun() 
	{
		TargettingStrategy ts = target.getStrategy();		
		if(targets.contains(target.getName()))
		{
			double firePower = ts.aim();
			setTurnGunRightRadians(ts.getGunOffset());
			if(firePower > 0){
				if(getEnergy() > firePower){
					setFire(firePower);
				}
			}
		}
	}
	
	
	
	/**
	* Target whoever hit you
	*/
	public void onHitRobot(HitRobotEvent e)
	{
		if(!target.nameEquals(e.getName()))
		{
			ScannedRobotEvent evt = new ScannedRobotEvent(e.getName(), e.getEnergy(), e.getBearing(),
				36, 0, 0);
			Target t = targets.get(evt.getName());
			if(t != null){
				selectTarget(t);
			}
			
		}	
	}
	
	void sweepRadar()
	{
		double maxBearingAbs = 0, maxBearing = 0;
		int scannedBots = 0;
		for(int i = 0; i < targets.size(); i++)
		{
			Target tmp = targets.get(i);
			if(getTime() - tmp.getLastScanTime() < 20)
			{
				double bearing = BotMath.normaliseBearing((tmp.getBearing() + getHeadingRadians()) - getRadarHeadingRadians());
				if(Math.abs(bearing) > maxBearingAbs){
					maxBearingAbs = Math.abs(bearing);
					maxBearing = bearing;
				}
				scannedBots++;
			}
		}
		double radarTurn = maxBearing + (BotMath.sign(maxBearing) * (PI/8));
		if(scannedBots < getOthers()){
			
			radarTurn = PI * radarDirection;
		}
		setTurnRadarRightRadians(radarTurn);
		radarDirection = BotMath.sign(radarTurn);
	}		
	
	
	
	public void onWin(WinEvent e)
	{
		SaveThread t = new SaveThread();
		t.start();
		for (int i = 0; i < 50; i++)
		{
			turnRight(30);
			turnLeft(30);
			turnRight(30);
			turnLeft(30);
			turnRight(30);
			turnLeft(30);
		}
		
	}

}


