package gh.mini;
import robocode.*;
import robocode.util.Utils;
import java.awt.Color;
import java.awt.geom.*;
import java.util.*;

/**
 * Griezel - the first specific meleebot of Gert Heijenk (GrubbmGait)
 *
 * Inspired by the ideas on the MeleeStrategy page and on Coriantumr
 * Created because Gruwel was threatened by
 * Loki's (mini)Freya
 * kc's   Logic
 * MAP's  Ugluk
 * deewiant's Anomaly
 *
 * Revision information:
 * v0.1  20060410 Put Griezel 0.3 into a mini
 * v0.2  20090902 Remove teamsupport, Slightly improved gun when enemy is near wall
 *                only change destination when new one is significantly better
 */
public class Griezel extends AdvancedRobot
{
	// movement variables
//	private final static int WALLAVOID = 35;	// the area to avoid near walls
	private final static int RADARTIMEOUT = 20;	// turn radar at least once in this time
	private Point2D	myPosition;					// my current position
	private	Point2D	myGoalPosition;				// my desired position, continuously changing
	private static Point2D	centerPoint;		// the centre of the battlefield
	private long	currTime;					// the current time (mainloop)

	// scanning and locking variables
	private long	lastscan;				// time of last radarscan

	// some info about the targets
	private double	oldHeading;				// the previous heading of the selected target
	private EnemyInfo currentTarget;		// the current target
	private HashMap	enemies;				// all available targets

	// some statistics
	private static double	fieldWidth;		// the width of the battlefield
	private static double	fieldHeight;	// the heigth of the battlefield
    private RoundRectangle2D.Double playField; // the playable battlefield
	private Rectangle2D.Double fireField;	// the shootable battlefield

	//global declarations:
//	static int[] finishes;

	/**
	 * run: mini.Griezel's default behavior
	 */
	public void run() {
		// Give the robot an appealing look
		setColors( Color.red, Color.orange, Color.yellow);

		// First set the statistics
		fieldWidth = getBattleFieldWidth();
		fieldHeight = getBattleFieldHeight();
        playField = new RoundRectangle2D.Double( 35, 35,
	    fieldWidth - 70, fieldHeight - 70, 50, 50);
        fireField = new Rectangle2D.Double( 17, 17, fieldWidth - 34, fieldHeight - 34);
		centerPoint = new Point2D.Double( fieldWidth / 2, fieldHeight / 2 );

//		if (finishes == null)
//			finishes = new int[getOthers()+1];

		// Let gun and radar move independently
		setAdjustGunForRobotTurn( true);
		setAdjustRadarForGunTurn( true);
		setTurnRadarRight( Double.POSITIVE_INFINITY);

		// Do not keep anything between rounds
		enemies = new HashMap();
		myPosition = new Point2D.Double();
		myGoalPosition = new Point2D.Double( getX(), getY());

		// mainloop
		while( true ) {
			currTime = getTime();
			// radar plus gunnery is done in onScannedRobot()
			// movement is done in a routine to keep the mainloop clean
			doMeleeMovement();
			if (getRadarTurnRemaining() == 0.0) {
				setTurnRadarRight( Double.POSITIVE_INFINITY);
			}
			execute();
		}
	}

	/**
	 * onScannedRobot: What to do when you see another robot
	 */
	public void onScannedRobot(ScannedRobotEvent e) {

		double absBearing;
		double dist;
		double power;
		boolean fired = false;
		EnemyInfo enemy = (EnemyInfo)enemies.get( e.getName());
		if (enemy == null)
			enemies.put( e.getName(), enemy = new EnemyInfo());

		myPosition.setLocation( getX(), getY());
		// fill in the data of this enemy
		absBearing = getHeadingRadians() + e.getBearingRadians();
		dist = e.getDistance();
		enemy.setLocation( myPosition.getX() + Math.sin( absBearing) * dist, myPosition.getY() + Math.cos( absBearing) * dist);

		// calculate 'attractivity' of this enemy and decide whether to attack or not
		enemy.attrac = (e.getEnergy() < 20 ? dist * 0.75 : dist);
		if (enemy != currentTarget) {
			if ((currentTarget == null) || ((enemy.attrac < (currentTarget.attrac * 0.8)) && (currentTarget.attrac > 80) && (dist < 1200))) {
				currentTarget = enemy;	// new more attractive current target
			}
		}
		if (enemy != currentTarget) return;		// forget about those distant loosers

		// At start of round, always perform 2.5 radarsweeps without locking, but do turn the gun already!!
		if (currTime < 10) {
			setTurnGunRightRadians( Utils.normalRelativeAngle( absBearing - getRadarHeadingRadians()));
			return;
		}

		// set the power of the bullet, including energymanagement
		power = ( dist > 850 ? 0.1 : (dist > 700 ? 0.49 : (dist > 250 ? 1.9 : 3.0)));
		power = Math.min( getEnergy()/5, Math.min( (e.getEnergy()/4) + 0.2, power));

		// perform lineair and circular targeting
		long deltahittime;
		Point2D.Double point = new Point2D.Double();
		double head, chead, bspeed;
		double tmpx, tmpy;

		// perform an iteration to find a reasonable accurate expected position
		tmpx = enemy.getX();
		tmpy = enemy.getY();
		head = e.getHeadingRadians();
		chead = head - oldHeading;
		oldHeading = head;
		point.setLocation( tmpx, tmpy);
		deltahittime = 0;
		do {
			tmpx += Math.sin( head) * e.getVelocity();
			tmpy +=	Math.cos( head) * e.getVelocity();
			head += chead;
			deltahittime++;
			// if position not in field, adapt
			if (!fireField.contains( tmpx, tmpy)) {
				bspeed = point.distance( myPosition) / deltahittime;
				power = Math.max( Math.min( (20 - bspeed) / 3.0, 3.0), 0.1);
				break;
			}
			point.setLocation( tmpx, tmpy);
		} while ( (int)Math.round( (point.distance( myPosition) - 18) / Rules.getBulletSpeed( power)) > deltahittime);
		point.setLocation( Math.min( fieldWidth  - 34, Math.max( 34, tmpx)), Math.min( fieldHeight - 34, Math.max( 34, tmpy)));
				
		// If the gun is too hot or not aligned, don't bother trying to fire.
		if ((getGunHeat() == 0.0) && (getGunTurnRemaining() == 0.0) && (power > 0.0) && (getEnergy() > 0.1)) {
			// Only fire the gun when it is locked and loaded, do a radarsweep afterwards.
			setFire( power);
			fired = true;
		}
		if ((fired == true) || (currTime >= (lastscan + RADARTIMEOUT))) {
			setTurnRadarRightRadians( Double.NEGATIVE_INFINITY * Utils.normalRelativeAngle( absBearing - getRadarHeadingRadians()));
			lastscan = currTime;
		}
		else {
			// when not firing, lock on target.
			setTurnRadarRightRadians( 2.2 * Utils.normalRelativeAngle( absBearing - getRadarHeadingRadians()));
		}
		// Turn gun after eventual firing (robocode does it also this way)
		setTurnGunRightRadians( Utils.normalRelativeAngle(((Math.PI / 2) - Math.atan2( point.y - myPosition.getY(), point.x - myPosition.getX())) - getGunHeadingRadians()));
	}

	public void onRobotDeath(RobotDeathEvent e)
	{
		// remove dead enemy and check if it was my currentTarget
		if (enemies.remove(e.getName()) == currentTarget) {
			currentTarget = null;
			setTurnRadarRight( Double.POSITIVE_INFINITY);
		}
	}

	public void doMeleeMovement()
	{
		double travel;
		double goalrisk;
		double testrisk;
		double temprisk;
		double angle;
		Point2D testPoint = new Point2D.Double();
		Point2D tempPoint = new Point2D.Double();

		myPosition.setLocation( getX(), getY());
		if (currentTarget != null)
		{
			// if enemy is close, keep moving
			travel = Math.max( 60, myPosition.distance(currentTarget) * 0.35);
			temprisk = goalrisk = calcRisk( myGoalPosition, getHeadingRadians());			// Is my Goal-path still good enough??

			for (angle = 0; angle < Math.PI*2; angle += Math.PI/36) {
				testPoint.setLocation( myPosition.getX() + Math.sin( angle) * travel, myPosition.getY() + Math.cos( angle) * travel);
				if ((playField.contains( testPoint)) && ((testrisk = calcRisk( testPoint, angle)) < temprisk )) {
					tempPoint.setLocation( testPoint);
					temprisk = testrisk;
				}
			}
			// only if best point is significantly better than old goalposition, change
			if (temprisk < (goalrisk * 0.9)) {
				myGoalPosition.setLocation( tempPoint);
				goalrisk = temprisk;
			}

			// set the turn and the distance
			angle = Utils.normalRelativeAngle( Math.atan2( myPosition.getX() - myGoalPosition.getX(), myPosition.getY() - myGoalPosition.getY()) - getHeadingRadians());
			setTurnRightRadians( Math.atan( Math.tan( angle)));
			setMaxVelocity( 10 - (4 * Math.abs(getTurnRemainingRadians())));
			setAhead( (angle == Math.atan( Math.tan( angle)) ? -1.0 : 1.0) * travel);
//			out.println(angle+"  "+myGoalPosition);
		}
	}

	private double calcRisk(Point2D point, double movang)
	{
		// copied the framework from Kawigi's Coriantumr

		double totrisk;
		double botrisk;
		double botangle;
		Collection enemySet;
		Iterator it = (enemySet = enemies.values()).iterator();

		// stay away from centre and corners
		totrisk = (8 * (getOthers() - 1)) / point.distanceSq( centerPoint);
		totrisk += 3 / point.distanceSq( 0, 0);
		totrisk += 3 / point.distanceSq( fieldWidth, 0);
		totrisk += 3 / point.distanceSq( 0, fieldHeight);
		totrisk += 3 / point.distanceSq( fieldWidth, fieldHeight);

		do
		{
			EnemyInfo e = (EnemyInfo)it.next();
			// calculate the perpendicularity (0 = head-on)
		    botangle = Utils.normalRelativeAngle( Math.atan2( e.getX() - point.getX(), e.getY() - point.getY()) - movang);
			// standard risk not energy-related
			botrisk = 100;
			// current target is probably more dangerous
			if (e == currentTarget) botrisk += 40;
			// if they have hit me recently, the threat is bigger
//			if (currTime - e.lastHit < 100) botrisk += 30;
			// add perpendicularity to the threat (a bit linear on angle seems better!)
			botrisk *= (1.0 + ((1 - (Math.abs(Math.sin(botangle)))) + Math.abs(Math.cos(botangle))) / 2);

			// TODO:
			// try to never be the closest to anyone

			// all risks from any bot is distance related
			totrisk += (botrisk / point.distanceSq( e));
		}
		while (it.hasNext());
		// and finally make a random threat on my current location
		totrisk += (Math.random() * 0.5) / point.distanceSq( myPosition);
		return totrisk;
	}

	/**
	 * onWin: Show my private victory dance
	 */
//	public void onWin(WinEvent e)
//	{
//		printStats();
//	}

	/**
	 * onDeath: Show some statistics
	 */
//	public void onDeath(DeathEvent e)
//	{
//		printStats();
//	}

	/**
	 * printStats: Print statistics on end of round
	 */
//	public void printStats( )
//	{
//		finishes[getOthers()]++;
//		for (int i=0; i<finishes.length; i++)
//			out.print(finishes[i] + " ");
//		out.println();
//	}
}

/**
 * EnemyInfo - used for keeping information on all enemies
 * 		Based upon Coriantumr (smart move of Kawigi)
 */
class EnemyInfo extends Point2D.Double
{
	double attrac;
}
