package ags.rougedc.gun;

import static robocode.util.Utils.normalAbsoluteAngle;

import java.util.List;
import java.util.Map;
import java.util.HashMap;
import ags.rougedc.base.Rules;
import ags.rougedc.gun.util.DCEntry;
import ags.rougedc.gun.util.MovementProfile;
import ags.rougedc.gun.util.Predictor;
import ags.rougedc.robots.*;
import ags.rougedc.waves.SelfWave;
import ags.utils.Range;
import ags.utils.newtree.KdTree;
import ags.utils.points.*;

/**
 * Predicts their movement profile with DC targeting
 * 
 * @author Alex Schultz
 */
public class DCPredictor implements Predictor {
    private final Rules rules;
    private final StatusRobot status;
    
    private static final int dimensions = 6;
    private static final KdTree<Range> tree = new KdTree<Range>(dimensions);
    
    private final Map<SelfWave, double[]> waves = new HashMap<SelfWave, double[]>();
    private double[] position;
    
    public DCPredictor(Rules rules, StatusRobot status) {
        this.rules = rules;
        this.status = status;
    }

    // Look back at the wave and see what happened
    public void waveHit(SelfWave wave) {
        if (!waves.containsKey(wave))
            return; // Shouldn't happen, but just in case
        
        double[] p = waves.get(wave);
        KdTree.addPoint(tree, p, wave.getHitRange());
        waves.remove(wave);
    }
    
    public void remember(EnemyRobot target) {}
    
    // Set "position" here
    public void run(AbsolutePoint nextLocation, EnemyRobot target, double bulletpower) {
        final double w1 = getWallSegment(1, nextLocation, target, bulletpower);
        final double w2 = getWallSegment(-1, nextLocation, target, bulletpower);
        position = DCEntry.getPosition(nextLocation, target, w1, w2);
    }

    // Associate the wave with the "position"
    public void waveFired(SelfWave wave) {
        waves.put(wave, position);
    }
    
    // Construct the movement profile here
    public MovementProfile getPredictedProfile() {
        final MovementProfile profile = new MovementProfile();
        final List<KdTree.Entry<Range>> entries = KdTree.nearestNeighbor(tree, position, 60);
        if (entries.size() == 0)
            return null;
        for (KdTree.Entry<Range> entry : entries) {
            final double weight = 1.0 / (1 + 100*entry.distance);
            profile.add(entry.value, weight);
        }
        return profile;
    }

    final double WALL_MARGIN = 18;
    // eDist  = the distance from you to the enemy
    // eAngle = the absolute angle from you to the enemy
    // oDir   =  1 for the clockwise orbit distance
    //          -1 for the counter-clockwise orbit distance
    // returns: the positive orbital distance (in radians) the enemy can travel
    //          before hitting a wall (possibly infinity).
    double wallDistance(double eDist, double eAngle, int oDir) {
        final double S = WALL_MARGIN;
        final double W = WALL_MARGIN;
        final double N = rules.BATTLEFIELD_HEIGHT - WALL_MARGIN;
        final double E = rules.BATTLEFIELD_WIDTH - WALL_MARGIN;
        
        return Math.min(Math.min(Math.min(
            distanceWest(N - status.getLocation().y, eDist, eAngle - Math.PI/2, oDir),
            distanceWest(E - status.getLocation().x, eDist, eAngle + Math.PI, oDir)),
            distanceWest(status.getLocation().y - S, eDist, eAngle + Math.PI/2, oDir)),
            distanceWest(status.getLocation().x - W, eDist, eAngle, oDir));
    }
    
    double distanceWest(double toWall, double eDist, double eAngle, int oDir) {
        if (eDist <= toWall) {
            return Double.POSITIVE_INFINITY;
        }
        double wallAngle = Math.acos(-oDir * toWall / eDist) + oDir * Math.PI/2;
        return normalAbsoluteAngle(oDir * (wallAngle - eAngle));
    }
    
    public double getWallSegment(int inv, AbsolutePoint source, EnemyRobot target, double power) {
        final AbsolutePoint p = target.getLocation();
        final double HOTAngle = normalAbsoluteAngle(Math.atan2(p.x-source.x, p.y-source.y));
        
        final RelativePoint v = target.getVelocity();
        final double lateralVelocity = (v.magnitude * Math.sin(v.getDirection() - HOTAngle));
        final int odir; 
        if (lateralVelocity >= 0)
            odir = 1;
        else
            odir = -1;
        
        final double angle = wallDistance(source.distance(p), HOTAngle, inv*odir);
        
        return angle/Math.asin(rules.MAX_VELOCITY / rules.getBulletSpeed(power));
    }
}
