package gh.micro;
import robocode.*;
import robocode.util.Utils;
import robocode.Rules;
import java.awt.Color;
import java.awt.geom.*;
/**
 * GrubbmThree - a robot by Gert Heijenk
 *
 * Ramming. Tracks and rams the nearest robot it sees
 *
 * Revision information:
 * v0.1  20041124 Initial version
 * v0.2  20041127 Solved sharp-turn bug, removed 10-tick wait at startup
 * v0.3  20041202 Changed power-distance ratio and made sharp-turn less sharp
 * v0.4  20041218 Use a mix between CT and HOT gun
 * v0.5  20041219 v0.3 with sharp sharp-turn and fire also when angle between 10-20 degrees
 * v0.6  20050127 improved radar, improved rambonus and removed sharp-turn, it slows down the bot too much
 * v0.7  20050130 improved and refactored CT-gun
 * v0.8  20050316 shoot at the edge of the robot, not at the middle
 * v0.9  20060216 shrinking, position aiming-point always in field, less energy-preservation
 * v0.9x 20170925 wobble in movement, take speedchange into account, fire full power when close and close to wall
 */
public class GrubbmThree extends AdvancedRobot
{
	static final double WALL_MARGIN  = 17;
	static final double DWALL_MARGIN = 34;

	// scanning and locking variables
	static double	oldHeading;		// the previous heading of the selected opponent
	static double	oldSpeed;		// same for speed
	static double	fieldWidth;
	static double	fieldHeight;
	static Rectangle2D.Double fireField;	// the shootable battlefield
	static double	wobble = 0.2;

	static Point2D.Double	myNextPos = new Point2D.Double();	// my next position

	/**
	 * run: Grubbm_Three's default behavior
	 */
	public void run() {

		// Give the robot an appealing look
		setColors( Color.red, Color.yellow, Color.red);

		// First set the statistics
		fieldWidth = getBattleFieldWidth();
		fieldHeight = getBattleFieldHeight();
		fireField = new Rectangle2D.Double( WALL_MARGIN, WALL_MARGIN, fieldWidth - DWALL_MARGIN, fieldHeight - DWALL_MARGIN);

		// Let gun and radar move independently
		setAdjustGunForRobotTurn( true);
		setAdjustRadarForGunTurn( true);

		// mainloop
		do {
			turnRadarRight( Double.NEGATIVE_INFINITY);
		} while(true);
	}

	/**
	 * onScannedRobot: What to do when you see another robot
	 */
	public void onScannedRobot(ScannedRobotEvent e) {

		double power;		// the power of the bullet
		double bearingFromGun;
		double turnAngle;
		double tmpbearing = getHeadingRadians() + e.getBearingRadians();

		// Lock target with radar
		setTurnRadarRightRadians( 2.2 * Utils.normalRelativeAngle( tmpbearing - getRadarHeadingRadians()));

		// set the power of the bullet, 
		power = Math.min( getEnergy() - 0.1, (e.getDistance() > 200 ? 1.9 : 3.0));

		// perform iterative circular targeting
		Point2D.Double point = new Point2D.Double();
		long deltahittime;
		double head, chead;
		double speed, cspeed;
		double tmpx, tmpy;

		// Estimate next position
		myNextPos.setLocation( getX() + Math.sin( getHeadingRadians()) * getVelocity(), getY() + Math.cos( getHeadingRadians()) * getVelocity());

		// perform an iteration to find a reasonable accurate expected position
		tmpx = getX() + Math.sin( tmpbearing) * e.getDistance();
		tmpy = getY() + Math.cos( tmpbearing) * e.getDistance();
		head = e.getHeadingRadians();
		chead = head - oldHeading;
		oldHeading = head;
		speed = e.getVelocity();
		cspeed = Math.min(1.0, Math.max(speed - oldSpeed, -1.0)); // bit rough, but close enough
		oldSpeed = speed;
//		point.setLocation( tmpx, tmpy);
		deltahittime = -1;				// necessary to shoot to enemies NEXT position
		do {
			tmpx += Math.sin( head) * speed;
			tmpy +=	Math.cos( head) * speed;
			head += chead;
			speed = Math.min(8.0, Math.max(speed + cspeed, -8.0)); 
			deltahittime++;
			// if position not in field, adapt position
			if (!fireField.contains( tmpx, tmpy)) {
				point.setLocation( warpPoint( tmpx, tmpy));
				// only adapt power/speed to intercept, if not really close
				if (deltahittime > 9) {
					power = (20.0 - (point.distance( myNextPos) - 18) / deltahittime) / 3.0;
				}
//				else { System.out.println(getTime() + "  Skipping adaption " + deltahittime); }
				break;
			}
			point.setLocation( tmpx, tmpy);
		} while ( (int)Math.round( (point.distance( myNextPos) - 18) / Rules.getBulletSpeed(power)) > deltahittime);

		point.setLocation( warpPoint( tmpx, tmpy));
		// Turn gun
		tmpbearing = ((Math.PI / 2) - Math.atan2( point.y - myNextPos.getY(), point.x - myNextPos.getX()));
		setTurnGunRightRadians( bearingFromGun = Utils.normalRelativeAngle( tmpbearing - getGunHeadingRadians()));
		// If the gun is too hot or not aligned, don't bother trying to fire.
		if ((e.getDistance() <= 350) && (power > 0.0) && (getGunHeat() == 0.0) && (Math.abs( bearingFromGun) < 0.35))
		{
			// Only fire the gun when less than 20 degrees to turn.
			setFire( power);
		}
		turnAngle = Utils.normalRelativeAngle( tmpbearing - getHeadingRadians());
		// Move towards target with a wobble when turn is not too sharp
		if ((getTime() % 6) == 0 && Math.abs(getTurnRemainingRadians()) < 2*Rules.getTurnRateRadians( getVelocity())) {
//			System.out.println("Time: " + getTime() + "  Skip wobble : " + getTurnRemainingRadians());
			wobble = -wobble;
		}
		setTurnRightRadians( Math.atan( Math.tan( Utils.normalRelativeAngle( turnAngle += wobble))));
		// sharp turn when necessary
//		setMaxVelocity( Math.abs(myTurnAngle) > 0.785 ? 4.0 : 8.0); // 0.785 -> 45 degrees
		setAhead( Double.POSITIVE_INFINITY * Math.cos( turnAngle));
	}

	/**
	 * warpPoint: Put point into field with a margin of 2 * WALL_MARGIN (34)
	 */
	public Point2D.Double warpPoint( double x, double y)
	{
		Point2D.Double pos = new Point2D.Double();

		pos.x = Math.min( fieldWidth  - DWALL_MARGIN, Math.max( DWALL_MARGIN, x));
		pos.y = Math.min( fieldHeight - DWALL_MARGIN, Math.max( DWALL_MARGIN, y));
		return pos;
	}

	/**
	 * onWin: Show my private victory dance
	 */
	public void onWin(WinEvent e)
	{
		//Victory dance	
		setTurnGunRight( Double.POSITIVE_INFINITY);
		setTurnRadarLeft( Double.POSITIVE_INFINITY);
		setTurnRight(10);
		ahead(10);
		waitFor(new RadarTurnCompleteCondition(this));
	}
}
