package tjk.utils;

import tjk.universe.Wave;

/**
 * HitTracker - a class by tkiesel
 * Copyright (c) 2012 Tom Kiesel (Tkiesel @ Robowiki)
 * 
 * This software is provided 'as-is', without any express or implied warranty. In no event will the authors be held liable for any damages arising from the use of this software.
 * 
 * Permission is granted to anyone to use this software for any purpose, including commercial applications, and to alter it and redistribute it freely, subject to the following restrictions:
 * 
 *     1. The origin of this software must not be misrepresented; you must not claim that you wrote the original software. 
 *        If you use this software in a product, an acknowledgment in the product documentation would be appreciated but is not required.
 * 
 *     2. Altered source versions must be plainly marked as such, and must not be misrepresented as being the original software.
 * 
 *     3. This notice may not be removed or altered from any source distribution.
 * 
 */
 
public class HitTracker
{
	
	// These two track overall hitrate.
	private double enemyHits = 0;	
	// A bit hacky, but prevents /0
	private double enemyMisses = 0.00000001;
	
	// This tracks the rolling average hitrate.
	private double rollingHitRateShort = 0.05;
	private double rollingHitRateMedium = 0.05;
	private double rollingHitRateLong = 0.05;
	
	// Rolling depth of the buffer.
	private static final double ROLLING_DEPTH_SHORT = 20.0;
	private static final double ROLLING_DEPTH_MEDIUM = 40.0;
	private static final double ROLLING_DEPTH_LONG = 90.0;
	
	// Bullets seen so far.
	private double bulletsSeen = 0.0000000000001;
	
	public double totalHitRate()
	{
		return enemyHits / (enemyHits + enemyMisses);
	}
	
	public double rollingHitRateShort()
	{
		return rollingHitRateShort;
	}

	public double rollingHitRateMedium()
	{
		return rollingHitRateMedium;
	}

	public double rollingHitRateLong()
	{
		return rollingHitRateLong;
	}
	
	public void addMiss()
	{
		enemyMisses += 1.0;
		logHit(0.0);
	}
	
	public void addHit(double hit)
	{
		if ( Double.compare(hit, 0.0) <= 0.0 )
		{
			addMiss();
			return;
		}
		// Total rate.
		enemyHits += hit;
		
		// Rolling rates.
		logHit(hit);
	}

	public void addHit(Wave w)
	{
		addHit(normalizeHit(w));
	}
	
	public static double normalizeHit(Wave w)
	{
		double[] mcm = w.getMCM();
		return w.getRadius()/400.0*(mcm[0]+mcm[2])/(2.0*FastTrig.asin(8.0/w.getVelocity()));
	}
	
	private void logHit(double hitSize)
	{
		bulletsSeen += 1.0;
		logHitShort(hitSize);
		logHitMedium(hitSize);
		logHitLong(hitSize);
	}

	private void logHitShort(double hitSize)
	{
		double roll_depth = Math.min(bulletsSeen,ROLLING_DEPTH_SHORT);
		rollingHitRateShort = ( hitSize + rollingHitRateShort*(roll_depth-1.0) ) / roll_depth;
	}
	
	private void logHitMedium(double hitSize)
	{
		double roll_depth = Math.min(bulletsSeen,ROLLING_DEPTH_MEDIUM);
		rollingHitRateMedium = ( hitSize + rollingHitRateMedium*(roll_depth-1.0) ) / roll_depth;
	}
	
	private void logHitLong(double hitSize)
	{
		double roll_depth = Math.min(bulletsSeen,ROLLING_DEPTH_LONG);
		rollingHitRateLong = ( hitSize + rollingHitRateLong*(roll_depth-1.0) ) / roll_depth;
	}
	
}
												