package can;
import java.awt.Color;
//import java.awt.Graphics2D;
import java.awt.geom.Point2D;
import java.awt.geom.RoundRectangle2D;
import java.util.ArrayList;
import java.util.Collections;

import robocode.*;
import robocode.util.*;

/**
 * @author Chad Nunemaker
 */
public class Pookie extends AdvancedRobot
{
	/**
	 * The amount of energy the enemy had.
	 */
	static double prevEnergy;
	
	/**
	 * The amount to move when a shot is detected.
	 */
	static final double MOVEAMOUNT = 37.0;
	
	/**
	 * The amount and direction to move when a shot is detected.
	 */
	static double direction = MOVEAMOUNT;
	
	/**
	 * The number of times killed.
	 */
	static int numOfDeaths = 0;
	
	/**
	 * The current movement tactic.
	 * 1: Stop-N-Go
	 * 2: Random Movement w/ Wall Avoidance
	 */
	static int movementTactic = 1;

	/**
	 * The amount of energy to use in a shot.
	 */
	static final double FIREPOWER = 2.5;
	
	/**
	 * The bullet velocity.
	 */
	static final double BULLETVEL = 20 - 3 * FIREPOWER;
	
	/**
	 * The maximum angle offset a robot can move before the bullet would reach them.
	 */
	static final double MAXESCAPE = Math.asin(8.0 / BULLETVEL);
	
	/**
	 * The distance in pixels that is considered close to a robot.
	 */
	static final double CLOSE = 36;
	
	/**
	 * A list of all waves in motion for the virtual gun system.
	 */
	static ArrayList<Wave> waves;
	
	/**
	 * The number of hits for each of the different targeting methods using virtual bullets.
	 * The hits are segmented by distance of 100px.
	 */
	static ArrayList<ArrayList<Integer>> hits = new ArrayList<ArrayList<Integer>>();
	
	/**
	 * The number of targeting methods used in the virtual gun system.
	 */
	static final int NUMTARGETMETHODS = 4;
	
	/**
	 * The length of the pattern we are trying to match.
	 */
	static final int PATTERN_LENGTH = 30;
	
	/**
	 * A log of the enemy's lateral velocities.
	 */
	static String eLog = "000000000000000000000000000000888888888888888765432100888765432101234567888765432100";

	/**
	 * The distance to stay away from the walls.
	 */
	static final int WALL_MARGIN = 25;
	
	/**
	 * The lateral angle to move.
	 */
	static double movementLateralAngle = 0.2;
	
	/**
	 * The index of the best targeting method.
	 */
	static int bestTargetingNum;

	/**
	 * The number of waves fired so far at the given segments.
	 */
	private ArrayList<Integer> numOfWaves = new ArrayList<Integer>();
	
	/**
	 * Run the robot.
	 */
	@Override
	public void run()
    {
		// Set body color to dark grey, turret to black, and radar to white.
		setColors(Color.darkGray, Color.black, Color.white);
		
		// Initialize the number of hits to zero for each segment.
		for (int i = 0; i < NUMTARGETMETHODS; i++)
		{
			ArrayList<Integer> arr = new ArrayList<Integer>(13);
			for (int j = 0; j < 13; j++)
			{
				arr.add(0);
			}
			hits.add(arr);
		}
		
		// Initialize the number of waves in each segment to zero.
		for (int i = 0; i < 13; i++)
		{
			numOfWaves.add(0);
		}
		
		// Reset the waves between battles.
		waves = new ArrayList<Wave>();
		
		// Allow the gun and radar to float during turns.
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		
		// Search for a radar lock.
		// If one is maintained, then continue scanning the enemy.
		while (true)
		{
			if (getRadarTurnRemaining() == 0)
			{
				setTurnRadarRight(Double.POSITIVE_INFINITY);
			}
			scan();
		}
    }
	
	/**
	 * Do when an enemy robot is found and scanned.
	 */
	@Override
	public void onScannedRobot(ScannedRobotEvent e)
    {
		doMovement(e);
        doTarget(e);
        doFire(e);
        doLock(e, 1.9);
    }

	/**
	 * The movement tactics to use.
	 * @param e the scanned robot
	 */
	public void doMovement(ScannedRobotEvent e)
	{
		// The wall-free battlefield.
		final RoundRectangle2D fieldRectangle = new RoundRectangle2D.Double(WALL_MARGIN,
				WALL_MARGIN, getBattleFieldWidth() - WALL_MARGIN * 2,
			    getBattleFieldHeight() - WALL_MARGIN * 2, 75, 75);
		
		// Our location.
		final Point2D robotLocation = new Point2D.Double(getX(), getY());
		
		// The enemy's location.
		final double enemyDistance = e.getDistance();
		final double absoluteBearing = getHeadingRadians() + e.getBearingRadians();
		final double enemyX = e.getDistance() * Math.sin(absoluteBearing) + robotLocation.getX();
		final double enemyY = e.getDistance() * Math.cos(absoluteBearing) + robotLocation.getY();
		final Point2D enemyLocation = new Point2D.Double(enemyX, enemyY);
		
		if (movementTactic == 1)
		{
			// Use Stop-N-Go movement.	
			if (prevEnergy > e.getEnergy())
			{
				setAhead(direction);
			}
			setTurnRightRadians(Math.cos(e.getBearingRadians()));
		} else
		{
			// Random movement w/ iterative wall smoothing.
			// Consider changing direction w/
			// flattener factor of 0.05.
			if (Math.random() < 0.05)
			{
			    movementLateralAngle = -movementLateralAngle;
			}

			// Pick a new point in our escape area.
			Point2D robotDestination;
			int tries = 0;
			do
			{
				final double moveAngle = getAbsoluteBearing(enemyLocation, robotLocation) + movementLateralAngle;
				final double moveDistance = enemyDistance * (1.1 - tries / 100.0);
				robotDestination = new Point2D.Double(moveDistance * Math.sin(moveAngle) + enemyLocation.getX(),
						moveDistance * Math.cos(moveAngle) + enemyLocation.getY());
			} while (++tries < 100 && !fieldRectangle.contains(robotDestination));
			
			// Goto the new robot destination.
	        double angle = Utils.normalRelativeAngle(
	        		getAbsoluteBearing(robotLocation, robotDestination) - getHeadingRadians());
	    	double turnAngle = Math.atan(Math.tan(angle));
	        setTurnRightRadians(Math.atan(Math.tan(angle)));
	        if (angle == turnAngle)
	        {
	        	setAhead(robotLocation.distance(robotDestination));
	        } else
	        {
	        	setAhead(-robotLocation.distance(robotDestination));
	        }
	        
	    	// Hit the brake hard if we need to turn sharply
	        if (Math.abs(getTurnRemaining()) > 33)
	        {
	        	setMaxVelocity(0);
	        } else
	        {
	        	setMaxVelocity(Rules.MAX_VELOCITY);
	        }
		}
		prevEnergy = e.getEnergy();
	}

    static double getAbsoluteBearing(Point2D source, Point2D target)
	{
        return Math.atan2(target.getX() - source.getX(), target.getY() - source.getY());
    }
	
	/**
	 * The targeting tactics to use.
	 * @param e the scanned robot
	 */
	public void doTarget(ScannedRobotEvent e)
	{
		// A list of all gun's bearings.
		ArrayList<Double> bearings = new ArrayList<Double>();
		
        // Calculate the Head-On Targeting bearing.
		final double absoluteBearing = getHeadingRadians() + e.getBearingRadians();
		final double headOn = absoluteBearing;
		bearings.add(headOn);
		
		// Calculate the Non-iterative Linear Targeting bearing.
		final double lateral = e.getVelocity() * Math.sin(e.getHeadingRadians() - absoluteBearing);
		final double angleOffset = Math.asin(lateral / BULLETVEL);
		final double linear = angleOffset + absoluteBearing;
		bearings.add(linear);
			
		// Calculate the Random Targeting bearing.
		double guessFactor = 2 * (Math.random() - 0.5);
		if (e.getEnergy() == 0)
		{
			guessFactor = 0;
		}
		final double random = guessFactor * MAXESCAPE + absoluteBearing;
		bearings.add(random);
		
		// Calculate Symbolic Pattern Matching Targeting bearing.
		double absB = absoluteBearing;
		
		// Insert the newest lateral velocity into the beginning of the log.
		eLog = String.valueOf((char) Math.round(e.getVelocity() *
				Math.sin(e.getHeadingRadians() - absB))).concat(eLog);
		
		// Reduce the pattern depth until a match is found storing the index of the matching pattern.
		int i, index;
		int mLen = PATTERN_LENGTH;
		do
		{
			i = (int) (e.getDistance() / BULLETVEL);
			String substring = eLog.substring(0, mLen--);
			index = eLog.indexOf(substring, i);
		} while (index < 0);
		
		// Convert the enemy's lateral velocities in the estimated pattern to angular velocities,
		// then add them to the absolute bearing.
		do
		{
			absB += Math.asin(((byte) eLog.charAt(index--)) / e.getDistance());
		}
		while(--i > 0);

		final double pattern = absB;
		bearings.add(absB);

		// Our robot's coordinates.
		double robotX = getX();
		double robotY = getY();
		
		// The enemy robot's coordinates.
		Point2D enemyLocation = new Point2D.Double(
				e.getDistance() * Math.sin(absoluteBearing) + robotX,
				e.getDistance() * Math.cos(absoluteBearing) + robotY);
		
		// The enemy's segmented distance.
		final int segDist = (int) (e.getDistance() / 100);
		
		// Paint a target on the enemy robot.
		/*Graphics2D g = getGraphics();
		g.setColor(Color.black);
		g.drawOval((int) (enemyX - CLOSE / 2), (int) (enemyY - CLOSE / 2), (int) CLOSE, (int) CLOSE);*/
		
		// Paint all the virtual bullets.
		for (int j = 0; j >= 0 && j < waves.size(); j++)
		{
			// The wave in question.
			Wave w = waves.get(j);
			
			// The origin of the wave.
			Point2D oldRobotLocation = w.getOrigin();
			
			// The distance the wave has traveled.
			double waveDistance = BULLETVEL * (getTime() - w.getTimeFired());
			
			int b = 0;
			for (Double bearing : w.getBearings())
			{
				// The virtual bullet's coordinates.
				Point2D bulletLocation = new Point2D.Double(
						oldRobotLocation.getX() + Math.sin(bearing) * waveDistance,
						oldRobotLocation.getY()	+ Math.cos(bearing) * waveDistance);
				
				// Set the color of the bullet.
				// Head-On targeting is orange.
				// Non-iterative Linear targeting is yellow.
				// Random targeting is red.
				// Pattern Matching is pink.
				/*switch(b)
				{
				case 0:
				{
					g.setColor(Color.orange);
					break;
				}
				case 1:
				{
					g.setColor(Color.yellow);
					break;
				}
				case 2:
				{
					g.setColor(Color.red);
					break;
				}
				default: // Case 3:
				{
					g.setColor(Color.pink);
				}
				}
				
				// Draw the bullet.
				g.fillOval((int) bulletX, (int) bulletY, 4, 4);*/
				
				// Check to see if a wave has broken.
				if (waveDistance >= oldRobotLocation.distance(enemyLocation)) // A wave has broken.
				{
					// Is the bullet close to the enemy?
					if (bulletLocation.distance(enemyLocation) <= CLOSE) // The virtual bullet is close.
					{
						// Increment the close bearing's hit counter.
						ArrayList<Integer> segBearings = hits.get(b);
						segBearings.set(segDist, segBearings.get(segDist) + 1);
					}
					waves.remove(j); // Remove the broken wave.
				}
				b++;
			}
		}
		
		// Create a wave.
		waves.add(new Wave(new Point2D.Double(robotX, robotY), getTime(), bearings));
		numOfWaves.set(segDist, numOfWaves.get(segDist) + 1);
		
		// Display the hit counts for the different targeting methods.
		/*out.println("HO: " + hits.get(0) + "\nNL: " + hits.get(1) +
						"\nRT: " + hits.get(2) + "\nPM: " + hits.get(3));
		out.println("W#: " + numOfWaves.toString());*/
		
		// Gather all hit values at the current segment.
		ArrayList<Integer> segBearings = new ArrayList<Integer>(NUMTARGETMETHODS);
		for (int k = 0; k < NUMTARGETMETHODS; k++)
		{
			segBearings.add(hits.get(k).get(segDist));
		}
		
		// Choose the targeting method with the highest hit count.
		bestTargetingNum = segBearings.indexOf(Collections.max(segBearings));
		
		double bestBearing;
		switch (bestTargetingNum)
		{
		case 0: // Head-On Targeting is best.
		{
			bestBearing = headOn;
			//out.println("Using Head-On.");
			break;
		}
		case 1: // Non-iterative Targeting is best.
		{
			bestBearing = linear;
			//out.println("Using Linear.");
			break;
		}
		case 2: // Random Targeting is best.
		{
			bestBearing = random;
			//out.println("Using Random.");
			break;
		}
		default: // Case 3:
		{
			bestBearing = pattern;
			//out.println("Using Pattern.");
		}
		}

		// Turn the gun to the best bearing.
    	setTurnGunRightRadians(Utils.normalRelativeAngle(bestBearing  - getGunHeadingRadians()));
	}
	
	/**
	 * Fire the cannon.
	 * @param e 
	 */
	public void doFire(ScannedRobotEvent e)
	{
    	// Do not fire unless able to fire.
    	// Also, do not shoot oneself disabled.
		// Shoot maximum power bullets if enemy is within 100px. Good against rammers.
    	if(getGunHeat() == 0 && getEnergy() > FIREPOWER)
    	{
        	fire(Rules.MAX_BULLET_POWER - Math.min(e.getDistance() / 100, (3 - FIREPOWER)));
    	}
	}
	
	/**
	 * Perform a narrow radar lock on the enemy.
	 * @param e the scanned robot
	 * @param factor the sweep factor to use
	 */
	public void doLock(ScannedRobotEvent e, double factor)
	{
		setTurnRadarRightRadians(factor
				* Utils.normalRelativeAngle(getHeadingRadians() + e.getBearingRadians() - getRadarHeadingRadians()));
	}
	
	/**
	 * Reverse direction when a wall is hit.
	 */
	@Override
	public void onHitWall(HitWallEvent e)
	{
	    direction = -direction;
	}
	
	/**
	 * Note when an enemy losses energy from a shot.
	 */
	@Override
	public void onBulletHit(BulletHitEvent e)
	{
		prevEnergy -= Rules.getBulletDamage(FIREPOWER);
	}
	
	/**
	 * Note when an enemy gains energy from a successful hit.
	 */
	@Override
	public void onHitByBullet(HitByBulletEvent e)
	{
		prevEnergy += Rules.getBulletHitBonus(e.getPower());
	}
	
	/**
	 * On death increment the death counter.
	 * If the number of deaths is greater than or equal to 2,
	 * then change movement tactics.
	 */
	@Override
	public void onDeath(DeathEvent e)
	{
		if (++numOfDeaths >= 2)
		{
			++movementTactic;
		}
	}

	/**
	 * A mechanic used to gather bearing offsets.
	 * @author Chad Nunemaker
	 */
	public class Wave
	{
		/**
		 * The origin location of the wave.
		 * The location a shot was fired from.
		 * In (X, Y) coordinates.
		 */
		private final Point2D.Double origin;

		/**
		 * The time the wave was created.
		 * The time the shot was fired.
		 * In turns.
		 */
		private final long timeFired;

		/**
		 * The direction the shots were sent towards using different targeting methods.
		 * In radians.
		 */
		private final ArrayList<Double> bearings;

		/**
		 * Create a wave.
		 * @param origin The origin location of the wave. The location a shot was fired from. In (X, Y) coordinates.
		 * @param velocity The velocity of the wave/shot. In pixels/second.
		 * @param startTime The time the wave was created. The time the shot was fired. In turns.
		 * @param bearings The direction the shots were sent towards using different targeting methods. In radians.
		 */
		public Wave(Point2D.Double origin, long startTime, ArrayList<Double> bearings)
		{
			this.origin = origin;
			this.timeFired = startTime;
			this.bearings = bearings;
		}

		/**
		 * @return the origin
		 */
		public Point2D.Double getOrigin()
		{
			return origin;
		}

		/**
		 * @return the timeFired
		 */
		public long getTimeFired()
		{
			return timeFired;
		}

		/**
		 * @return the bearings
		 */
		public ArrayList<Double> getBearings()
		{
			return bearings;
		}
	}

}