/**
 * abstract class Operator, that all the robot operator should inherit.
 *
 * it receive a instance of AdvancedRobot and all the robot event,so it 
 * can control the robot by call the function supplied by AdvancedRobot 
 * and get battle info from robot event.
 *
 * the main thread will call the funtion work(), so you can control the 
 * robot in this function, for example implement it like this:
 *    public void  work(){
 *          // init
 *          while(true){
 *             // control sentence
 *             robot.XXX(); 
 *          }    
 *    }
 * you can never return from this function.
 *
 * this class contain a PathManager that manage the original path info shared
 * by all operotor, all operator can get the path info( old and current ) 
 * from it. if you overlay those function: onScannedRobot(), onBulletHit(),
 * onHitByBullet(), you must call its super version, and if your robot 
 * operator know when enemy fire,you should call function onEnemyFire(),
 * and when your robot fire a bullet, you should call function onFire(), so 
 * that the pathManager can update the path info.
 *
 * @see robocode.AdvancedRobot
 * @see path.PathManager
 */


package cx;
import cx.path.*;
import robocode.*;


public abstract class Operator
{
	protected AdvancedRobot robot;
	private PathManager pathManager;
    private String name = null;
    
	//------------------------------------------------------
	public Operator( AdvancedRobot robot ){
		this.robot = robot;
		pathManager = new PathManager( robot );
    }

    public PathManager getPathManager(){
		return pathManager;
	}

    /**
	 * set the operator's name.
	 */
    public void setName(String name){
		this.name = name;
	}
	/**
	 * return the operator's name.
	 */
	public String getName(){
		return name;
	}

	// -------------------- interface ------------------------
	/**
	 * perform the robot in this method
	 */
	public abstract void work();
	
	// ----------------------------- event handle -----------------------------
	// it just like a event, must call pathManager
	public void onFire( double power ){ pathManager.onFire( power ); }
	public void onEnemyFire( String name, double power ){ pathManager.onEnemyFire(name, power); }
	// system event, must call pathManager
	public void onBulletHit(BulletHitEvent e){ pathManager.onBulletHit(e); }
	public void onHitByBullet(HitByBulletEvent e){ pathManager.onHitByBullet(e); }
	public void onScannedRobot( ScannedRobotEvent e ){ pathManager.onScannedRobot(e); }

    // system event
	public void onFinish(){}
	public void onDeath(DeathEvent e){}
	public void onWin(WinEvent e){}
	public void onRobotDeath( RobotDeathEvent e ){}
	public void onHitWall(HitWallEvent e){}
	public void onHitRobot(HitRobotEvent e){}
	public void onBulletMissed(BulletMissedEvent e){}
	public void onBulletHitBullet(BulletHitBulletEvent e){}
	// test if computing too much
	public void onSkippedTurn(SkippedTurnEvent e){}
	// Custom Event
	public void onMoveComplete( CustomEvent e ){}
	public void onTurnComplete( CustomEvent e ){}
    public void onGunTurnComplete( CustomEvent e ){}			
    public void onRadarTurnComplete( CustomEvent e ){}
    public void onCustomEvent( CustomEvent event ){}
		

} // class Operator