package ag.movement;

import java.awt.BasicStroke;
import java.awt.Graphics2D;
import java.awt.Color;
import java.io.File;
import java.util.Iterator;
import java.util.Vector;

import robocode.Bullet;
import robocode.BulletHitEvent;
import robocode.DeathEvent;
import robocode.HitByBulletEvent;
import robocode.HitRobotEvent;
import robocode.BulletHitBulletEvent;
import ag.team.Team_BulletData;
import ag.Gir;
import ag.battledata.DataRobot;
import ag.battledata.DataVector;
import ag.battledata.DataRobotSnapshot;
import ag.battledata.VirtualBullet;
import ag.neuralgir.NeuralNet;
import ag.neuralgir.NeuralNetData;
import ag.neuralgir.TrainDataSet;


/**
 * Handles the robot movement
 * @author Andree Grosse
 */
public class Movement
{
	private Gir mGir;
	
	private MoveTeam mMoveTeam;
	private Vector<GravObject> mGravWallsAndCorners;
	private GravPoint mGirGravPoint;
	private int mDodgeInputSize, mRobotsStatsUseRounds;

	private double mMoveFactors[];
	
	//	-- painting --
	private Color mColorVB, mColorGravBullets, mColorGravTeam,
				mColorGravWalls, mColorDesired, mColorGravGir;
	private BasicStroke mStrokeForces;
	private Vector<DataVector> mGravityForces;
	

	private Vector<DataVector> mLinearTargeting;
	private Vector<DataVector> mCircularTargeting;
	
	
	/**
	 * Constructor
	 * @param gir The Gir robot
	 */
	public Movement(Gir gir) {
		mGir = gir;

		mMoveTeam = new MoveTeam(mGir);
		mGravWallsAndCorners = new Vector<GravObject>();
		mDodgeInputSize = 4;
		
		mRobotsStatsUseRounds = 4;
		
		mMoveFactors = new double[]{.0, .0};
		
		mGravityForces = new Vector<DataVector>();
		mLinearTargeting = new Vector<DataVector>();
		mCircularTargeting = new Vector<DataVector>();
		
		mGirGravPoint = new GravPoint(new DataVector(),	0, 5, 300);
		
		double wallForce = 120;
		double wallExp = 6;
		double wallMax = 250;
		// walls
		mGravWallsAndCorners.add(new GravLine(new DataVector(0, 
													mGir.getBattleFieldHeight() * 0.5), 
									wallForce, wallExp, wallMax, new DataVector(1, 0)));
		mGravWallsAndCorners.add(new GravLine(new DataVector(mGir.getBattleFieldWidth(), mGir.getBattleFieldHeight() * 0.5), 
									wallForce, wallExp, wallMax, new DataVector(-1, 0)));
		mGravWallsAndCorners.add(new GravLine(new DataVector(mGir.getBattleFieldWidth() * 0.5, 0), 
									wallForce, wallExp, wallMax, new DataVector(0, 1)));
		mGravWallsAndCorners.add(new GravLine(new DataVector(mGir.getBattleFieldWidth() * 0.5, mGir.getBattleFieldHeight()), 
									wallForce, wallExp, wallMax, new DataVector(0, -1)));
		// corners
		mGravWallsAndCorners.add(new GravPoint(new DataVector(0, 0), 
				wallForce, wallExp, wallMax));
		mGravWallsAndCorners.add(new GravPoint(new DataVector(0, mGir.getBattleFieldHeight()), 
				wallForce, wallExp, wallMax));
		mGravWallsAndCorners.add(new GravPoint(new DataVector(mGir.getBattleFieldWidth(), 0), 
				wallForce, wallExp, wallMax));
		mGravWallsAndCorners.add(new GravPoint(new DataVector(mGir.getBattleFieldWidth(), mGir.getBattleFieldHeight()), 
				wallForce, wallExp, wallMax));
		
		mStrokeForces = new BasicStroke(1.5f);
		mColorVB = Color.RED;
		mColorGravBullets = Color.YELLOW;
		mColorGravWalls = Color.GREEN;
		mColorGravTeam = Color.CYAN;
		mColorGravGir = new Color(255, 128, 0);
		mColorDesired = new Color(255, 64, 0);
	}
	
	/**
	 * Main method - should be called every frame.
	 */
	public void run() {		
		//System.out.println("--" + mGir.getTime());

		mGravityForces.removeAllElements();

		if(mMoveTeam.hasTeamTarget()){
			DataVector[] teamgrav = mMoveTeam.getTeamGravForces(); 
			mGravityForces.add(teamgrav[0]);
			mGravityForces.add(teamgrav[1]);
			mGravityForces.add(teamgrav[2]);
		}
		
		if(mGir.getNumTeammates() == 0){
			// Check for Energy drops between currenttick and 
			// currenttick - 1 for all robots
			checkEnergyDrops();
		}
		else{
			// Check for Energy drops between currenttick - 1 and 
			// currenttick - 2 for all robots
			checkEnergyDropsTeam();
		}

		Iterator<GravObject> iwalls = mGravWallsAndCorners.iterator();
		GravObject grav = null;
		DataVector desired = new DataVector(.0, .0);
		DataVector girpos = mGir.getData().getPosition(0);

		// add all wall gravity forces
		while (iwalls.hasNext()){
			grav = iwalls.next();
			mGravityForces.add(grav.getForceOn(girpos));
		}	

		// add all enemies gravity forces (including their bullets) 
		Iterator<DataRobot> irob = mGir.getDataBattle().getRobots().iterator();
		Iterator<VirtualBullet> ivb;
		DataRobot robot = null;		
		VirtualBullet bullet = null;
				
		while(irob.hasNext()){
			robot = irob.next();
			if(robot.getEnemy()){
				grav = robot.getGravPoint();
				mGravityForces.add(grav.getForceOn(girpos));
			}
			// move and add the bullets -->
			ivb = robot.getVirtualBullets().iterator();
			while(ivb.hasNext()){
	  			bullet = ivb.next();
	  			bullet.move();
	  			if((bullet.getDistance() > 
	  			girpos.minus(bullet.getOriginalPosition()).length() + 40))
	  				ivb.remove();
	  			else{
	  				mGravityForces.add(bullet.getForceOn(girpos));
	  			}
	  		}	
			// <--	
		}

		// process Gir GravPoint
		processGirGravityPoint(girpos);
		
		mGravityForces.add(mGirGravPoint.getForceOn(girpos));
			
		
		//add up all gravity forces
		Iterator<DataVector> ivec = mGravityForces.iterator();
		while(ivec.hasNext()){
			desired = desired.plus(ivec.next());
		}		
		
		// calc momentum
		desired.normalize();

		mGravityForces.add(desired);
		moveVector(desired);
			
	}	
	
	// ####################### setter & getter #######################

	/**
	 * Returns the MoveTeam object
	 * @return the MoveTeam object
	 */
	public MoveTeam getMoveTeam(){
		return mMoveTeam;
	}
	
	// ######################### reaction #################
	
	/**
	 * This method should be called when Gir hits another robot
	 */
	public void onBulletHit(BulletHitEvent e) {
		//System.out.println("Bullet hit on " + mGir.getTime());
		DataRobot robot = mGir.getDataBattle().getRobotByName(e.getName());
							
		robot.adjustEnergyChange(0, -bulletDamage(e.getBullet().getPower()));
		
	}
	
	/**
	 * This method should be called when Gir is hit by a bullet
	 */	
	public void onHitByBullet(HitByBulletEvent e) {
		//System.out.println("hit bb on " + mGir.getTime());
		DataRobot robot = mGir.getDataBattle().getRobotByName(e.getName());
				
		if(robot.getEnemy()){
			
			robot.adjustEnergyChange(0, bulletEnergyGain(e.getBullet().getPower()));	
			Bullet b = e.getBullet();
			
			hitByBullet(robot, b.getX(), b.getY(), b.getHeadingRadians(), b.getPower());
		}
	}
	
	/**
	 * This method should be called when one of Girs
	 * bullet hits another bullet
	 */	
	public void onBulletHitBullet(BulletHitBulletEvent e){
		DataRobot robot = mGir.getDataBattle().getRobotByName(e.getHitBullet().getName());
				
		if(robot.getEnemy()){
			
			Bullet b = e.getHitBullet();	
			
			hitByBullet(robot, b.getX(), b.getY(), b.getHeadingRadians(), b.getPower());
		}
	}
	
	/**
	 * This method should be called when Gir dies 
	 */	
	public void onDeath(DeathEvent e){
		Iterator<DataRobot> irob = mGir.getDataBattle().getRobots().iterator();
		DataRobot robot;		
		TrainDataSet dataset;
		
		while(irob.hasNext()){
			robot = irob.next();
			if(robot.getEnemy())
				while(robot.getTrainSet().size() > 0){
		  			dataset = robot.getTrainSet().remove(0);  				  			
		  			robot.getDodgeNet().trainBackpropagation(dataset.getInput(), dataset.getOutput());
				}
				writeDodgeNet(robot);			
		}		
	}
	
	/**
	 * This method should be called when another robot dies
	 */	
	public void onRobotDeath(DataRobot dead) {
		// Train all trainin sets again
		TrainDataSet dataset;
		
		while(dead.getTrainSet().size() > 0){
  			dataset = dead.getTrainSet().remove(0);  				  			
  			dead.getDodgeNet().trainBackpropagation(dataset.getInput(), dataset.getOutput());
		}
		
		writeDodgeNet(dead);
		
	}
	
	/**
	 * This method should be called when Gir collides with another robot
	 */
	public void onHitRobot(HitRobotEvent e){
		DataRobot robot = mGir.getDataBattle().getRobotByName(e.getName());
		robot.adjustEnergyChange(0, -0.6);
		
	}
	
	/**
	 * This method should be called for painting.
	 */
	public void onPaint(Graphics2D g){
		
		Vector<DataRobot> vec = mGir.getDataBattle().getRobots();
		Iterator<VirtualBullet> ivb;
		DataRobot robot = null;		
		VirtualBullet bullet = null;
		Color color;
		
		for(int i = 0; i < vec.size(); i++){
			robot =vec.get(i);
			color = mGir.getColor(i);
			//if(robot.getEnemy()){
			//	grav = robot.getGravPoint();
			//}
			// draw the bullets -->
			ivb = robot.getVirtualBullets().iterator();
			while(ivb.hasNext()){
	  			bullet = ivb.next();
	  			drawVirtualBullet(g, color, bullet.getPosition());
	  		}	
			// <--	
		}
		

		//drawTargeting(g, mLinearTargeting);
		//drawTargeting(g, mCircularTargeting);		
		
		drawGravityForces(g);
		drawGirGravityPoint(g);
	}
	
	
	//	 ####################### onMessage #####################################

	/**
	 * This method should be called when a teammate hits another robot
	 */
	public void onMessageBulletHit(String victim, double power) {
		//System.out.println("MSG: Bullet hit on " + (mGir.getTime() - 1));
		
		DataRobot robot = mGir.getDataBattle().getRobotByName(victim);
		
		robot.adjustEnergyChange(1, -bulletDamage(power));

	}
	
	/**
	 * This method should be called when a teammate is hit by a bullet
	 */	
	public void onMessageHitByBullet(Team_BulletData data) {
		//System.out.println("MSG: hit bb on " + (mGir.getTime() - 1));
		//System.out.println("hit bb on " + mGir.getTime());
		DataRobot robot = mGir.getDataBattle().getRobotByName(data.getOwner());
			
		if(robot.getEnemy()){
			robot.adjustEnergyChange(1, bulletEnergyGain(data.getPower()));
			hitByBullet(robot, data.getX(), data.getY(), data.getHeading(), data.getPower());
		}
	}
		
	/**
	 * This method should be called when a teammate is hit by a bullet
	 */	
	public void onMessageHitRobot(String robot) {
		//System.out.println("MSG: hit bb on " + (mGir.getTime() - 1));
		//System.out.println("hit bb on " + mGir.getTime());
		DataRobot rob = mGir.getDataBattle().getRobotByName(robot);
		rob.setRobotCollision(true);
		rob.adjustEnergyChange(1, -0.6);		
	}
	
	/**
	 * This method should be called when a teammate fired a bullet
	 * @param data the data of the fired bullet
	 */
	public void onMessageBulletFired(Team_BulletData data){	

		DataRobot robot = mGir.getDataBattle().getRobotByName(data.getOwner());
		DataVector originalPos = new DataVector(data.getX(), data.getY());
		
		VirtualBullet vb = new VirtualBullet(data.getTick(), originalPos, 
				bulletVelocity(data.getPower()), new DataVector(data.getHeading()));
		robot.addVirtualBullet(vb);
		
		// Bullet was fired last tick - move it one step
		vb.move();
			
	}

	/**
	 * This method should be called when a teammate collided with another
	 * robot
	 * @param robot the robot our teammate collided with
	 */
	public void onMessageRobotCollision(String robot){
		DataRobot rob = mGir.getDataBattle().getRobotByName(robot);
		rob.setRobotCollision(true);
		rob.adjustEnergyChange(1, -0.6);
	}
	
	// ###############################################################
		
	/**
	 * Finds the corresponding virtual bullet if we are hit by a bullet and
	 * trains the neural net with the obtained data.
	 * @param robot robot that hit us
	 * @param x x-Position of the bullet
	 * @param y y-Position of the bullet
	 * @param heading heading of the bullet (in Robocode-angle)
	 * @param power power of the bullet
	 */
	private void hitByBullet(DataRobot robot, double x, double y, double heading, double power){
		// find the corresponding virtual bullet
		Iterator<VirtualBullet> ivb = robot.getVirtualBullets().iterator();
		VirtualBullet vb = null;
		DataVector diff, bulletdir, bulletpos;
		bulletpos = new DataVector(x, y);
		
		while(ivb.hasNext()){
  			vb = ivb.next();
  			diff = bulletpos.minus(
  					vb.getOriginalPosition());
  			// check if virtual bullet has the right distance
  			if(diff.normalize() - vb.getDistance() < 12){
  				bulletdir = new DataVector(heading);
  				// check if virtual bullet has the right heading
  				if(Math.abs(bulletdir.angleTo(diff)) < 0.1){
  					
	  				if(mGir.getNumTeammates() == 0){
	  					/* train and add to statistics if we're the only
	  					*  robot left (otherwise we won't know if the robot 
	  					*  used scan information about us for his aiming)
	  					*/
	  					trainDodgeNet(robot, vb.getTickFired(), vb.getOriginalPosition(), 
	  							bulletpos, bulletVelocity(power));
	  					robot.hitGir();
  					}
  					// remove bullet
  					ivb.remove();
  					break;
  				}
  			}
  		}	
	}
	
	/**
	 * Adds a virtual bullet. The data at 'tickfired' must be available for
	 * the robot and Gir.
	 */
	private VirtualBullet addVirtualBullet(DataRobot robot, double bulletvel, long tickfired){
		DataRobotSnapshot dataRob = robot.getSnapshot(tickfired);
		DataVector originalPos = dataRob.getPosition();

		DataRobotSnapshot dataGir = mGir.getData().getSnapshot(tickfired);
		DataVector direction = dataGir.getPosition().minus(originalPos);
		direction.normalize();
		
		double angle;
		// create input data for neural net
		double[] input = createNeuralDataInput(dataRob, dataGir, bulletvel);
		
		if(robot.getDodgeNet().getNumTrainings() > 12){
			//	calculate net output and rotate direction by this
			angle = unscaleAngleChange(robot.getDodgeNet().calcOutput(input)[0]);

			if(angle > 0.815)
				angle = 0.8;	
			else if(angle < -0.815)
				angle = -0.8;
		}
		else{
			angle = unscaleAngleChange(0.5 * (input[0] + input[1]));
		}

		direction = direction.rotate(angle);
				
		VirtualBullet vb = new VirtualBullet(tickfired, originalPos, 
				bulletvel, direction);
		robot.addVirtualBullet(vb);

		// bullet has already moved - move it one step
		vb.move();
		
		return vb;		
	}
		
	/**
	 * Creates a TrainDataSet with input Data that can be
	 * fed into a neural net. The DataRobot set for the target
	 * mus be greater than 5.
	 * @param bulletvel Planned bullet velocity
	 * @return
	 */
	private double[] createNeuralDataInput(DataRobotSnapshot dataRob, 
			DataRobotSnapshot dataGir, double bulletvel){
				
		double[] set = new double[mDodgeInputSize];		

		// Calculate angle for linear targeting
		DataVector originaldiff = dataGir.getPosition().minus(dataRob.getPosition());
		double dist = originaldiff.normalize();
		
		double angle = linearTargeting(dataRob.getPosition(), dataGir.getPosition(),
				originaldiff, dist, dataGir.getAbsDirection(), dataGir.getAbsVelocity(), bulletvel);
		set[0] = scaleAngleChange(angle);
		
		DataRobotSnapshot dataGirLast = mGir.getData().getSnapshot(dataGir.getTick() - 1);
		DataVector girOldDir = dataGirLast.getAbsDirection();
		
		// Calculate angle for circular targeting		
		double headingchange = girOldDir.angleTo(dataGir.getAbsDirection());
		angle = circularTargeting(dataRob.getPosition(), dataGir.getPosition(), 
				originaldiff, dist, dataGir.getAbsDirection(), headingchange, dataGir.getAbsVelocity(), bulletvel);
				
		set[1] = scaleAngleChange(angle);
		set[2] = scaleVelocityChange(dataGir.getAbsVelocity() - dataGirLast.getAbsVelocity());
		set[3] = scaleDistance(dist);
		
		return set;
	}
	
	/**
	 * Trains a dodgeNet
	 * @param robot robot whoose net should be trained
	 * @param tickFired tick the bullet was fired
	 * @param originalPos original position of the enemy when he fired the bullet
	 * @param hitpos position where the bullet hit
	 * @param bulletvel velocity of the bullet
	 */
	public void trainDodgeNet(DataRobot robot, long tickFired, DataVector originalPos, 
					DataVector hitpos, double bulletvel){						
		
		DataRobotSnapshot dataGir = mGir.getData().getSnapshot(tickFired);		
		
		TrainDataSet dataset = new TrainDataSet();
		dataset.setInput(createNeuralDataInput(robot.getSnapshot(tickFired), dataGir, bulletvel));
				
		DataVector originaldiff = dataGir.getPosition().minus(originalPos);		
		DataVector currentdiff = hitpos.minus(originalPos);
		
		originaldiff.normalize();
		currentdiff.normalize();				
				
		dataset.setOutput(new double[]{scaleAngleChange(
				originaldiff.angleTo(currentdiff))});

		robot.getTrainSet().add(dataset);
		
		if(mGir.getWriteData() && mGir.isDataSpaceAvailable())
			mGir.getFileManager().appendDataSet(robot.getName() + "_dodgedata.dat", dataset.getDataSet());
		
		robot.getDodgeNet().trainBackpropagation(dataset.getInput(), dataset.getOutput());
		
		/*
		System.out.println("TRAINSET input:");
		for(int j = 0; j < mDodgeInputSize; j++){
			System.out.println(dataset.getInput()[j]);
		}		
		System.out.println("TRAIN output: " + dataset.getOutput()[0]);
		System.out.println("NET output: " + robot.getDodgeNet().calcOutput(dataset.getInput())[0]);
			*/
  			
		// Train a random trainin set (again) and remove it
		if(robot.getTrainSet().size() > 4){
  			dataset = robot.getTrainSet().remove(
				(int)Math.round(Math.random()*(robot.getTrainSet().size() - 1)));
  				  			
  			robot.getDodgeNet().trainBackpropagation(dataset.getInput(), dataset.getOutput());
  			
		}
		
		
	}
	
	/**
	 * Moves Gir in a specific direction
	 * @param diff direction in which to move - must be normalised.
	 */
	private void moveVector(DataVector diff){
		double angle = mGir.getData().getAbsDirection(0).angleTo(diff);
		
		double movedist;
		
		if(Math.random() > mMoveFactors[1]){
			// long move
			movedist = (1 + Math.random()) * mMoveFactors[0] * 5;
		}else{
			// short move
			movedist = (1 + Math.random()) * mMoveFactors[0];
		}
		
		if(Math.abs(angle) > Math.PI * 0.5){
			movedist = -movedist;
		}else{
			mGir.setTurnRightRadians(angle);
		}		
		
		if(mGir.getDistanceRemaining() == .0)
			mGir.setAhead(mGir.getVelocity() < 0 ? -movedist : movedist);
	}

	/**
	 * Checks if between 'currenttick' and 'currenttick - 1' an energy drop has occured
	 * and if so, adds a VirtualBullet
	 */
	private void checkEnergyDrops(){
		Iterator<DataRobot> i = mGir.getDataBattle().getRobots().iterator();
		DataRobot robot;
		double change;
  		while (i.hasNext()){
			robot = i.next();
			if(robot.getEnemy() && robot.checkData(mGir.getTime(), 2) == 0){
				// no teammates - calculate for current tick

				change = robot.getEnergy(0) - robot.getEnergy(-1);
				change -= robot.getEnergyChange(0);
								
				if(change < -0.000001){
					change += checkForWallCollision(robot, 0);
					if((change < -0.000001) && (change >= -3.0)){
						//System.out.println("Energydrop in " + mGir.getTime() + ": " + change);
						addVirtualBullet(robot, bulletVelocity(-change), mGir.getTime());
						// add a fired bullet to robot's gun statistic,
						// ad we assume we are in a 1on1 or 1 team against 1 team,
						// so as no teammate is left, 
						robot.firedAtGir();
					}
				}
				robot.setEnergyChange(0, .0);
			}
		}
	}
		
	/**
	 * Checks if between 'currenttick - 1' and 'currenttick - 2' an energy drop has occured
	 * and if so, adds a VirtualBullet
	 */
	private void checkEnergyDropsTeam(){
		Iterator<DataRobot> i = mGir.getDataBattle().getRobots().iterator();
		DataRobot robot;
		double change;
  		while (i.hasNext()){

			robot = i.next();
  			
			int offset = robot.checkData(mGir.getTime(), 3);
			  			
			if(robot.getEnemy() &&  offset >= 0 && offset <= 1){
				// there are teammates - calculate for last tick
				change = robot.getEnergy(-1) - robot.getEnergy(-2);
				change -= robot.getEnergyChange(1);
				change += checkForWallCollision(robot, offset - 1);
		
				if(change < -0.000001){
					change += checkForWallCollision(robot, -1);
					if((change < -0.000001) && (change >= -3.0)){
						//System.out.println("Energydrop in " + (mGir.getTime() - 1) + ": " + change);
						// Bullet was fired to ticks ago - move it one additional step

			  			addVirtualBullet(robot, bulletVelocity(-change), mGir.getTime() - 1).move();						

					}
					
					//System.out.println("Energydrop in " + (mGir.getTime()- 1) + ": " + change);
					//robot.addBulletToHistory(mGir.getTime()- 1, bulletVelocity(-change));
					
				}
				robot.setEnergyChange(1, robot.getEnergyChange(0));
				robot.setEnergyChange(0, .0);
			}
		}
	}

	/**
	 * Checks if a robot has collided with a wall
	 * @param robot robot for which to check
	 * @param offset tick offset
	 * @return a double representing the energy that the robot lost (i.e. 0 if 
	 * there was no collision)
	 */
	private double checkForWallCollision(DataRobot robot, int offset){
		if(!robot.getRobotCollision()){
			DataVector predicted = robot.getPosition(offset - 1).plus(
						robot.getAbsDirection(offset).multiple(robot.getAbsVelocity(offset)));
			if(predicted.minus(robot.getPosition(offset)).length() > 0.1){
				// Collision: Calculate lost Energy
				return Math.max(0.0, robot.getAbsVelocity(offset - 1) * 0.5 - 1);
			}
		}
		else{
			robot.setRobotCollision(false);
		}
		return 0.0;
	}
	//	 ####################################### helpers #######################################

	/**
	 * Draws the virtual bullets.
	 */
	private void drawVirtualBullet(Graphics2D g, Color middle, DataVector v){
			g.setColor(mColorVB);
			int x = (int)Math.round(v.x);
			int y =(int)Math.round(v.y);
			g.fillOval(x-3, y-3, 6, 6);
			g.setColor(middle);
			g.fillOval(x-2, y-2, 4, 4);
	}
		
	/**
	 * Draws the gravity forces.
	 */
	private void drawGravityForces(Graphics2D g){
		DataVector vec1 = mGir.getData().getPosition(0);
		int x = (int)Math.round(vec1.x);
		int y = (int)Math.round(vec1.y);
		g.setStroke(mStrokeForces);
		int i, team = 0;
		
		// if there is a teammate, then first two DataVectors are teamspot
		if(mMoveTeam.hasTeamTarget()){
			team = 2;
			g.setColor(mColorGravTeam);
			for(i = 0; i < 2; i++){
				vec1 = mGravityForces.get(i);	
				g.drawLine(x, y, x + 20 * (int)Math.round(vec1.x), y + 20 * (int)Math.round(vec1.y));
			}	
		}
		
		//walls and corners
		g.setColor(mColorGravWalls);
		for(i = team; i < 8 + team; i++){
			vec1 = mGravityForces.get(i);	
			g.drawLine(x, y, x + 20 * (int)Math.round(vec1.x), y + 20 * (int)Math.round(vec1.y));
		}	
		// rest
		g.setColor(mColorGravBullets);
		for(i = 8 + team; i < mGravityForces.size() - 1; i++){
			vec1 = mGravityForces.get(i);	
			g.drawLine(x, y, x + 20 * (int)Math.round(vec1.x), y + 20 * (int)Math.round(vec1.y));
		}	
		
		g.setColor(mColorDesired);
		vec1 = mGravityForces.get(mGravityForces.size() - 1);		
		g.drawLine(x, y, x + 40 * (int)Math.round(vec1.x), y + 40 * (int)Math.round(vec1.y));
			
	}
	
	/**
	 * Draws the targeting lines.
	 */
	private void drawTargeting(Graphics2D g, Vector<DataVector> vec){
		if(vec == null)
			return;
		DataVector vec1, vec2 = null;
		g.setStroke(mStrokeForces);
		g.setColor(mColorGravTeam);
		for(int i = 0; i < vec.size() - 1; i++){
			vec1 = vec.get(i);
			vec2 = vec.get(i+1);
			
			g.drawLine((int)Math.round(vec1.x), (int)Math.round(vec1.y), 
					(int)Math.round(vec2.x), (int)Math.round(vec2.y));
		}
	}
	
	/**
	 * Draws the GirGravityPoint
	 */
	private void drawGirGravityPoint(Graphics2D g){		
		g.setStroke(mStrokeForces);
		g.setColor(mColorGravGir);
		int len = (int)Math.round(mGirGravPoint.getForce() * 0.5);
		g.drawOval((int)Math.round(mGirGravPoint.x) - len, (int)Math.round(mGirGravPoint.y) - len, 
					2*len, 2*len);
		
	}
	
	/**
	 * Processes the GirGravityPoint which prevents Gir from getting stucked
	 * near a wall
	 * @param girpos the current position of Gir
	 */
	private void processGirGravityPoint(DataVector girpos){
		if(mGirGravPoint.minus(girpos).length() < 140){
			mGirGravPoint.setForce(1.05 * (mGirGravPoint.getForce() + 1));	
		}else{
			double xdiff, ydiff, newx, newy;
			if(girpos.x < mGir.getBattleFieldWidth() * 0.5)	{			
				xdiff = girpos.x;
				newx = 0;
			}else{
				xdiff = mGir.getBattleFieldWidth() - girpos.x;
				newx = mGir.getBattleFieldWidth();
			}
			
			if(girpos.y < mGir.getBattleFieldHeight() * 0.5){			
				ydiff = girpos.y;
				newy = 0;
			}else{
				ydiff = mGir.getBattleFieldHeight() - girpos.y;
				newy = mGir.getBattleFieldHeight();
			}
			
			if(xdiff < ydiff){
				mGirGravPoint.x = newx;
				mGirGravPoint.y = girpos.y;
			}else{
				mGirGravPoint.x = girpos.x;
				mGirGravPoint.y = newy;
			}
			mGirGravPoint.setForce(0);
		}
	}
	
	/**
	 * Scales the absolute change in angle to (0,1) (for dodgeNet)
	 * @param angle angle to scale
	 * @return scaled angle
	 */
	private double scaleAngleChange(double angle){
		//Angle can't be greater than 0.815 (approx. asin(8/11))
		double maxAbsAngle = 0.815;
		return 0.5 + (0.5 * angle) / maxAbsAngle;
	}	

	/**
	 * Unscales the absolute change in angle to (-PI,PI) (for dodgeNet)
	 * @param angle angle to unscale
	 * @return unscaled angle
	 */
	private double unscaleAngleChange(double angle){
		double maxAbsAngle = 0.815;
		return (angle - 0.5) * maxAbsAngle * 2;
	}
	
	/**
	 * Scales the signed velocity to (0,1(
	 * @param velocity velocity to scale
	 * @return scaled velocity
	 */
	private double scaleVelocityChange(double velocity){
		return (velocity + 8) * 0.0625;
	}	
	
	/**
	 * Calculates a bullets velocity
	 * @param power Power of the fired bullet
	 * @return Velocity of the bullet
	 */
	private double bulletVelocity(double power) {
		return (double) 20 - 3*power;
	}
	
	/**
	 * Calculates a bullets damage
	 * @param power Power of the fired bullet
	 * @return Damage of the bullet
	 */
	private double bulletDamage(double power){
		if(power > 1)
			return 6 * power - 2;
		else
			return 4 * power;
	}
	
	/**
	 * Calculates the energy a robot would gain when
	 * one of his bullets hits
	 * @param power Power of the fired bullet
	 * @return Energy we would gain by this hit
	 */
	private double bulletEnergyGain(double power){
		return 3 * power;
	}
	
	/**
	 * Checks if a robot position is in the battlefield
	 * @param v Position to be checked
	 * @return true if v is in the battlefield, otherwise false
	 */	
	private boolean isInBattleField(DataVector v) {
		double dist = 4;
		if(v.x < dist || v.y < dist || v.x > mGir.getBattleFieldWidth()-dist || v.y > mGir.getBattleFieldHeight()-dist)
			return false;
		return true;
	}	
	
	// ########### Data handling ##########################

	/**
	 * create "robotname_dodgedata.dat" file if it does not exist
	 */ 
	public void writeDodgeDataFile(DataRobot robot){
		if(mGir.getWriteData()){
			// create "dodgedata" file if it does not exist
			File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_dodgedata.dat");
			if(!file.exists())
				mGir.getFileManager().writeNetLayersize(file, robot.getDodgeNet());
		}
	}
	
	/**
	 * Reads the dodge net for this robot.
	 */
	public void readDodgeNet(DataRobot robot){

		readEnemyGunStats(robot);
		
		File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_dodge.net");
		if((robot.getSizeRobotStats() >= mRobotsStatsUseRounds) && robot.getAverageRobotStats() > 0.2){
			// after 'mSizeRobotsUseRounds' round, test if the gun statistics for the enemy
			// are very high. If so, start with a new randomly initialised net.
			System.out.println("!!! Low dodge rate. Creating new dodge net for " + robot.getName());
			int[] size = {mDodgeInputSize, mDodgeInputSize, mDodgeInputSize, 1};	
			
			NeuralNetData data = new NeuralNetData(size, 0.7, -0.0003, 3, 0.4);
			robot.setDodgeNet(new NeuralNet(data));
			addNewMovementFlag(robot);
			
		}else{
			
			if(file.exists()){
				robot.setDodgeNet(mGir.getFileManager().readNet(file));
				System.out.println("--READ: DodgeNet for " + robot.getName());
			}else{
				file = new File(mGir.getDataDirectory() + "/_initial_dodge.net");
				//read initial net
				if(file.exists()){
					robot.setDodgeNet(mGir.getFileManager().readNet(file));
					System.out.println("--READ: Initial DodgeNet");
				}
				else{							
					int[] size = {mDodgeInputSize, mDodgeInputSize, mDodgeInputSize, 1};	
					
					NeuralNetData data = new NeuralNetData(size, 0.7, -0.0003, 3, 0.4);
					robot.setDodgeNet(new NeuralNet(data));
					System.out.println("--DodgeNet created for " + robot.getName());
				}
			}	
		}
	}
		
	/**
	 * Writes the dodge net for this robot (overwrites older files)
	 * @param robot
	 */
	public void writeDodgeNet(DataRobot robot){
		if(robot.getEnemy()){

			writeEnemyGunStats(robot);	
			
			File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_dodge.net");
			mGir.getFileManager().writeNetRC(file, robot.getDodgeNet());
			System.out.println("--WRITTEN: Dodge net for " + robot.getName());
			if(!mGir.isSpaceAvailable()){
				file.delete();
				System.out.println("--Not enough disk space available: Deleted dodge net for " + robot.getName());
			}
		}
	}
	
	
	// ********** for enemies targeting ******************
		
	/**	  
	 * Aims ahead of the target for linear movement. mTarget must not be null.
	 */
	public double linearTargeting(DataVector attackerpos, DataVector girpos, DataVector originalDiff, 
				double dist, DataVector girDirection, double girvelocity, double bulletvel){
				
		mLinearTargeting.removeAllElements();
		//	calculate iterative circular targeting
		double travelled = .0;
		while(girpos.minus(attackerpos).length() > travelled){
			mLinearTargeting.add(girpos);
			girpos = girpos.plus(girDirection.multiple(girvelocity));
			if(!isInBattleField(girpos)){
				girpos = girpos.minus(girDirection.multiple(girvelocity));
				break;
			}
			travelled += bulletvel;
		}
		
		girpos = girpos.minus(attackerpos);
		girpos.normalize();
		return originalDiff.angleTo(girpos);

	}	
		
	/**	  
	 * Aims ahead of the target for circular movement. mTarget must not be null.
	 */
	public double circularTargeting(DataVector attackerpos, DataVector girpos, DataVector originalDiff, 
			double dist, DataVector girDirection, double headingchange, double girvelocity, double bulletvel){
		
		mCircularTargeting.removeAllElements();
		// calculate iterative circular targeting
		double travelled = .0;

		while(girpos.minus(attackerpos).length() > travelled){
			mCircularTargeting.add(girpos);
			girpos = girpos.plus(girDirection.multiple(girvelocity));
			if(!isInBattleField(girpos)){
				girpos = girpos.minus(girDirection.multiple(girvelocity));
				break;
			}
			girDirection = girDirection.rotate(headingchange);
			travelled += bulletvel;
		}
		
		girpos = girpos.minus(attackerpos);
		girpos.normalize();
		return originalDiff.angleTo(girpos);

	}
		
	/**
	 * Scales the distance from 0-5000 to 0-1
	 * @param distance Distance to be scaled
	 * @return A value between 0 and 1
	 */
	private double scaleDistance(double distance){
		return Math.sqrt(distance / 5000.0);
	}	
		
	/**
	 * Adds a flag to a robot's gun statistics Gir has changed his movement.
	 * @param robot the robot for which to set this flag
	 */
	@SuppressWarnings("unchecked")
	private void addNewMovementFlag(DataRobot robot){
		File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_robotstats.stt");
		
		Vector<Double> stats;
		
		if(!file.exists() || (mGir.getRoundNum() == 0))
			stats = new Vector<Double>();
		else				
			stats = mGir.getFileManager().readVector(file);
		
		stats.add(-1.0);
		mGir.getFileManager().writeVector(file, stats);
	}
		
	/**
	 * Writes the gun statistics for a robot
	 * @param robot the robot for which the statistics shall
	 * 			be written - must not be null
	 */
	@SuppressWarnings("unchecked")
	private void writeEnemyGunStats(DataRobot robot) {
						
		File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_robotstats.stt");
		
		Vector<Double> stats;
		
		if(!file.exists() || (mGir.getRoundNum() == 0))
			stats = new Vector<Double>();
		else				
			stats = mGir.getFileManager().readVector(file);
		
		if(robot.getCurrentRobotStats() != -2)
			stats.add(robot.getCurrentRobotStats());
		
		double temp = .0;
		int size = 0;
		double current;
  		
		for(int i = 0; i < stats.size(); i++){
  			current = (Double)stats.get(i);
  			if(current == -1.0){
  				// minus one means a new net has been created
  				temp = 0;
  				size = 0;
  			}else{
	  			if(i >= stats.size() - mRobotsStatsUseRounds){  					
	  				// only use the last 'mRobotsStatsUseRounds' rounds
	  				temp += current;
	  				size++;
  				}
  			}
		}
  		  		
		if(!mGir.getWriteData() && mGir.isLastRound()){
			file.delete();
			System.out.println("~~Last round: deleted " + robot.getName() + "'s gun statistics file for Gir");
			System.out.println("~~" + robot.getName() + "'s GunStats for Gir: "+
					"Average for last " + size + " rounds: " + temp / size);
		}
		else{
			mGir.getFileManager().writeVector(file, stats);	
			System.out.println("~~WRITTEN: " + robot.getName() + "'s GunStats for Gir: "+
					"Average for last " + size + " rounds: " + temp / size);
		}	
	}
	
	/**
	 * Reads the gun statistics for a robot
	 * @param robot The robot for which statistics should be read
	 */
	private void readEnemyGunStats(DataRobot robot){
		File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_robotstats.stt");
			
		if(!file.exists() || (mGir.getRoundNum() == 0)){
			robot.setAverageRobotStats(0.5);
			return;
		}
			
		Vector stats = mGir.getFileManager().readVector(file);
		
		double temp = .0;
		double current;
		int size = 0;
				
  		for(int i = 0; i < stats.size(); i++){
  			current = (Double)stats.get(i);
  			if(current == -1.0){
  				// minus one means a new net has been created
  				temp = 0;
  				size = 0;
  			}else if(i >= stats.size() - mRobotsStatsUseRounds){
  				// only use the last 'mRobotsStatsUseRounds' rounds
  				temp += current;
  				size++;
  			}
		}
  		System.out.println("~~READ: " + robot.getName() + "'s GunStats for Gir: "+
					"Average for last " + size + " rounds: " + temp / size);
		robot.setAverageRobotStats(temp / size);
		robot.setSizeRobotStats(size);
	}
	
	/**
	 * Reads the move factors for a robot
	 * @param robot The robot for which move factors should be read 
	 * - must not be null
	 */
	public void readMoveFactors(DataRobot robot){
		File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_move.fct");
		
		if(!file.exists()){
			mMoveFactors[0] = 60;
			mMoveFactors[1] = 0.2;
		}else{
			Vector factors = mGir.getFileManager().readVector(file);
			mMoveFactors[0] = (Double)factors.get(0);
			mMoveFactors[1] = (Double)factors.get(1);
	  		System.out.println("~~READ: Move Factors for " + robot.getName() + ": " + 
	  				mMoveFactors[0] + ", " + mMoveFactors[1]);
		}	
			
		if((robot.getSizeRobotStats() >= mRobotsStatsUseRounds) && robot.getAverageRobotStats() > 0.2){
			// after 'mSizeRobotsUseRounds' round, test if the gun statistics for the enemy
			// are very high. If so, start with new move factors.
			System.out.println("!!! Low dodge rate. Creating new move factors for " + robot.getName());

			
			if(mMoveFactors[0] > 10)
				mMoveFactors[0] /= 2;
			else
				mMoveFactors[0] = 100;
				
			if(mMoveFactors[1] > 0.2)
					mMoveFactors[1] /= 2;
				else
					mMoveFactors[1] = 0.9;
			
		}
	}
	
	/**
	 * Writes the move factors for a robot
	 * @param robot the robot for which the move factors shall
	 * 			be written - must not be null
	 */
	public void writeMoveFactors(DataRobot robot) {
						
		File file = new File(mGir.getDataDirectory() + "/" + robot.getName() + "_move.fct");
		
		Vector<Double> factors = new Vector<Double>();
		factors.add(mMoveFactors[0]);
		factors.add(mMoveFactors[1]);
		  		  		
		if(!mGir.getWriteData() && mGir.isLastRound()){
			file.delete();
			System.out.println("~~Last round: deleted move factors file for " + robot.getName());
		}
		else{
			mGir.getFileManager().writeVector(file, factors);	
			System.out.println("~~WRITTEN: Move factors for " + robot.getName());	
		}	
	}
	
}
