/*
 * MovableData - container for information of any moving object.
 * Copyright (C) 2002  Joachim Hofer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * You can contact the author via email (qohnil@johoop.de) or write to
 * Joachim Hofer, Feldstr. 12, D-91052 Erlangen, Germany.
 */

package qohnil.blot;

import robocode.ScannedRobotEvent;
import robocode.AdvancedRobot;

import java.io.Serializable;

import qohnil.util.Coords;
import qohnil.util.BotMath;
import qohnil.util.Polar;

public class MovableData implements Serializable {

    public Coords coords = null;
    public double heading = 0.0;
    public double velocity = 0.0;
    double bearing = 0.0;
    double distance = 0.0;
    double energy = 0.0;
    long lastAttackTime = -1000;
    boolean wasAlreadyAttacked = false;

    long time = 0;
    public double turnVelocity = 0.0;
    String name = null;
    boolean isTarget = false;
    boolean isDroid = false;
    boolean isLeader = false;
    boolean isAlive = true;

    public MovableData() {}

    public MovableData(ScannedRobotEvent e, double origX, double origY, double origBearing, MovableData previous) {
        bearing = BotMath.normalizeAbsoluteAngle(
                e.getBearingRadians() + origBearing);
        distance = e.getDistance();
        coords = BotMath.polarToCoords(bearing, distance, new Coords(origX, origY));
        heading = e.getHeadingRadians();
        velocity = e.getVelocity();
        time = e.getTime();
        name = e.getName();
        energy = e.getEnergy();
        isAlive = true;
        if (previous != null) {
            lastAttackTime = previous.lastAttackTime;
            wasAlreadyAttacked = previous.wasAlreadyAttacked;
            isTarget = previous.isTarget;
            isDroid = previous.isDroid;
            isLeader = previous.isLeader;
            turnVelocity
                    = BotMath.normalizeRelativeAngle(heading - previous.heading)
                    / (time - previous.time);
        }
    }

    public MovableData(AdvancedRobot bot, MovableData previous) {
        bearing = 0.0;
        distance = 0.0;
        coords = new Coords(bot.getX(), bot.getY());
        heading = bot.getHeadingRadians();
        time = bot.getTime();
        name = bot.getName();
        energy = bot.getEnergy();
        velocity = bot.getVelocity();
        if (previous != null) {
            turnVelocity
                    = BotMath.normalizeRelativeAngle(heading - previous.heading)
                    / (time - previous.time);
        }
        isAlive = true;
    }

    public void setAsTarget(boolean b) { isTarget = b; }

    public void setLeader() { isLeader = true; }
    public void setDroid() { isDroid = true; }

    public boolean isLeader() { return isLeader; }
    public boolean isDroid() { return isDroid; }
    public boolean isScout() { return !isDroid; }

    public void setLastAttackTime(long time) { lastAttackTime = time; }

    public void updatePolar(double origX, double origY) {
        Polar p = BotMath.coordsToPolar(coords, new Coords(origX, origY));
        bearing = p.getBearing();
        distance = p.getDistance();
    }

    Coords getLinearPredictedCoords(long passedTime, long currentTime) {
        passedTime += currentTime - time;
        return getLinearPredictedCoords(passedTime, coords, heading, velocity);
    }

    public static Coords getLinearPredictedCoords(long passedTime,
                                    Coords coords, double heading,
                                    double velocity) {
        double factor = passedTime * velocity;
        Coords newCoords = new Coords(
            coords.getX() + factor * Math.sin(heading),
            coords.getY() + factor * Math.cos(heading));

        return newCoords;
    }

    Coords getCircularPredictedCoords(long passedTime, long currentTime) {
        if (turnVelocity == 0.0) {
            return getLinearPredictedCoords(passedTime, currentTime);
        }

        double angle = (passedTime + currentTime - time) * turnVelocity;
        double cosAngle = Math.cos(angle);
        double sinAngle = Math.sin(angle);
        double cosHeading = Math.cos(heading);
        double sinHeading = Math.sin(heading);

        return new Coords(
                ((1.0 - cosAngle) * cosHeading
                        + sinAngle * sinHeading)
                        * velocity / turnVelocity + coords.getX(),
                (sinAngle * cosHeading
                        - (1.0 - cosAngle) * sinHeading)
                        * velocity / turnVelocity + coords.getY());
    }

}
