package tzu.intel;

import tzu.util.*;
import tzu.gun.*;
import java.io.*;
import robocode.*;

/**
 * Extensively describes an enemy bot and provides methods for
 * improving your chances of shooting the enemy bot.
 */
public class Bot extends BasicBot {

    public  static final int    FILE_VERSION = 11;
    public  static final int    ROBOT_FILE_SIZE =
    PreviousReactions.REACTION_FILE_SIZE + 300;

    double prevBearing;         // Previous bearing from us.
    double prevDistance;        // Previous distance from us.

    double  maxSpeed;           // Max speed clocked thus far.
    double  totSpeed;           // Used to calculate average speed.
    double  totSqSpeed;         // Used to calculate standard deviation.
    double  avgSpeed;           // Average speed of robot thus far.
    double  totAcceleration;    // Total of all acceleration amounts.
    int     accelerationCount;  // Total number of times scanned accelerating.
    double  avgAcceleration;    // Average acceleration.
    double  totDeacceleration;  // Total of all deacceleration amounts.
    int     deaccelerationCount;// Total number of times scanned deaccelerating.
    double  avgDeacceleration;  // Average deacceleration.
    double  sdSpeed;            // Standard deviation from average speed.
    int     totStops;           // Number of total stops by this robot.
    double  avgStopsPer100Turns;// Average number of stops per 100 turns.

    double  totPivotSpeed;      // Used to calculate pivot speed;
    double  totSqPivotSpeed;    // Used to calultate standard deviation.
    int     pivotCount;         // Used to calculate pivot speed;
    double  pivotIndicator;     // Indicator of how often robot varies speed.
    double  pivotSpeed;         // Point at which robot switches from
                                // acceleration to deacceleration or vise versa.
    double  sdPivotSpeed;       // Standard deviation from pivot speed.

    long    totTimeTopSpeed;    // Used to calulate average time at top speed.
    long    totSqTimeTopSpeed;  // Used to calultate standard deviation.
    long    avgTimeTopSpeed;    // Average time robot spends at top speed.
    long    prevTimeTopSpeed;   // Previous time spent at top speed.
    long    sdTimeTopSpeed;     // Standard deviation of above field.
    long    maxTimeTopSpeed;    // Maximum time at top speed.
    long    startTimeTopSpeed;  // Time at which robot reached top speed.
    long    lastTimeTopSpeed;   // Last time this robot reached top speed.
    double  startXTopSpeed;     // X coordinate where robot reached top speed.
    double  startYTopSpeed;     // Y coordinate where robot reached top speed.
    int     topSpeedCount;      // Number of times robot reached top speed.
    double  totDistTopSpeed;    // Used to calculate average distance covered.
    double  totSqDistTopSpeed;  // Used to calculate standard deviation.
    double  avgDistTopSpeed;    // Average distance covered at top speed.
    double  sdDistTopSpeed;     // Standard deviation of above field.
    double  maxDistTopSpeed;    // Maximum distance covered at top speed.
    long    avgLast2TimesTopSpeed;  // Average time spent at top speed last
                                    // two times.

    double  prevX;              // Previous x coordinate.
    double  prevY;              // Previous y coordinate.
    double  prevStoppedX;       // Previous stop point x coord.
    double  prevStoppedY;       // Previous stop point y coord.

    double  minDistToWall;      // Minimum distance to closest wall.
    double  totDistToWall;      // Used to calculate average distance to wall.
    double  avgDistToWall;      // Average distance to closest wall.

    double  minDistToCorner;    // Minimum distance to closest corner.
    double  totDistToCorner;    // Used to calculate avg distance to corner.
    double  avgDistToCorner;    // Average distance to closest corner.

    int     inCornerCount;      // Number of times this bot in corner.
    double  inCornerPct;        // Percent of time spent in corner.

    /* LOS = line of site - point A to point B */

    double  currLosMove;    // Current LOS move thus far.
    double  prevLosMove;    // Previous LOS move.
    double  maxLosMove;     // Max LOS move thus far.
    double  totLosMove;     // Sum of LOS moves thus far.
    double  totSqLosMove;   // Used to calculate standard deviation.
    double  avgLosMove;     // Average LOS move thus far.
    double  avgLast2LosMove;// Average LOS distance moved last 2 times.
    double  sdLosMove;      // Standard deviation of above field.

    /* EST = estimated using short straight lines added together */

    double  currEstMove;    // Current EST move.
    double  prevEstMove;    // Previous EST move.
    double  maxEstMove;     // Max EST move thus far.
    double  totEstMove;     // Sum of EST moves thus far.
    double  totSqEstMove;   // Used to calculate standard deviation.
    double  avgEstMove;     // Average EST move thus far.
    double  avgLast2EstMove;// Average EST distance moved last 2 times.
    double  sdEstMove;      // Standard deviation of above field.

    int     shotMeCount;    // Number times this robot shot us.
    long    lastShotMeTime; // Last time this robot shot us.
    double  totShotMePower; // Used to calculate average bullet power.
    double  maxShotMePower; // Max bullet power used to shoot us.
    double  avgShotMePower; // Avg bullet power used to shoot us.

    int     shotAtCount;    // Number bullets shot at this robot by us.
    double  totShotAtPower; // Used to calculate average bullet power.
    double  maxShotAtPower; // Max bullet power used by us to shoot at this bot.
    double  avgShotAtPower; // Avg bullet power used by us to shoot at this bot.

    int     hitCount;       // Number of direct bullet hits by us.
    double  hitPct;         // Percentage of direct bullet hits by us.
    double  totHitPower;    // Used to calculate average hit power.
    double  maxHitPower;    // Max bullet power that hit this bot.
    double  avgHitPower;    // Avg bullet power that hit this bot.

    double  prevHitDamage;  // Damage caused by our last bullet hit.
    double  prevHitTime;    // Last time we shot this bot.

    int     rammedMeCount;      // Number of times this bot rammed us.
    int     diedBeforeMeCount;  // Number of times this bot died before us.
    double  diedBeforeMePct;    // Percentage of times robot died before us.
    int     roundsPlayed;       // Number of rounds played against this bot.
    long    timePlayed;         // Total time played against this robot.

    int     predictAimShotAtCount;  // Bullets shot using predictive aiming.
    int     predictAimHitCount;     // Bullet hits using predictive aiming.
    double  predictHitPct;          // Predict hit percentage.
    int     estimateAimShotAtCount; // Bullets shot using estimate aiming.
    int     estimateAimHitCount;    // Bullet hits using estimate aiming.
    double  estimateHitPct;         // Estimate hit percentage.
    int     patternAimShotAtCount;  // Bullets shot using pattern aiming.
    int     patternAimHitCount;     // Bullet hits using pattern aiming.
    double  patternHitPct;          // Pattern aim hit percentage.
    int     averageAimShotAtCount;  // Bullets shot using average aiming.
    int     averageAimHitCount;     // Bullet hits using average aiming.
    double  averageHitPct;          // Average aim hit percentage.

    int     moveForwardCount;       // Number of times scanned moving forward.
    int     moveBackwardCount;      // Number of times scanned moving backward.
    double  biDirectionalPct;       // Bi-directional percentage.
    double  prevDirection;          // Previous direction of movement
    int     directionChangeCount;   // # times switched from forward to reverse.
    int     sideToSideCount;        // Number of side to side moves counted.
    double  sideToSidePct;          // Percentage of side to side moves.
    int     prevDriveDirection;     // Previous direction of movement.
    int     sameDirectionCount;     // Number of times robot continued in same
                                    // direction after stopping.
    double  sameDirectionPct;       // Percentage of times robot continues to
                                    // move in same direction after stopping.
    int     moveCount;              // Number of moves thus far.
    int     wiggleCount;            // Number of short moves thus far
                                    // (5 or less pixels in length).
    double  wigglePct;              // Percentage of wiggle moves.

    double  totAngleOfApproach;     // Used to calculate avg angle of approach.
    double  totSqAngleOfApproach;   // Used to calculate standard deviation.
    double  avgAngleOfApproach;     // Average angle of approach.
    double  sdAngleOfApproach;      // Standard deviation angle of approach.

    PreviousReactions   pr;         // Previous reactions to bullets.

    /**
     * Create a new Bot object.
     * @param e     A ScannedRobotEvent used to initialize this bot.
     * @param ar    Your AdvancedRobot.
     */
    public Bot(ScannedRobotEvent e, AdvancedRobot ar) {

        super(e, ar);

        pr                      = new PreviousReactions();

        prevBearing             = bearing;
        prevDistance            = distance;

        maxSpeed                = speed;
        totSpeed                = speed;
        totSqSpeed              = speed * speed;
        avgSpeed                = speed;
        sdSpeed                 = 0.0;
        totStops                = speed == 0 ? 1 : 0;
        avgStopsPer100Turns     = 0.0;

        totAcceleration         = 0.0;
        totDeacceleration       = 0.0;
        accelerationCount       = 0;
        deaccelerationCount     = 0;
        avgAcceleration         = ROBOT_ACCELERATION;
        avgDeacceleration       = ROBOT_DEACCELERATION;

        totPivotSpeed           = 0.0;
        totSqPivotSpeed         = 0.0;
        pivotCount              = 0;
        pivotIndicator          = 0.0;
        pivotSpeed              = 0.0;
        sdPivotSpeed            = 0.0;

        totTimeTopSpeed         = 0;
        prevTimeTopSpeed        = 0;
        totSqTimeTopSpeed       = 0;
        startTimeTopSpeed       = 0;
        lastTimeTopSpeed        = 0;
        startXTopSpeed          = 0.0;
        startYTopSpeed          = 0.0;
        topSpeedCount           = 0;
        avgTimeTopSpeed         = 0;
        sdTimeTopSpeed          = 0;
        maxTimeTopSpeed         = 0;
        totDistTopSpeed         = 0.0;
        totSqDistTopSpeed       = 0.0;
        avgDistTopSpeed         = 0.0;
        sdDistTopSpeed          = 0.0;
        maxDistTopSpeed         = 0.0;

        prevX                   = x;
        prevY                   = y;
        prevStoppedX            = x;
        prevStoppedY            = y;

        minDistToWall           = BotMath.min(
                                  BotMath.min(x, y),
                                  BotMath.min(BattleField.getWidth() - x,
                                  BattleField.getHeight() - y));
        totDistToWall           = minDistToWall;
        avgDistToWall           = minDistToWall;

        minDistToCorner         = BattleField.distanceToClosestCorner(x, y);
        totDistToCorner         = minDistToCorner;
        avgDistToCorner         = minDistToCorner;
        inCornerCount          += (BattleField.isInCorner(x, y)) ? 1 : 0;
        inCornerPct             = 0;

        currLosMove             = 0.0;
        prevLosMove             = 0.0;
        maxLosMove              = 0.0;
        totLosMove              = 0.0;
        totSqLosMove            = 0.0;
        avgLosMove              = 0.0;
        sdLosMove               = 0.0;

        currEstMove             = 0.0;
        prevEstMove             = 0.0;
        maxEstMove              = 0.0;
        totEstMove              = 0.0;
        totSqEstMove            = 0.0;
        avgEstMove              = 0.0;
        sdEstMove               = 0.0;

        shotMeCount             = 0;
        lastShotMeTime          = 0;
        totShotMePower          = 0.0;
        maxShotMePower          = 0.0;
        avgShotMePower          = 0.0;

        shotAtCount             = 0;
        totShotAtPower          = 0.0;
        maxShotAtPower          = 0.0;
        avgShotAtPower          = 0.0;

        hitCount                = 0;
        hitPct                  = 0.0;
        totHitPower             = 0.0;
        maxHitPower             = 0.0;
        avgHitPower             = 0.0;

        prevHitDamage           = 0.0;
        prevHitTime             = 0;

        rammedMeCount           = 0;
        diedBeforeMeCount       = 0;
        diedBeforeMePct         = 0.0;
        roundsPlayed            = 1;
        timePlayed              = e.getTime();

        predictAimShotAtCount   = 0;
        predictAimHitCount      = 0;
        predictHitPct           = 0.0;
        estimateAimShotAtCount  = 0;
        estimateAimHitCount     = 0;
        estimateHitPct          = 0.0;
        patternAimShotAtCount   = 0;
        patternAimHitCount      = 0;
        patternHitPct           = 0.0;
        averageAimShotAtCount   = 0;
        averageAimHitCount      = 0;
        averageHitPct           = 0.0;

        moveForwardCount       += (gameSpeed >= 0.0) ? 1 : 0.0;
        moveBackwardCount      += (gameSpeed < 0.0)  ? 1 : 0.0;
        prevDirection           = Double.MAX_VALUE;
        directionChangeCount    = 0;
        sideToSideCount         = 0;
        sideToSidePct           = 0.0;
        moveCount               = 0;
        wiggleCount             = 0;
        wigglePct               = 0;
        sameDirectionCount      = 0;
        prevDriveDirection      = (gameSpeed >= 0.0) ? FORWARD : REVERSE;

        totAngleOfApproach      = 0.0;
        avgAngleOfApproach      = 0.0;
        totSqAngleOfApproach    = 0.0;
        sdAngleOfApproach       = 0.0;
    }

    /**
     * A method to reinitialize a bot on the second and subsequent
     * rounds of a battle.
     */
    public void reinitialize() {

        super.reinitialize();

        prevBearing             = bearing;
        prevDistance            = distance;

        startTimeTopSpeed       = 0;
        lastTimeTopSpeed        = 0;
        startXTopSpeed          = 0.0;
        startYTopSpeed          = 0.0;
        prevTimeTopSpeed        = 0;

        prevX                   = 0.0;
        prevY                   = 0.0;
        prevStoppedX            = 0.0;
        prevStoppedY            = 0.0;

        currLosMove             = 0.0;
        prevLosMove             = 0.0;
        currEstMove             = 0.0;
        prevEstMove             = 0.0;

        prevHitDamage           = 0.0;
        prevHitTime             = 0;
        lastShotMeTime          = 0;

        roundsPlayed++;

        prevDirection           = Double.MAX_VALUE;
        prevDriveDirection      = FORWARD;
    }


    //*************************************************************************
    //  ACCESSOR METHODS
    //*************************************************************************

    public double getPrevBearing()          { return prevBearing; }
    public double getPrevDistance()         { return prevDistance; }
    public double getPrevHitDamage()        { return prevHitDamage; }
    public double getPrevHitTime()          { return prevHitTime; }
    public double getCurrEstMove()          { return currEstMove; }
    public double getCurrLosMove()          { return currLosMove; }
    public double getMaxEstMove()           { return maxEstMove; }
    public double getMaxLosMove()           { return maxLosMove; }
    public double getMaxSpeed()             { return maxSpeed; }
    public double getPrevX()                { return prevX; }
    public double getPrevY()                { return prevY; }
    public int    getDiedBeforeMeCount()    { return diedBeforeMeCount; }
    public double getAvgDistToCorner()      { return avgDistToCorner; }
    public double getAvgDistTopSpeed()      { return avgDistTopSpeed; }
    public double getAvgDistToWall()        { return avgDistToWall; }
    public double getAvgEstMove()           { return avgEstMove; }
    public double getAvgHitPower()          { return avgHitPower; }
    public double getAvgLast2LosMove()      { return avgLast2LosMove; }
    public double getAvgLast2EstMove()      { return avgLast2EstMove; }
    public long getAvgLast2TimesTopSpeed()  { return avgLast2TimesTopSpeed; }
    public double getAvgLosMove()           { return avgLosMove; }
    public double getWigglePct()            { return wigglePct; }
    public double getAvgShotAtPower()       { return avgShotAtPower; }
    public double getAvgShotMePower()       { return avgShotMePower; }
    public long getLastShotMeTime()         { return lastShotMeTime; }
    public double getAvgSpeed()             { return avgSpeed; }
    public long getAvgTimeTopSpeed()        { return avgTimeTopSpeed; }
    public double getBiDirectionalPct()     { return biDirectionalPct; }
    public double getEstimateHitPct()       { return estimateHitPct; }
    public double getHitPct()               { return hitPct; }
    public double getInCornerPct()          { return inCornerPct; }
    public double getMaxDistTopSpeed()      { return maxDistTopSpeed; }
    public double getMaxHitPower()          { return maxHitPower; }
    public double getMaxShotAtPower()       { return maxShotAtPower; }
    public double getMaxShotMePower()       { return maxShotMePower; }
    public long getMaxTimeTopSpeed()        { return maxTimeTopSpeed; }
    public double getMinDistToCorner()      { return minDistToCorner; }
    public double getMinDistToWall()        { return minDistToWall; }
    public double getPivotIndicator()       { return pivotIndicator; }
    public double getPivotSpeed()           { return pivotSpeed; }
    public double getPredictHitPct()        { return predictHitPct; }
    public double getPatternHitPct()        { return patternHitPct; }
    public double getAverageHitPct()        { return averageHitPct; }
    public double getPrevEstMove()          { return prevEstMove; }
    public double getPrevLosMove()          { return prevLosMove; }
    public long getPrevTimeTopSpeed()       { return prevTimeTopSpeed; }
    public double getSdDistTopSpeed()       { return sdDistTopSpeed; }
    public double getSdEstMove()            { return sdEstMove; }
    public double getSdLosMove()            { return sdLosMove; }
    public double getSdPivotSpeed()         { return sdPivotSpeed; }
    public double getSdSpeed()              { return sdSpeed; }
    public long getSdTimeTopSpeed()         { return sdTimeTopSpeed; }
    public double getSideToSidePct()        { return sideToSidePct; }
    public long getStartTimeTopSpeed()      { return startTimeTopSpeed; }
    public double getStartXTopSpeed()       { return startXTopSpeed; }
    public double getStartYTopSpeed()       { return startYTopSpeed; }
    public int getRammedMeCount()           { return rammedMeCount; }
    public int getRoundsPlayed()            { return roundsPlayed; }
    public long getTimePlayed()             { return timePlayed; }
    public double getAvgStopsPer100Turns()  { return avgStopsPer100Turns; }
    public double getSameDirectionPct()     { return sameDirectionPct; }
    public double getAvgAngleOfApproach()   { return avgAngleOfApproach; }
    public double getSdAngleOfApproach()    { return sdAngleOfApproach; }
    public double getDiedBeforeMePct()      { return diedBeforeMePct; }
    public int getShotAtCount()             { return shotAtCount; }
    public int getPredictAimShotAtCount()   { return predictAimShotAtCount; }
    public int getEstimateAimShotAtCount()  { return estimateAimShotAtCount; }
    public int getPatternAimShotAtCount()   { return patternAimShotAtCount; }
    public int getAverageAimShotAtCount()   { return averageAimShotAtCount; }

    public Reaction getPreviousReaction() {
        return pr.get(heading, bearing, speed, chgSpeed, distance);
    }

    //*************************************************************************
    //  MUTATOR METHODS
    //*************************************************************************

    /** Update the rammed me count. */
    public void rammedMe() { rammedMeCount++; }
    /** Increment the predict aim shot at count. */
    public void predictAimShotAt() { predictAimShotAtCount++; }
    /** Increment the estimate aim shot at count. */
    public void estimateAimShotAt() { estimateAimShotAtCount++; }
    /** Increment the pattern aim shot at count. */
    public void patternAimShotAt() { patternAimShotAtCount++; }
    /** Increment the average aim shot at count. */
    public void averageAimShotAt() { averageAimShotAtCount++; }

    /** Update the predict aim hit count and percentage. */
    public void predictAimHit() {
        predictAimHitCount++;
        if (predictAimShotAtCount > 0) {
            predictHitPct =
                (double)predictAimHitCount
                / predictAimShotAtCount
                * 100;
        }
    }


    /** Update the estimate aim hit count and percentage. */
    public void estimateAimHit() {
        estimateAimHitCount++;
        if (estimateAimShotAtCount > 0) {
            estimateHitPct =
                (double)estimateAimHitCount
                / estimateAimShotAtCount
                * 100;
        }
    }


    /** Update the pattern aim hit count and percentage. */
    public void patternAimHit() {
        patternAimHitCount++;
        if (patternAimShotAtCount > 0) {
            patternHitPct =
                (double)patternAimHitCount
                / patternAimShotAtCount
                * 100;
        }
    }


    /** Update the average aim hit count and percentage. */
    public void averageAimHit() {
        averageAimHitCount++;
        if (averageAimShotAtCount > 0) {
            averageHitPct =
                (double)averageAimHitCount
                / averageAimShotAtCount
                * 100;
        }
    }


    /** Increment the shot at count.  Keep track of bullet power used. */
    public void shotAt(double power) {

        shotAtCount++;
        totShotAtPower += power;
        maxShotAtPower  = BotMath.max(maxShotAtPower, power);
        avgShotAtPower  = totShotAtPower / shotAtCount;
    }


    /** Increment the bullet hit count.  Keep track of bullet power used. */
    public void directHit(double power) {

        double damage = BotMath.calcBulletDamage(power);

        hitCount++;
        totHitPower    += power;
        maxHitPower     = BotMath.max(maxHitPower, power);
        avgHitPower     = totHitPower / hitCount;
        prevHitDamage   = damage;
        prevHitTime     = myRobot.getTime();
        if (energy > -1) energy -= damage;
        if (shotAtCount > 0) {
            hitPct = (double)hitCount / shotAtCount * 100;
        }
    }


    /**
     * Increment the shot me count; keep track of bullet power used and
     * update this robot's energy level.
     */
    public void   shotMe(double power) {

        shotMeCount++;
        totShotMePower += power;
        maxShotMePower  = BotMath.max(maxShotMePower, power);
        avgShotMePower  = totShotMePower / shotMeCount;
        energy         += power * BULLET_HIT_REWARD;
        lastShotMeTime  = myRobot.getTime();
    }

    /**
     * Estimate this robot's x,y position at a given point in time based
     * on its last recorded movement.
     * @param time point in time
     * @return x,y coordinates as a Point object.
     */
    public Point estimatePositionAt(long time) {

        Point  p    = new Point(x, y);
        double s    = speed;
        double h    = heading;
        double a    = BotMath.limit(
                      chgSpeed,
                      ROBOT_DEACCELERATION,
                      ROBOT_ACCELERATION);

        double t    = chgHeading;
        long   i    = lastUpdate;
        double minX = BattleField.getMinX();
        double minY = BattleField.getMinY();
        double maxX = BattleField.getMaxX();
        double maxY = BattleField.getMaxY();

        while (s != 0.0 && i < time &&
               p.x >= minX && p.y >= minY  &&
               p.x <= maxX && p.y <= maxY) {

            i++;

            /* Turning rate varies with speed. */
            if (t != 0.0) {
                h += (BotMath.calcTurnRate(s) * ((t < 0.0) ? -1 : +1));
            }

            s = BotMath.limit(s + a, 0, MAX_SPEED);
            p.x = BotMath.calcX(p.x, h, s);
            p.y = BotMath.calcY(p.y, h, s);
        }

        return p;
    }


    /**
     * Estimate this robot's distance at a given point in time based
     * on its last recorded movement.
     *
     * @param time point in time.
     * @return distance at the given point in time.
     */
    public double estimateDistanceAt(long time) {

        Point p = estimatePositionAt(time);
        return BotMath.calcDistance(
                myRobot.getX(),
                myRobot.getY(),
                p.x, p.y);
    }


    /**
     * Estimate this robot's bearing at a given point in time based on
     * its last recorded movement.
     * @param time point in time.
     * @return bearing in degrees at given point in time.
     */
    public double estimateBearingAt(long time) {

        Point p = estimatePositionAt(time);
        return BotMath.calcHeading(
                myRobot.getX(),
                myRobot.getY(),
                p.x, p.y);
    }


    //*************************************************************************
    //  PUBLIC UPDATE METHODS
    //*************************************************************************

    /**
     * Update this robot using radar scan data.
     */
    public void update(ScannedRobotEvent e) {

        updateCount++;
        energy = e.getEnergy();
        updateXYFields(e);
        updateSpeedFields(e);
        updateTopSpeedFields(e);
        updatePivotSpeedFields(e);
        updateHeadingFields(e);
        updateForwardBackwardFields(e);
        updateMoveFields(e);
        updateDistanceFields(e);
        timePlayed += (e.getTime() - lastUpdate);
        lastUpdate = e.getTime();
//        if (DEBUG) myRobot.out.println("UPDATED: " + this);
    }


    /**
     * Flag a robot as dead.
     */
    public void updateDead() {
        energy = -1;
        diedBeforeMeCount++;
        diedBeforeMePct = (double)diedBeforeMeCount / roundsPlayed * 100;
    }


    /**
     * Update this robot's reaction to being fired at.
     */
    public void updateReactions(double heading,
                                double bearing,
                                double speed,
                                double acceleration,
                                double distance,
                                Reaction reaction) {

        pr.update(heading, bearing, speed, acceleration, distance, reaction);
    }

    //*************************************************************************
    //  PRIVATE HELPER UPDATE METHODS
    //*************************************************************************

    /**
     * Update fields related to position (x,y coordinates).
     */
    private void updateXYFields(ScannedRobotEvent e) {

        prevBearing  = bearing;
        bearing      = BotMath.zeroTo360(myRobot.getHeading() + e.getBearing());
        prevDistance = distance;
        distance     = e.getDistance();
        prevX        = x;
        prevY        = y;
        x            = myRobot.getX() + BotMath.sin(bearing) * distance;
        y            = myRobot.getY() + BotMath.cos(bearing) * distance;
    }


    /**
     * Update fields related to this robot's speed.
     */
    private void updateSpeedFields(ScannedRobotEvent e) {

        prevGameSpeed   = gameSpeed;
        gameSpeed       = e.getVelocity();
        prevSpeed       = speed;
        speed           = BotMath.abs(gameSpeed);
        prevChgSpeed    = chgSpeed;
        chgSpeed        = (speed - prevSpeed) / (e.getTime() - lastUpdate);
        if (prevGameSpeed * gameSpeed < 0) chgSpeed = ROBOT_ACCELERATION;
        maxSpeed        = BotMath.max(maxSpeed, speed);
        totSpeed       += speed;
        totSqSpeed     += (speed * speed);
        avgSpeed        = totSpeed / updateCount;
        sdSpeed         = BotMath.sqrt(
                        (totSqSpeed -
                        (totSpeed * totSpeed / updateCount))
                        /updateCount);

        if (chgSpeed > 0.0 && chgSpeed <= ROBOT_ACCELERATION) {
            accelerationCount++;
            totAcceleration += chgSpeed;
            avgAcceleration = totAcceleration / accelerationCount;
        } else if (chgSpeed < 0.0 && chgSpeed >= ROBOT_DEACCELERATION) {
            deaccelerationCount++;
            totDeacceleration += chgSpeed;
            avgDeacceleration = totDeacceleration / deaccelerationCount;
        }

        if ((speed == 0.0 && prevSpeed != 0.0) ||
            (gameSpeed * prevGameSpeed < 0.0)) {
            totStops++;
            avgStopsPer100Turns = (double)totStops
                    / (((timePlayed + e.getTime())) / 100);
        }
    }


    /**
     * Update fields that keep track of how much time and how much distance
     * is covered by this robot at maximum speed.
     */
    private void updateTopSpeedFields(ScannedRobotEvent e) {

        if (speed >= 7 && prevSpeed < 7) {

            startTimeTopSpeed   = e.getTime();
            startXTopSpeed      = x;
            startYTopSpeed      = y;

        } else if (speed < 7        &&
                   prevSpeed >= 7   &&
                   startTimeTopSpeed > 0)   {

            topSpeedCount++;
            lastTimeTopSpeed = lastUpdate;

            long    t = e.getTime() - startTimeTopSpeed;
            avgLast2TimesTopSpeed = (t + prevTimeTopSpeed) / 2;
            prevTimeTopSpeed = t;

            double  d = BotMath.sqrt(
                        BotMath.pow(startXTopSpeed - x, 2) +
                        BotMath.pow(startYTopSpeed - y, 2));

            totTimeTopSpeed        += t;
            totSqTimeTopSpeed      += (t * t);
            maxTimeTopSpeed         = (long)BotMath.max(maxTimeTopSpeed, t);
            avgTimeTopSpeed         = totTimeTopSpeed / topSpeedCount;
            sdTimeTopSpeed          = (long)BotMath.sqrt(
                                    (totSqTimeTopSpeed -
                                    (totTimeTopSpeed
                                    * totTimeTopSpeed
                                    / topSpeedCount))
                                    / topSpeedCount);

            totDistTopSpeed        += d;
            totSqDistTopSpeed      += (d * d);
            maxDistTopSpeed         = BotMath.max(maxDistTopSpeed, d);
            avgDistTopSpeed         = totDistTopSpeed / topSpeedCount;
            sdDistTopSpeed          = BotMath.sqrt(
                                    (totSqDistTopSpeed -
                                    (totDistTopSpeed
                                    * totDistTopSpeed
                                    / topSpeedCount))
                                    / topSpeedCount);

            startTimeTopSpeed       = 0;
            startXTopSpeed          = 0.0;
            startYTopSpeed          = 0.0;
        }
    }


    /**
     * Update fields related to pivot speed.  Some robots make themselves
     * harder to hit by varying their speed.  The average speed at which
     * they change from deacceleration to acceleration is referred to as
     * the pivotSpeed.  This update method calculates the pivot speed and
     * pivot indicator.  The higher the pivot indicator is, the more often
     * this robot varies its speed.
     */
    private void updatePivotSpeedFields(ScannedRobotEvent e) {

        if (chgSpeed            >= 0.0          &&
            prevChgSpeed        <  0.0          &&
            chgSpeed            != prevChgSpeed &&
            speed * prevSpeed   != 0.0          &&
            prevSpeed           != MAX_SPEED    &&
            speed               != MAX_SPEED)   {

            pivotCount++;
            double ps       = BotMath.min(speed, prevSpeed);
            totPivotSpeed   += ps;
            totSqPivotSpeed += (ps * ps);
            pivotIndicator  = (moveCount != 0)
                    ? (double)pivotCount / moveCount
                    : 0;

            pivotSpeed      = totPivotSpeed / pivotCount;
            sdPivotSpeed    = BotMath.sqrt(
                            (totSqPivotSpeed -
                            (totPivotSpeed * totPivotSpeed / pivotCount))
                            / pivotCount);
        }
    }


    /**
     * Update fields related to heading.
     */
    private void updateHeadingFields(ScannedRobotEvent e) {

        prevHeading     = heading;
        prevGameHeading = gameHeading;
        gameHeading     = e.getHeading();

        heading = (gameSpeed < 0 ?
                  (gameHeading + A_180) % A_360 :
                   gameHeading);

        chgHeading =
                BotMath.plusMinus180(
                prevGameHeading, gameHeading) /
                (e.getTime() - lastUpdate);

        double aoa =
                BotMath.abs(
                BotMath.plusMinus180(
                heading, bearing));

        if (aoa > 90) aoa = 180 - aoa;

        totAngleOfApproach   += aoa;
        totSqAngleOfApproach += aoa * aoa;

        avgAngleOfApproach = totAngleOfApproach / updateCount;
        sdAngleOfApproach  = BotMath.sqrt(
                            (totSqAngleOfApproach -
                            (totAngleOfApproach *
                            totAngleOfApproach
                            / updateCount))
                            / updateCount);
    }


    /**
     * Update fields that track forward versus backward movement.
     */
    private void updateForwardBackwardFields(ScannedRobotEvent e) {

        moveForwardCount       += (gameSpeed > 0) ? 1 : 0;
        moveBackwardCount      += (gameSpeed < 0) ? 1 : 0;

        if (moveForwardCount > 0 && moveBackwardCount > 0) {
            if (moveForwardCount > moveBackwardCount) {
                biDirectionalPct =
                        (double)moveBackwardCount
                        / moveForwardCount
                        * 100;
            } else {
                biDirectionalPct =
                        (double)moveForwardCount
                        / moveBackwardCount
                        * 100;
            }
        }
    }


    /**
     * Update fields associated with movement.
     * Move distances are measured two ways:<p><p>
     *
     *  1) EST - Estimated.<p>
     *     The total distance travelled between two stop
     *     points is approximated by breaking up the robots
     *     path into short lines which are then added together
     *     to estimate the total distance travelled.<p><p>
     *
     *  2) LOS - Line of site.<p>
     *     The distance between two stop points is measured
     *     as a straight line between the two points, even if
     *     this bot did not follow a straight path.
     *
     */
    private void updateMoveFields(ScannedRobotEvent e) {

        if (prevStoppedX != 0) {

            currEstMove += BotMath.calcDistance(prevX, prevY, x, y);
            currLosMove  = BotMath.calcDistance(prevStoppedX, prevStoppedY, x, y);

            /*
             * If this robot has just finished it's move,
             * then update move fields accordingly.
             */

            if ((gameSpeed == 0.0 && prevGameSpeed != 0.0) ||
                (gameSpeed  > 0.0 && prevGameSpeed  < 0.0) ||
                (gameSpeed  < 0.0 && prevGameSpeed  > 0.0)) {

                updateFinishedMove();

            /*
             * If the robot slows down to almost zero and then speeds up
             * again, this will also be treated as a finished move.
             */
            } else if (
                    chgSpeed            >= 0.0              &&
                    prevChgSpeed        <  0.0              &&
                    chgSpeed            != prevChgSpeed     &&
                    speed * prevSpeed   != 0.0              &&
                    prevSpeed           != MAX_SPEED        &&
                    speed               != MAX_SPEED        &&
                    (e.getTime() - lastTimeTopSpeed < 6)    &&
                    (speed <= 3 || prevSpeed <= 3))   {

                updateFinishedMove();
            }

        } else {
            prevStoppedX = x;
            prevStoppedY = y;
        }
    }


    /**
     * Update fields associated with movement based on move the
     * robot just finished executing.
     */
    private void updateFinishedMove() {

        /*
         * If the robot is making very short moves back and forth, ignore
         * these movements, as they will throw off our statistics.
         */

//        if (DEBUG) myRobot.out.println(name + " moved " + currEstMove + " pixels.");

        if (currEstMove > 5) {
            moveCount++;
            updateLOSMove();
            updateESTMove();
            updateSideToSide();
            updateSameDirectionFields();

        } else {

            wiggleCount++;
            wigglePct = (double)wiggleCount / (moveCount + wiggleCount) * 100;
            currEstMove = 0.0;
            currLosMove = 0.0;
        }

        prevStoppedX = x;
        prevStoppedY = y;
    }


    /**
     * Update the line of site (LOS) distance moved.
     */
    private void updateLOSMove() {

        totLosMove     += currLosMove;
        totSqLosMove   += (currLosMove * currLosMove);
        maxLosMove      = BotMath.max(maxLosMove, currLosMove);
        avgLast2LosMove = (prevLosMove + currLosMove) / 2;
        prevLosMove     = currLosMove;
        currLosMove     = 0.0;
        avgLosMove      = totLosMove / moveCount;
        sdLosMove       = BotMath.sqrt(
                          (totSqLosMove -
                          (totLosMove * totLosMove / moveCount))
                          / moveCount);
    }


    /**
     * Update the estimate (EST) distanced moved.
     */
    private void updateESTMove() {

        totEstMove     += currEstMove;
        totSqEstMove   += (currEstMove * currEstMove);
        maxEstMove      = BotMath.max(maxEstMove, currEstMove);
        avgLast2EstMove = (prevEstMove + currEstMove) / 2;
        prevEstMove     = currEstMove;
        currEstMove     = 0.0;
        avgEstMove      = totEstMove / moveCount;
        sdEstMove       = BotMath.sqrt(
                          (totSqEstMove -
                          (totEstMove * totEstMove / moveCount))
                          / moveCount);
    }


    /**
     * Update fields that track side-to-side movement.
     * If the general heading of this robot is about 180 degrees
     * from the previous general heading (give or take 20 degrees),
     * then it has just zig-zagged back and forth.  Update the
     * side to side count and percentage accordingly.
     */
    private void updateSideToSide() {

        double newDirection;
        double angle;

        newDirection = BotMath.calcHeading(
                prevStoppedX, prevStoppedY, x, y);

        if (prevDirection != Double.MAX_VALUE) {

            angle = BotMath.abs(
                    BotMath.plusMinus180(
                    prevDirection, newDirection));

            if (angle > 160) {
                sideToSideCount++;
                sideToSidePct =
                (double)sideToSideCount
                / moveCount * 100;
            }
        }

        prevDirection = newDirection;
    }


    private void updateSameDirectionFields() {

        if (prevDriveDirection * prevGameSpeed > 0) {
            sameDirectionCount++;
            sameDirectionPct = (double)sameDirectionCount / moveCount * 100;
        }
        prevDriveDirection = (prevGameSpeed < 0) ? REVERSE : FORWARD;
    }

    /**
     * Update fields related to distance.  Two distances are monitored:
     * the distance from walls and the distance from corners.
     */
    private void updateDistanceFields(ScannedRobotEvent e) {

        double minDist;
        double ne;
        double se;
        double sw;
        double nw;

        minDist = BotMath.min(x, y);
        minDist = BotMath.min(minDist, BattleField.getWidth() - x);
        minDist = BotMath.min(minDist, BattleField.getHeight() - y);

        totDistToWall += minDist;
        avgDistToWall = totDistToWall / updateCount;
        minDistToWall = BotMath.min(minDistToWall, minDist);

        ne = BattleField.distanceToCorner(x, y, NORTHEAST_CORNER);
        se = BattleField.distanceToCorner(x, y, SOUTHEAST_CORNER);
        sw = BattleField.distanceToCorner(x, y, SOUTHWEST_CORNER);
        nw = BattleField.distanceToCorner(x, y, NORTHWEST_CORNER);

        minDist = Math.min(ne, Math.min(se, Math.min(sw, nw)));
        totDistToCorner += minDist;
        avgDistToCorner = totDistToCorner / updateCount;
        minDistToCorner = BotMath.min(minDistToCorner, minDist);

        if (BattleField.isInCorner(x, y)) {
            inCornerCount++;
            inCornerPct = (double) inCornerCount / updateCount * 100;
        }
    }

    //*************************************************************************
    //  FILE IO METHODS
    //*************************************************************************

    /**
     * Read previous data on this robot from file.
     */
    public void readFromFile() {

        try {

            File robotFile = myRobot.getDataFile(determineFileName());

            if (robotFile.length() > PreviousReactions.REACTION_FILE_SIZE) {

                DataInputStream in =
                        new DataInputStream(
                        new FileInputStream(
                        robotFile));

                readObject(in);
                in.close();
            }

        } catch (IOException e) {}
    }


    /**
     * Store data on this robot to file.
     */
    public void writeToFile() {

        try {

            DataOutputStream out =
                    new DataOutputStream(
                    new RobocodeFileOutputStream(
                    myRobot.getDataFile(determineFileName())));

            writeObject(out);
            out.close();

        } catch (IOException e) {}
    }


    /**
     * Determine the file name for this bot.  The file name will be the
     * same as the bot name, minus any copy number (ie: (1), (2) used
     * when there are multiple instances of the same bot) plus a ".dat"
     * extension.
     */
    private String determineFileName() {

        String fileName = name;
        int i = fileName.lastIndexOf(" (");
        if (i > -1) {
            int j = fileName.indexOf(')', i + 1);
            if ((j > i) && (j == fileName.length() - 1)) {
                fileName = fileName.substring(0, i);
            }
        }
        return fileName + ".dat";
    }


    /**
     * Write this bot object to file.
     *
     * NOTE: ANY CHANGES TO THIS METHOD MUST ALSO BE REFLECTED IN
     * THE readObject() METHOD BELOW.  THE FILE_VERSION CONSTANT
     * MUST ALSO BE UPDATED TO AVOID VERSIONING PROBLEMS AND THE
     * ROBOT_FILE_SIZE CONSTANT MUST ALSO BE UPDATED.
     */
    private void writeObject(DataOutputStream stream) throws IOException {

        stream.writeInt(FILE_VERSION);
        stream.writeDouble(maxSpeed);
        stream.writeDouble(totSpeed);
        stream.writeDouble(totSqSpeed);
        stream.writeInt(totStops);
        stream.writeDouble(totPivotSpeed);
        stream.writeDouble(totSqPivotSpeed);
        stream.writeInt(pivotCount);
        stream.writeDouble(pivotSpeed);
        stream.writeLong(totTimeTopSpeed);
        stream.writeLong(totSqTimeTopSpeed);
        stream.writeInt(topSpeedCount);
        stream.writeLong(maxTimeTopSpeed);
        stream.writeDouble(totDistTopSpeed);
        stream.writeDouble(totSqDistTopSpeed);
        stream.writeDouble(maxDistTopSpeed);
        stream.writeDouble(minDistToWall);
        stream.writeDouble(totDistToWall);
        stream.writeDouble(minDistToCorner);
        stream.writeDouble(totDistToCorner);
        stream.writeInt(inCornerCount);
        stream.writeInt(updateCount);
        stream.writeDouble(maxLosMove);
        stream.writeDouble(totLosMove);
        stream.writeDouble(totSqLosMove);
        stream.writeDouble(maxEstMove);
        stream.writeDouble(totEstMove);
        stream.writeDouble(totSqEstMove);
        stream.writeInt(shotMeCount);
        stream.writeDouble(totShotMePower);
        stream.writeDouble(maxShotMePower);
        stream.writeInt(shotAtCount);
        stream.writeDouble(totShotAtPower);
        stream.writeDouble(maxShotAtPower);
        stream.writeInt(hitCount);
        stream.writeDouble(hitPct);
        stream.writeDouble(totHitPower);
        stream.writeDouble(maxHitPower);
        stream.writeInt(rammedMeCount);
        stream.writeInt(diedBeforeMeCount);
        stream.writeInt(roundsPlayed);
        stream.writeLong(timePlayed);
        stream.writeInt(predictAimShotAtCount);
        stream.writeInt(predictAimHitCount);
        stream.writeInt(estimateAimShotAtCount);
        stream.writeInt(estimateAimHitCount);
        stream.writeInt(patternAimShotAtCount);
        stream.writeInt(patternAimHitCount);
        stream.writeInt(averageAimShotAtCount);
        stream.writeInt(averageAimHitCount);
        stream.writeInt(moveForwardCount);
        stream.writeInt(moveBackwardCount);
        stream.writeInt(directionChangeCount);
        stream.writeInt(sideToSideCount);
        stream.writeInt(moveCount);
        stream.writeInt(wiggleCount);
        stream.writeInt(sameDirectionCount);
        stream.writeDouble(totAngleOfApproach);
        stream.writeDouble(totSqAngleOfApproach);

        pr.writeObject(stream);
    }


    /**
     * Restore this bot's state from a previously written file.
     *
     * NOTE: ANY CHANGES TO THIS METHOD MUST ALSO BE REFLECTED IN
     * THE writeObject() METHOD ABOVE.  THE FILE_VERSION CONSTANT
     * MUST ALSO BE UPDATED TO AVOID VERSIONING PROBLEMS AND THE
     * ROBOT_FILE_SIZE CONSTANT MUST ALSO BE UPDATED.
     */
    private void readObject(DataInputStream stream) throws IOException {

        if (stream.readInt() == FILE_VERSION) {

            maxSpeed                = stream.readDouble();
            totSpeed                = stream.readDouble();
            totSqSpeed              = stream.readDouble();
            totStops                = stream.readInt();
            totPivotSpeed           = stream.readDouble();
            totSqPivotSpeed         = stream.readDouble();
            pivotCount              = stream.readInt();
            pivotSpeed              = stream.readDouble();
            totTimeTopSpeed         = stream.readLong();
            totSqTimeTopSpeed       = stream.readLong();
            topSpeedCount           = stream.readInt();
            maxTimeTopSpeed         = stream.readLong();
            totDistTopSpeed         = stream.readDouble();
            totSqDistTopSpeed       = stream.readDouble();
            maxDistTopSpeed         = stream.readDouble();
            minDistToWall           = stream.readDouble();
            totDistToWall           = stream.readDouble();
            minDistToCorner         = stream.readDouble();
            totDistToCorner         = stream.readDouble();
            inCornerCount           = stream.readInt();
            updateCount             = stream.readInt();
            maxLosMove              = stream.readDouble();
            totLosMove              = stream.readDouble();
            totSqLosMove            = stream.readDouble();
            maxEstMove              = stream.readDouble();
            totEstMove              = stream.readDouble();
            totSqEstMove            = stream.readDouble();
            shotMeCount             = stream.readInt();
            totShotMePower          = stream.readDouble();
            maxShotMePower          = stream.readDouble();
            shotAtCount             = stream.readInt();
            totShotAtPower          = stream.readDouble();
            maxShotAtPower          = stream.readDouble();
            hitCount                = stream.readInt();
            hitPct                  = stream.readDouble();
            totHitPower             = stream.readDouble();
            maxHitPower             = stream.readDouble();
            rammedMeCount           = stream.readInt();
            diedBeforeMeCount       = stream.readInt();
            roundsPlayed            = stream.readInt() + 1;
            timePlayed              = stream.readLong() + timePlayed;
            predictAimShotAtCount   = stream.readInt();
            predictAimHitCount      = stream.readInt() - 1;
            estimateAimShotAtCount  = stream.readInt();
            estimateAimHitCount     = stream.readInt() - 1;
            patternAimShotAtCount   = stream.readInt();
            patternAimHitCount      = stream.readInt() - 1;
            averageAimShotAtCount   = stream.readInt();
            averageAimHitCount      = stream.readInt() - 1;
            moveForwardCount        = stream.readInt();
            moveBackwardCount       = stream.readInt();
            directionChangeCount    = stream.readInt();
            sideToSideCount         = stream.readInt();
            moveCount               = stream.readInt();
            wiggleCount             = stream.readInt();
            sameDirectionCount      = stream.readInt();
            totAngleOfApproach      = stream.readDouble();
            totSqAngleOfApproach    = stream.readDouble();

            pr.readObject(stream);

            /*
             * Hit percentages must be updated immediately, since they control
             * which aiming methods are used.  Calling the xxxxxAimHit methods
             * will cause the percentages to be updated.  They also increment
             * the xxxxxxHitCount, which is why stream.readInt() - 1 is used
             * above when reading past values from file.
             */
            predictAimHit();
            estimateAimHit();
            patternAimHit();
            averageAimHit();
        }
    }

    //*************************************************************************
    //  DEBUGGING METHODS
    //*************************************************************************

    /** Returns statistics on this bot. */
    public String stats() {
        return                                                                "\n" +
        "name                   = " + name                                  + "\n" +
        "avgSpeed               = " + BotMath.round2(avgSpeed)              + "\n" +
        "avgAcceleration        = " + BotMath.round2(avgAcceleration)       + "\n" +
        "avgDeacceleration      = " + BotMath.round2(avgDeacceleration)     + "\n" +
        "sdSpeed                = " + BotMath.round2(sdSpeed)               + "\n" +
        "avgStopsPer100Turns    = " + BotMath.round2(avgStopsPer100Turns)   + "\n" +
        "pivotIndicator         = " + BotMath.round2(pivotIndicator)        + "\n" +
        "pivotSpeed             = " + BotMath.round2(pivotSpeed)            + "\n" +
        "sdPivotSpeed           = " + BotMath.round2(sdPivotSpeed)          + "\n" +
        "avgTimeTopSpeed        = " + BotMath.round2(avgTimeTopSpeed)       + "\n" +
        "sdTimeTopSpeed         = " + BotMath.round2(sdTimeTopSpeed)        + "\n" +
        "maxTimeTopSpeed        = " + BotMath.round2(maxTimeTopSpeed)       + "\n" +
        "avgDistTopSpeed        = " + BotMath.round2(avgDistTopSpeed)       + "\n" +
        "sdDistTopSpeed         = " + BotMath.round2(sdDistTopSpeed)        + "\n" +
        "maxDistTopSpeed        = " + BotMath.round2(maxDistTopSpeed)       + "\n" +
        "minDistToWall          = " + BotMath.round2(minDistToWall)         + "\n" +
        "avgDistToWall          = " + BotMath.round2(avgDistToWall)         + "\n" +
        "minDistToCorner        = " + BotMath.round2(minDistToCorner)       + "\n" +
        "avgDistToCorner        = " + BotMath.round2(avgDistToCorner)       + "\n" +
        "inCornerPct            = " + BotMath.round2(inCornerPct)           + "\n" +
        "maxLosMove             = " + BotMath.round2(maxLosMove)            + "\n" +
        "avgLosMove             = " + BotMath.round2(avgLosMove)            + "\n" +
        "sdLosMove              = " + BotMath.round2(sdLosMove)             + "\n" +
        "maxEstMove             = " + BotMath.round2(maxEstMove)            + "\n" +
        "avgEstMove             = " + BotMath.round2(avgEstMove)            + "\n" +
        "sdEstMove              = " + BotMath.round2(sdEstMove)             + "\n" +
        "wigglePct              = " + BotMath.round2(wigglePct)             + "\n" +
        "rammedMeCount          = " + rammedMeCount                         + "\n" +
        "diedBeforeMePct        = " + BotMath.round2(diedBeforeMePct)       + "\n" +
        "avgShotMePower         = " + BotMath.round2(avgShotMePower)        + "\n" +
        "shotMeCount            = " + shotMeCount                           + "\n" +
        "avgShotAtPower         = " + BotMath.round2(avgShotAtPower)        + "\n" +
        "avgHitPower            = " + BotMath.round2(avgHitPower)           + "\n" +
        "hitPct                 = " + BotMath.round2(hitPct)                + "\n" +
        "predictHitPct          = " + BotMath.round2(predictHitPct)         + "\n" +
        "estimateHitPct         = " + BotMath.round2(estimateHitPct)        + "\n" +
        "patternHitPct          = " + BotMath.round2(patternHitPct)         + "\n" +
        "averageHitPct          = " + BotMath.round2(averageHitPct)         + "\n" +
        "predictAimShotAtCount  = " + predictAimShotAtCount                 + "\n" +
        "estimateAimShotAtCount = " + estimateAimShotAtCount                + "\n" +
        "patternAimShotAtCount  = " + patternAimShotAtCount                 + "\n" +
        "averageAimShotAtCount  = " + averageAimShotAtCount                 + "\n" +
        "biDirectionalPct       = " + BotMath.round2(biDirectionalPct)      + "\n" +
        "sameDirectionPct       = " + BotMath.round2(sameDirectionPct)      + "\n" +
        "avgAngleOfApproach     = " + BotMath.round2(avgAngleOfApproach)    + "\n" +
        "sdAngleOfApproach      = " + BotMath.round2(sdAngleOfApproach)     + "\n" +
        "sideToSidePct          = " + BotMath.round2(sideToSidePct);
    }
}