package cs;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.PriorityQueue;
import java.util.Queue;

import robocode.AdvancedRobot;
import robocode.HitByBulletEvent;
import robocode.Rules;
import robocode.ScannedRobotEvent;
import robocode.util.Utils;

class State implements Comparable<State> {
	private static final double MAX_ANGLE_DIFF = Math.PI / 128.0;
	private static final double MAX_VELOCITY_DIFF = .125;
	
	double headingDelta;
	double velocityDelta;
	int depth;
	
	boolean matches(State state) {
		if(state == null || this instanceof Break || state instanceof Break) {
			return false;
		}
		return Math.abs(headingDelta - state.headingDelta) <= MAX_ANGLE_DIFF
			&& Math.abs(velocityDelta - state.velocityDelta) <= MAX_VELOCITY_DIFF;
	}
	
	public int compareTo(State o) {
		return (o.depth - depth);
//		return (depth - o.depth);
	}
}

class Break extends State { }

public class PumpkinPie extends AdvancedRobot {
	static final int MAX_PATTERN_LENGTH = 2000;
	static final int MIN_MATCH_LENGTH = 2;
	static final Rectangle2D.Double battlefield = new Rectangle2D.Double(17,17, 766, 566);
	static final ArrayDeque<State> history = new ArrayDeque<State>();
	
	private ArrayList<Point2D.Double> predicted = new ArrayList<Point2D.Double>();
	private double lastHeading, lastVelocity, eDistance;
	private double lastBulletPower = 0;
	
	
	///Raiko
	private static final double BEST_DISTANCE = 525;
	private static boolean flat = true;
	private static double lastReverseTime, circleDir = 1, enemyFirePower, numBadHits;
	
	//
	public Queue<State> getBestNextState(ArrayDeque<State> recent) {
		PriorityQueue<State> queue = new PriorityQueue<State>();
		ArrayDeque<State> hist = history.clone();
		
		/*
		 * If our history size is smaller than our best length we cannot
		 * possibly get a better match with the remaining history, so we don't
		 * try.
		 */
		while(hist.size() > 0) {
			Iterator<State> histIt = hist.descendingIterator();
			Iterator<State> recentIter = recent.descendingIterator();
			/* The next best state for this set of items. */
			State next = histIt.next();

			int length = 0;
			while(histIt.hasNext() && recentIter.hasNext()) {
				/* Match against our recent history. */
				State histState = histIt.next();
				State recentState = recentIter.next();

				/* If the states do not match exit the back track. */
				if(!histState.matches(recentState)) {
					break;
				}
				
				++length;
			}
			
			if(length > MIN_MATCH_LENGTH) {
				next.depth = length;
				queue.add(next);
			}
			
			hist.pollLast();
		}
		
		return queue;
	}
	
	public void addState(double headingDelta, double velocityDelta) {
		State state = new State();
		state.headingDelta = headingDelta;
		state.velocityDelta = velocityDelta;
		history.add(state);
	}
	
	public void addBreak() {
		history.add(new Break());
	}

	public void run() {
		//adjust for all the turns, by way of hierarchy
		//this adjusts the radar for robot turn aswell
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		
		//color is based on how much of a history we have :D
		setAllColors(Color.getHSBColor(0.1f, 1, 1));
		
		//Add a break to the pattern, so that we don't
		//track patterns that never happened
		addBreak();
		lastVelocity = 100;
		
		//spin the radar
		do {
			turnRadarRightRadians(Double.POSITIVE_INFINITY);
		} while(true);
	}
	
	public void onScannedRobot(ScannedRobotEvent e) {
		//This is the direct angle to the enemy
		double eAbsoluteBearing = getHeadingRadians() + e.getBearingRadians();
		if(getEnergy() < 0.05) {
			return;
		}
		
		//since this is one on one robot, reset the radar
		//we do this first as the lag may cause the robot
		//to skip a turn, this way we don't have broken scans
		
		//this is an advanced no-slip one on one radar
		double radar = Utils.normalRelativeAngle(eAbsoluteBearing - getRadarHeadingRadians());
		double arcToScan = Math.atan(36.0 / e.getDistance());
		radar += (radar < 0) ? -arcToScan : arcToScan;
		setTurnRadarRightRadians(radar);
		
		//Initialize some more variable
		Point2D.Double myLocation = new Point2D.Double(getX(), getY());
		Point2D.Double eLocation = project(myLocation,
				eAbsoluteBearing, eDistance = e.getDistance());
		
		if(lastVelocity == 100) {
			lastHeading = e.getHeadingRadians();
			lastVelocity = e.getVelocity();
			return;
		}
		
		double eHeading, eVelocity;
		//record the latest data into the pattern
		addState( (eHeading = e.getHeadingRadians()) - lastHeading,
				  (eVelocity = e.getVelocity()) - lastVelocity  );
		
		while(history.size() > MAX_PATTERN_LENGTH) {
			history.pop();
		}
		
		//update the last heading and velocity
		lastHeading = eHeading;
		lastVelocity = eVelocity;
		
		if (getGunHeat()/getGunCoolingRate() < 5 && history.size() > 3) {
			//determine the bullet power and speed
			double bulletPower = 0.1;
			if(eDistance < 750) bulletPower = 1.45;
			if(eDistance < 600) bulletPower = 1.95;
			if(eDistance < 150) bulletPower = 3;
			
			double bulletSpeed = Rules.getBulletSpeed(bulletPower);
			
			//predict forward
			ArrayDeque<State> future = history.clone();
			
			//State next = getBestNextState(future);
			Point2D.Double lastPosition = eLocation;
			double lastVelocity = eVelocity;
			double lastHeading = eHeading;
			
			int predictTime = -1;
			
			ArrayList<Point2D.Double> newPredicted = new ArrayList<Point2D.Double>();
			
			Queue<State> nextQueue = getBestNextState(future);
			while(nextQueue.size() > 0) {
				
				State next = nextQueue.poll();
				
				double nextHeading = Utils.normalRelativeAngle(lastHeading + next.headingDelta);
				double nextVelocity = lastVelocity + next.velocityDelta;
				if(nextVelocity < -Rules.MAX_VELOCITY || nextVelocity > Rules.MAX_VELOCITY) {
					continue;
				}
				
				Point2D.Double nextPosition = project(lastPosition, nextHeading, nextVelocity);
				
				/* 
				 * Since we discard any states that would lead the enemy to
				 * leave the battlefield, we gain super accurate wall collision
				 * pattern matching without having to do any extra work for it.
				 */
				if(battlefield.contains(nextPosition)
				&& predictTime < nextPosition.distance(myLocation)/bulletSpeed) {
					predictTime++;
					lastHeading = nextHeading;
					lastVelocity = nextVelocity;
					lastPosition = nextPosition;
					future.add(next);
					newPredicted.add(nextPosition);
					nextQueue = getBestNextState(future);
				}
				//otherwise try the next item in the queue
			}
			
			if(battlefield.contains(lastPosition)
				&& predictTime > lastPosition.distance(myLocation)/bulletSpeed) {
				//do targeting
				
				predicted.clear();
				predicted.addAll(newPredicted);

				double aimAngle = eAbsoluteBearing;
				if(lastPosition != null) {
					aimAngle = absoluteAngle(myLocation, lastPosition);
				}
				double angle = Utils.normalRelativeAngle(aimAngle - getGunHeadingRadians());
				setTurnGunRightRadians(angle);
				
				lastBulletPower = bulletPower;
			}
			
		}
		
		if(lastBulletPower == 0) {
			//Head on aiming
			setTurnGunRightRadians(Utils.normalRelativeAngle(eAbsoluteBearing - getGunHeadingRadians()));
		} else if(getGunTurnRemaining() < 0.01) {
			setFire(lastBulletPower);
			lastBulletPower = 0;
		}
		
		//////////////////// Raiko
		/*-------- setup data -----*/
		double theta;
		
		/* ---- Movement ---- */
		Point2D.Double newDestination;
		
		double distDelta = 0.02 + Math.PI/2 + (eDistance > BEST_DISTANCE  ? -.1 : .5);
		while (!battlefield.contains(newDestination =
			project(myLocation, eAbsoluteBearing + circleDir*(distDelta-=0.02), 170)));
		theta = 0.5952*(20D - 3D*enemyFirePower)/eDistance;
		if ( (flat && Math.random() > Math.pow(theta, theta))
				|| distDelta < Math.PI/5 ||
				(distDelta < Math.PI/3.5 && eDistance < 400) ){
			circleDir = -circleDir;
			lastReverseTime = getTime();
		}
		
		theta = absoluteAngle(myLocation, newDestination) - getHeadingRadians();
		setAhead(Math.cos(theta)*100);
		setTurnRightRadians(Math.tan(theta));
		
	}
	
	public void onPaint(Graphics2D g) {
		g.setColor(Color.white);
		g.drawString("Pattern Length: " + history.size(), 0, 20);
		
		
		
		for(int i = 0; i < predicted.size(); ++i) {
			Point2D.Double point = predicted.get(i);
			//lets say 0 to 1 for now
			g.setColor(Color.getHSBColor(0.5f - ((float)i / predicted.size()) / 2f, 1, 1));
			g.fillOval((int) point.x - 2, (int) point.y - 2, 4, 4);
		}
	}
	
    public void onHitByBullet(HitByBulletEvent e) {
		if ((double)(getTime() - lastReverseTime) >
			eDistance/e.getVelocity() && eDistance > 200 && !flat) { 
	    	flat = (++numBadHits/(getRoundNum()+1) > 1.1);
		}
    }
	
	public static final Point2D.Double project(Point2D.Double l, double a, double d){
		return new Point2D.Double(l.x + d*Math.sin(a), l.y + d*Math.cos(a));
	}
	
	public static final double absoluteAngle(Point2D source, Point2D target) {
		return Math.atan2(target.getX() - source.getX(), target.getY() - source.getY());
	}
}