package lucasslf.wavesurfing;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.Arc2D;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;

import lucasslf.LucasslfBot;
import lucasslf.utility.FastTrig;
import lucasslf.utility.Utils;
import lucasslf.wavesurfing.buffer.SegmentedBuffer;
import robocode.AdvancedRobot;
import robocode.Bullet;
import robocode.Condition;

public class EnemyWave extends Condition {
	// Just to keep track of the waves we fired
	private static int WAVE_IDS = 0;

	// Rolling Avgs
	private static int readings;

	private int id = WAVE_IDS++;
	private Point2D initialPosition;

	private double initialAngle;
	private int direction;
	private long fireTime;
	private double bulletPower;
	private double initialDistance;
	private double traveledDistance;
	private double enemyVelocity;
	private double myLateralVelocity;
	private Point2D myInitialPosition;
	private boolean virtual;
	private double myVelocity;
	private double myVelocityBefore;
	private double headingDifferenceBefore;
	private double headingDifference;
	private double advancingVelocity;
	private double deltaBearingBefore;
	private double deltaBearing;
	private int segMoveTimes;
	private LucasslfBot r;

	private List<Integer> shadowedBins = new ArrayList<Integer>();

	List<SegmentedBuffer> buffers = new ArrayList<SegmentedBuffer>();
	List<SegmentedBuffer> virtualBuffers = new ArrayList<SegmentedBuffer>();

	public EnemyWave(Point2D initialPosition, double initialAngle,
			long fireTime, double bulletPower, int direction,
			double initialDistance, double enemyVelocity,
			double myLateralVelocity, Point2D myInitialPosition,
			double myVelocity, double myVelocityBefore,
			double headingDifferenceBefore, double headingDifference,
			double advancingVelocity, double deltaBearingBefore,
			double deltaBearing, int segMoveTimes, boolean virtual,
			LucasslfBot r) {
		this.initialPosition = initialPosition;
		this.initialAngle = initialAngle;
		this.fireTime = fireTime;
		this.bulletPower = bulletPower;
		this.direction = direction;
		this.initialDistance = initialDistance;
		this.enemyVelocity = enemyVelocity;
		this.myInitialPosition = myInitialPosition;
		this.myVelocity = myVelocity;
		this.myVelocityBefore = myVelocityBefore;
		this.headingDifferenceBefore = headingDifferenceBefore;
		this.headingDifference = headingDifference;
		this.advancingVelocity = advancingVelocity;
		this.deltaBearing = deltaBearing;
		this.deltaBearingBefore = deltaBearingBefore;
		this.segMoveTimes = segMoveTimes;
		this.virtual = virtual;
		this.r = r;

		this.updateTravel();
	}

	public void addVirtualSegmentedBuffer(SegmentedBuffer segmentedBuffer) {
		virtualBuffers.add(segmentedBuffer);
	}

	public void addSegmentedBuffer(SegmentedBuffer segmentedBuffer) {
		buffers.add(segmentedBuffer);
	}

	public void addSegmentedBuffer(double[] buffer, double weight) {
		buffers.add(new SegmentedBuffer(buffer, weight));
	}

	// From Wiki. ---If target is accelerating, constant, or decelerating.---
	public int getSegAccel() {
		int delta = (int) (Math.round(5 * initialDistance
				* (Math.abs(deltaBearing) - Math.abs(deltaBearingBefore))));
		if (delta < 0) {
			return 0;
		} else if (delta > 0) {
			return 2;
		}
		return 1;
	}

	public int getSegMoveTimes() {
		return segMoveTimes;
	}

	// 0 - near wall
	// 1 - Corner
	// 2 - Middle
	// Didnt use fieldRect because I wanted a different margin
	public int getSegWallProximity() {
		int retorno = 2;
		double x = myInitialPosition.getX();
		double y = myInitialPosition.getY();
		boolean nearVerticalWalls = (x < 50 || (r.getBattleFieldWidth() - x < 50));
		boolean nearHorizontalWalls = (y < 50 || (r.getBattleFieldHeight() - y < 50));

		if (nearHorizontalWalls && nearVerticalWalls) {
			return 1;
		}
		if (nearVerticalWalls || nearHorizontalWalls)
			return 0;

		return retorno;
	}

	public int getSegHeadingDifference() {
		return (int) (headingDifferenceBefore + 360) / 180;
	}

	public int getSegBulletPower() {
		return (int) bulletPower;
	}

	public int getSegAdvancingVelocity() {
		return (int) (advancingVelocity / 4) + 2;
	}

	public int getSegMyVelocity() {
		return (int) (myVelocity / 4) + 2;
	}

	public int getSegMyVelocityBefore() {
		return (int) (myVelocityBefore / 4) + 2;
	}

	public int getSegLateralVelocity() {
		return (int) (myLateralVelocity / 4) + 2;
	}

	public int getSegInitialDistance() {
		return (int) initialDistance
				/ (1200 / (BasicWaveSurfer.SEG_DISTANCE - 1));
	}

	public double getEnemyVelocity() {
		return enemyVelocity;
	}

	public double getBulletSpeed() {
		return 20.0 - (bulletPower * 3.0);
	}

	public double getMaxEscapeAngle() {
		return FastTrig.asin(8.0 / getBulletSpeed());
	}

	public double getTraveledDistance() {
		return traveledDistance;
	}

	public Point2D getInitialPosition() {
		return initialPosition;
	}

	public double getInitialAngle() {
		return initialAngle;
	}

	public int getDirection() {
		return direction;
	}

	public void updateTravel() {
		this.traveledDistance += getBulletSpeed();
	}

	public void updateTraveledDistanceFrom(long time) {
		this.traveledDistance = (time - fireTime) * getBulletSpeed();
	}

	public double getInitialDistance() {
		return initialDistance;
	}

	public Point2D getMyInitialPosition() {
		return myInitialPosition;
	}

	@Override
	public String toString() {
		return "EnemyWave [id=" + id + ", initialPosition=" + initialPosition
				+ ", initialAngle=" + initialAngle + ", direction=" + direction
				+ ", fireTime=" + fireTime + ", bulletPower=" + bulletPower
				+ ", initialDistance=" + initialDistance
				+ ", traveledDistance=" + traveledDistance + ", enemyVelocity="
				+ enemyVelocity + ", myLateralVelocity=" + myLateralVelocity
				+ ", myInitialPosition=" + myInitialPosition + "]";
	}

	public void draw(Graphics2D g) {

		if (!virtual) {
			// Draw wave
			double raio = this.getTraveledDistance();
			double lado = raio * 2;
			double x = (initialPosition.getX() - raio);
			double y = (initialPosition.getY() - raio);
			double initialArcAngleDegrees = Utils.toDegrees(initialAngle
					- getMaxEscapeAngle()) - 90;
			double arcAngleDegrees = Utils.toDegrees(getMaxEscapeAngle() * 2
					/ BasicWaveSurfer.BINS);

			double maxFactor = getMaxFactorFromBuffers();

			if (direction > 0) {
				for (int i = 0; i < BasicWaveSurfer.BINS; i++) {
					double binDanger = getBinDanger(i);
					int colorSbtFactor = (int) (binDanger / maxFactor * 255);
					if (shadowedBins.contains(i))
						g.setColor(new Color(0, 0, 255));
					else
						g.setColor(new Color(255, 255 - colorSbtFactor,
								255 - colorSbtFactor));
					Arc2D arc = new Arc2D.Double(x, y, lado, lado,
							initialArcAngleDegrees, arcAngleDegrees, Arc2D.OPEN);
					g.draw(arc);
					if(binDanger>0){
						Point2D point = Utils.project(initialPosition, Utils.toRadians(initialArcAngleDegrees+90), raio);
						Ellipse2D e = new Ellipse2D.Double(point.getX()-5,point.getY()-5,10,10);
						if(maxFactor-binDanger<0.001){
							g.setColor(Color.red);
					 		g.draw(e);
						}else if(maxFactor-binDanger<0.01) {
							g.setColor(Color.yellow);
					 		g.draw(e);							
						}else if(maxFactor-binDanger<0.1) {
							g.setColor(Color.green);
					 		g.draw(e);							
						}
					}
					initialArcAngleDegrees += arcAngleDegrees;
				}
			} else {
				for (int i = BasicWaveSurfer.BINS - 1; i >= 0; i--) {
					double binDanger = getBinDanger(i);
					int colorSbtFactor = (int) (binDanger / maxFactor * 255);
					if (shadowedBins.contains(i))
						g.setColor(new Color(0, 0, 255));
					else
						g.setColor(new Color(255, 255 - colorSbtFactor,
								255 - colorSbtFactor));
					Arc2D arc = new Arc2D.Double(x, y, lado, lado,
							initialArcAngleDegrees, arcAngleDegrees, Arc2D.OPEN);
					g.draw(arc);
					
					if(binDanger>0){
						Point2D point = Utils.project(initialPosition, Utils.toRadians(initialArcAngleDegrees+90), raio);
						Ellipse2D e = new Ellipse2D.Double(point.getX()-5,point.getY()-5,10,10);
						if(maxFactor-binDanger<0.001){
							g.setColor(Color.red);
					 		g.draw(e);
						}else if(maxFactor-binDanger<0.01) {
							g.setColor(Color.yellow);
					 		g.draw(e);							
						}else if(maxFactor-binDanger<0.05) {
							g.setColor(Color.green);
					 		g.draw(e);							
						}
					}					
					initialArcAngleDegrees += arcAngleDegrees;
				}
			}
		}

	}

	public boolean isVirtual() {
		return virtual;
	}

	public double getMaxFactorFromVirtualBuffers() {
		double max = 0;

		for (int i = 0; i < BasicWaveSurfer.BINS; i++) {
			double danger = this.getVirtualBuffersBinDanger(i);
			if (max < danger)
				max = danger;
		}
		return max;

	}

	public double getMaxFactorFromBuffers() {
		double max = 0;

		for (int i = 0; i < BasicWaveSurfer.BINS; i++) {
			double danger = this.getBinDanger(i);
			if (max < danger)
				max = danger;
		}
		return max;

	}

	public List<Integer> getFactorIndexes(Point2D targetLocation,
			int ticksInTheFuture) {
		List<Arc2D> segments = getWaveSegments(ticksInTheFuture);
		List<Integer> retorno = new ArrayList<Integer>();
		int index = 0;
		Rectangle2D protectionRect = new Rectangle2D.Double(
				targetLocation.getX() - 30 / 2, targetLocation.getY() - 30 / 2,
				60, 60);
		for (Arc2D arc : segments) {
			if (arc.intersects(protectionRect)) {
				retorno.add(index);
			}
			index++;

		}

		return retorno;
	}

	public int getFactorIndex(Point2D targetLocation) {
		double offsetAngle = (Utils.absoluteBearing(this.getInitialPosition(),
				targetLocation) - this.getInitialAngle());

		double factor = robocode.util.Utils.normalRelativeAngle(offsetAngle
				* direction)
				/ this.getMaxEscapeAngle();
		return (int) Math.round((BasicWaveSurfer.BINS - 1) / 2 * (factor + 1));
	}

	private void logUsedOffset(Point2D position) {
		int index = getFactorIndex(position);

		for (SegmentedBuffer b : virtualBuffers) {
			b.updateBin(index, 1, 4, 1);
		}

	}

	public void logHit(Point2D targetLocation) {
		int index = getFactorIndex(targetLocation);
		for (SegmentedBuffer b : buffers) {
			b.updateBin(index, Math.min(400, readings++), 100, 1.0);
		}
	}

	private List<Arc2D> getWaveSegments(int ticksInTheFutre) {
		List<Arc2D> retorno = new ArrayList<Arc2D>();
		double radius = this.getTraveledDistance() + ticksInTheFutre
				* this.getBulletSpeed();
		double side = radius * 2;
		double x = (initialPosition.getX() - radius);
		double y = (initialPosition.getY() - radius);
		double initialArcAngleDegrees = Utils.toDegrees(initialAngle
				- (getMaxEscapeAngle() * direction)) - 90;
		double arcAngleDegrees = Utils.toDegrees(getMaxEscapeAngle() * 2
				/ BasicWaveSurfer.BINS);

		for (int i = 0; i < BasicWaveSurfer.BINS; i++) {

			Arc2D arc = new Arc2D.Double(x, y, side, side,
					initialArcAngleDegrees, arcAngleDegrees, Arc2D.OPEN);
			retorno.add(arc);
			initialArcAngleDegrees += arcAngleDegrees * direction;
		}
		return retorno;
	}

	public double getBinDanger(int i) {
		if (shadowedBins.contains(i))
			return 0;
		double weightedDangerSum = 0;
		double weightSum = 0;
		for (SegmentedBuffer b : buffers) {
			weightedDangerSum += b.getSegmentedBuffer()[i] * b.getWeight();
			weightSum += b.getWeight();
		}
		return weightedDangerSum / weightSum;
	}

	public double getVirtualBuffersBinDanger(int i) {
		double weightedDangerSum = 0;
		double weightSum = 0;
		for (SegmentedBuffer b : virtualBuffers) {
			weightedDangerSum += b.getSegmentedBuffer()[i] * b.getWeight();
			weightSum += b.getWeight();
		}
		return weightedDangerSum / weightSum;
	}

	public double getBulletPower() {
		return bulletPower;
	}

	public void updateShadows() {
		if (!virtual) {
			for (Bullet b : r.bullets) {
				if (b != null) {
					Point2D bulletLocation = new Point2D.Double(b.getX(),
							b.getY());
					Point2D nextBulletLocation = Utils.project(bulletLocation,
							b.getHeadingRadians(), b.getVelocity());
					double bulletDistance = bulletLocation
							.distance(initialPosition);
					double nextBulletDistance = nextBulletLocation
							.distance(initialPosition);

					if (bulletDistance > traveledDistance
							&& nextBulletDistance < traveledDistance
									+ getBulletSpeed()) {
						int shadowedIndex = getFactorIndex(bulletLocation);
						if (shadowedIndex >= 0
								&& shadowedIndex < BasicWaveSurfer.BINS) {
							shadowedBins.add(shadowedIndex);
						}
					}
				}
			}
		}
	}

	@Override
	public boolean test() {

		Point2D position = new Point2D.Double(r.getX(), r.getY());
		this.updateTravel();
		this.updateShadows();
		if (this.getTraveledDistance() > position.distance(this
				.getInitialPosition()) + 50.0) {
			this.logUsedOffset(position);
			return true;
		}
		return false;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + id;
		return result;
	}

	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		EnemyWave other = (EnemyWave) obj;
		if (id != other.id)
			return false;
		return true;
	}

}
