package rc.yoda.plugin.moves;

import robocode.*;
import java.awt.*;
import rc.yoda.utils.*;
import java.awt.geom.Point2D;

/** 
 * Ataru - by Robert Codd (Gorded) 
 *
 * This code is here by released under the RoboWiki Public Code Licence (RWPCL),
 * datailed on: http://robowiki.net/?RWPCL
 * (Basically it means you must keep the code public if you base any bot on it)
 *
 * Ataru.java : v1.0 -- 2007/05/12
 */


/**
 * Ataru is a Wave Surfing Movement for Yoda
 * that encapsulates the true surfing algorithm
 * 
 * @author Robert Codd
 * @version v1.0
 */
public class Ataru extends Surfer 
{	
	/**
	 * The number of wave in the air to surf at a time
	 */
    private static final int WAVES_TO_SURF = 2;

	/**
	 * The distance this robot should try to stay from the enemy
	 */
	private static final int DESIRED_DISTANCE = 650;
		
	/**
	 * The maximum angle this robot should dive when wall smoothing
	 */	
	private static final double	MAX_DIVE_ANGLE = Math.PI / (10 / 3);
	
	/**
	 * The bais to set for distancing this from from the enemy while surfing
	 */	
	private static final double	SURF_DISTANCE_FACTOR = 3.3;
											
	/**
	 * An array of position for where this bot will be when the next wave breaks
	 */
	private Point2D.Double[] surfLocations = new Point2D.Double[3];
	
	/**
	 * A reference to the closest wave in the air
	 */
	private Wave closestWave;
		
	/**
	 * The amount of ticks since this robot wall smoothed travelling left
	 */
	private int timeSinceLeftSmooth = 0;
	
	/**
	 * The amount of ticks since this robot wall smoothed travelling right
	 */	
	private int	timeSinceRightSmooth = 0;
	
	/**
	 * The orientation of this robot
	 */
	private int	orientation = 1;
		
	/**
	 * The amount of smooth danger there is in travelling left
	 */
	private double leftSmoothDanger = 0;
	
	/**
	 * The amount of smooth danger there is in travelling right
	 */
	private double rightSmoothDanger = 0;
	
	/**
	 * The energy level of our enemy
	 */
	private double enemyEnergy = 100.0;
		
	/**
	 * Constructor: What do we do when someone create an instance of Ataru
	 */
	public Ataru(AdvancedRobot robot) {
		super(robot);
		init(); 
	}	

	/**
	 * Event method called by this robot when it detects
	 * that the enemy fired a bullet
	 *
	 * @param double deltaEnergy the power of the bullet fired
	 */
	public void onRobotFire(double deltaEnergy) {
        super.onRobotFire(deltaEnergy);
	}
				
	/**
	 * Event method called by Robocode when this robot's scanner
	 * passes over another robot
	 * 
	 * @param ScannedRobotEvent information about the scanned robot
	 */
	public void onScannedRobot(ScannedRobotEvent e) {
		super.onScannedRobot(e);
		enemyEnergy = e.getEnergy();
        doSurf();
	}

	/**
	 * Event method called by Robocode when this robot
	 * gets hit by a bullet
	 * 
	 * @param HitByBulletEvent information about ther bullet
	 * that hit this robot
	 */
	public void onHitByBullet(HitByBulletEvent e) {
		super.onHitByBullet(e);
	}
	
	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired collides with a bullet fired by another robot
	 *
	 * @param BulletHitBulletEvent information about the bullets
	 */
	public void onBulletHitBullet(BulletHitBulletEvent e) {	
		super.onBulletHitBullet(e);
    }

	/**
	 * Event method called by Robocode when this robot is
	 * allowed to draw debugging graphics to the screen
	 *
	 * @param Graphics2D graphics that provides drawing method for painting
	 */	
	public void onPaint(Graphics2D g) {
		g.setColor(Color.white);
		g.draw(YUtils.battleField);
			
		if (closestWave == null) return;	
						
		for (int count = 0; count < surfLocations.length; count++)
		{
			g.setColor(new Color(Color.HSBtoRGB((count / (float)surfLocations.length), 1, 1)));
			g.fillOval((int)(surfLocations[count].x - 2),(int)(surfLocations[count].y - 2),4,4);
		}
		
		for (int count = 0; count < _Waves.size(); count++) 
		{
			Wave wave = (Wave)_Waves.get(count);
			double[] stats = getWaveStats(wave);
		
			double max = 0;
	        for (int bin = 0; bin < BINS; bin++) { max = (stats[bin] > max ? stats[bin] : max); }
					
			int color = 255 - (255 / Math.max(1, _Waves.size() - 1) * count);
			g.setColor(new Color(color, color, color));			
			g.drawOval((int)(wave.fireLocation.x - wave.distanceTraveled), (int)(wave.fireLocation.y - wave.distanceTraveled), (int)(wave.distanceTraveled * 2) - 1, (int)(wave.distanceTraveled * 2) - 1);
            for (int bin = 0; bin < BINS; bin++) 
			{
                double size = 3;
                Point2D.Double bullet = projectBin(wave, bin);
				g.setColor(new Color(Color.HSBtoRGB((float)((1 - (stats[bin] / max)) * 240) / 360,1,1)));
                g.fillOval((int)(bullet.x - size),(int)(bullet.y - size),(int)(size * 2),(int)(size * 2));
            }
        }
	}	

	/**
	 * initiates the needed attributes only on the first round
	 */
	private void init() {
		if (robot.getRoundNum() != 0) return;
		minimalSurfStats[BINS]++;
		for (int count = 0; count < BINS; count++)
		{
			minimalSurfStats[count] = YUtils.rollingAverage(minimalSurfStats[count], 
				Math.pow(0.65, Math.abs(MIDDLE_BIN - count)),Math.pow(minimalSurfStats[BINS], 2),minimalSurfStats[BINS]); 
		}		
	}

	/**
	 * Surfing algorithm to use when there is a wave in the air
	 * caculates the danger for travelling left, right and stopping
	 * and travels in the safest direction
	 */
    private void doSurf() {
        Wave surfWave = getClosestSurfableWaves();

        if ((closestWave = surfWave) == null) {
			nullSurf();
			return;
		}
	
		for (int count = 0; count < surfLocations.length; count++) { surfLocations[count] = predictPosition(surfWave, count - 1); }
			
		double dangerLeft = checkDanger(surfWave, surfLocations[0]) + smoothDanger(-1) * Math.pow(robotsLocation.distance(enemyLocation) / surfLocations[0].distance(enemyLocation), SURF_DISTANCE_FACTOR), 
        	dangerRight = checkDanger(surfWave, surfLocations[2]) + smoothDanger(1) * Math.pow(robotsLocation.distance(enemyLocation) / surfLocations[2].distance(enemyLocation), SURF_DISTANCE_FACTOR),
			dangerHere = checkDanger(surfWave, surfLocations[1]);

		if (dangerHere < dangerLeft && dangerHere < dangerRight) move(robot.getHeadingRadians() + (orientation * 0.01), 0);
		else if (dangerLeft < dangerRight) move(surfLocations[0], Laws.MAX_VELOCITY);
			else move(surfLocations[2], Laws.MAX_VELOCITY);
    }

	/**
	 * Surfing algorithm to use if there are no waves in the air
	 * caculates the danger for moving left and right and travels in the
	 * safest direction
	 */
	private void nullSurf() {		
		if (enemyEnergy == 0) { move(enemyLocation, Laws.MAX_VELOCITY); return; }
		
		for (int count = 0; count < surfLocations.length; count += 2) {	surfLocations[count] = predictPosition(count - 1); }
		
		double dangerLeft = smoothDanger(-1) * robotsLocation.distance(enemyLocation) / surfLocations[0].distance(enemyLocation),
			dangerRight = smoothDanger(1) * robotsLocation.distance(enemyLocation) / surfLocations[2].distance(enemyLocation);
		
        if (dangerLeft < dangerRight) move(surfLocations[0], Laws.MAX_VELOCITY);
		else move(surfLocations[2], Laws.MAX_VELOCITY);
	}			

	/**
	 * Uses the gnome sorting algorithm to sort the
	 * waves in order of flight time
	 */
	private void sortWaves() {
		int size = _Waves.size(), 
			count = 0;
		
		while (count < size) {
			if (count == 0 || ((Wave)_Waves.get(count-1)).flightTime(robotsLocation) <=
				 ((Wave)_Waves.get(count)).flightTime(robotsLocation)) count++;
			else 
			{
				Wave temp = (Wave)_Waves.get(count);
				_Waves.set(count,(Wave)_Waves.get(count-1));
				_Waves.set(--count,temp);
			}	 
		}
	}	

	/**
	 * Gets the enemy wave that will hit the robot first 
	 *
	 * @return the closest Enemy wave
	 */
    private Wave getClosestSurfableWaves() {
		double closestTime = 1000;
		Wave surfWave = null;

        for (int count = 0; count < _Waves.size(); count++) 
		{
       	    Wave wave = (Wave)_Waves.get(count);
           	double distance = robotsLocation.distance(wave.fireLocation) - wave.distanceTraveled;
			double flightTime = distance / wave.bulletVelocity;	

            if (distance > wave.bulletVelocity && flightTime < closestTime) 
			{
       	        surfWave = wave;
           	    closestTime = flightTime;
            }
		}	
		return surfWave;
    }						

	/**
	 * Calculates where the robot will in ten turns travelling
	 * in the specified direction
	 *
	 * @param the direction to predict for
	 * @return the location the robot will be in 10 turns
	 */
	private Point2D.Double predictPosition(int direction) {
        Wave fakeWave = new Wave();

		fakeWave.bulletVelocity = Laws.getBulletSpeed(3);
		fakeWave.fireLocation = (Point2D.Double)enemyLocation.clone();
        fakeWave.distanceTraveled = enemyLocation.distance(robotsLocation) -
			(Laws.getBulletSpeed(3) * 10);	
			
		return predictPosition(fakeWave,direction);
	}	

	/**
	 * Calculates where the robot will be when the enemy waves
	 * reaches him if he continues the surf the specified enemy wave 
	 * going in the specified direction
	 *
	 * @param the enemy wave to surf
	 * @param the direction to predict for
	 * @return the location the robot will be when the wave reaches him
	 * @credit ORIGINAL : BY rozu
	 */
    private Point2D.Double predictPosition(Wave surfWave, int direction) {
    	Point2D.Double predictedPosition = (Point2D.Double)robotsLocation.clone();
    	double moveAngle, moveDir;
		double predictedVelocity = robot.getVelocity(),
    		predictedHeading = robot.getHeadingRadians();
		
		if (direction == 0) return YUtils.project(predictedPosition, predictedHeading + (orientation * 0.01), YUtils.stopDistance(predictedVelocity) * orientation);

        int counter = 0; 
        boolean intercepted = false;
    	do {
    		moveAngle = YUtils.wallSmoothing(predictedPosition, YUtils.absoluteBearing(surfWave.fireLocation, predictedPosition) 
				+ (direction * subtleDistancing(predictedPosition.distance(surfWave.fireLocation))), direction) - predictedHeading;
    		
    		if ((moveDir= YUtils.sign(Math.cos(moveAngle))) < 0) moveAngle += Math.PI;

    		double maxTurning = Laws.getTurnRateRadians(predictedVelocity);
    		predictedHeading = YUtils.normalRelativeAngle(predictedHeading + YUtils.limit(-maxTurning, Math.tan(moveAngle), maxTurning));
    		predictedVelocity += (predictedVelocity * moveDir < 0 ? 2*moveDir : moveDir);
    		predictedVelocity = YUtils.limit(-8, predictedVelocity, 8);
    		predictedPosition = YUtils.project(predictedPosition, predictedHeading, predictedVelocity); 

            if (predictedPosition.distance(surfWave.fireLocation) < surfWave.distanceTraveled + (++counter * surfWave.bulletVelocity))
			{
                intercepted = true;
            }
    	} while(!intercepted && counter < 500);

    	return predictedPosition;
    }

	/**
	 * Checks the danger at a specified location on the given
	 * enemy wave
	 *
	 * @param the enemy wave to calculate danger on
	 * @param the location to check danger for
	 */
	private double checkDanger(Wave surfWave, Point2D.Double location) {
		double danger = 0;
		for (int count = Math.min(WAVES_TO_SURF, _Waves.size()) - 1; count >= 0; count--)
		{
			Wave wave = (Wave)_Waves.get(count);
			danger = YUtils.rollingAverage(danger, getWaveStats(wave)[getFactorIndex(wave, location)], 
				Math.min(WAVES_TO_SURF, _Waves.size()) - count, wave.bulletVelocity / (count + 1));
		}
		return danger;
	} 

	/**
	 * Calculates the smoothAngle for the specified direction and keeps
	 * a dampened rolling average of the danger
	 *
	 * @param the direction for which you want the smooth danger
	 * @return the current updated average for the specified direction
	 */
	private double smoothDanger(int direction) {
		double smooth = 0;
		Point2D.Double orbitCenter = enemyLocation;
		if (closestWave != null) orbitCenter = closestWave.fireLocation;
		switch (direction) {
			case -1 : smooth = Math.PI / 2 - YUtils.cosineLaw(surfLocations[0].distance(orbitCenter),surfLocations[0].distance(robotsLocation), robotsLocation.distance(orbitCenter));
				return leftSmoothDanger = YUtils.rollingAverage(leftSmoothDanger, Math.max(0, smooth - MAX_DIVE_ANGLE), Math.min(timeSinceLeftSmooth++,15), Math.min(timeSinceLeftSmooth,15));
			case 1 : smooth = Math.PI / 2 - YUtils.cosineLaw(surfLocations[2].distance(orbitCenter),surfLocations[2].distance(robotsLocation), robotsLocation.distance(orbitCenter));
				return rightSmoothDanger = YUtils.rollingAverage(rightSmoothDanger, Math.max(0, smooth - MAX_DIVE_ANGLE), Math.min(timeSinceRightSmooth++,15), Math.min(timeSinceRightSmooth,15));
		}
		return smooth;
	}

	/**
	 * Calculates what angle to move at 
	 * if subtling increasing distance
	 *
	 * @param the distance away from threat
	 * @return the angle at which to relative your bearing to him
	 */
	private double subtleDistancing(double distance) {
		double distanceOffset = distance - DESIRED_DISTANCE;
		return (Math.PI / 2) + (Math.pow(distanceOffset / DESIRED_DISTANCE, 2) * YUtils.sign(distanceOffset) * 1.2);
	}

	/**
	 * Move this robot to a specified location at
	 * specified velocity
	 *
	 * @param point at which to travel towards
 	 * @param velocity at which to travel
	 */
    private void move(Point2D.Double location, double velocity) {
        move(YUtils.absoluteBearing(robotsLocation, location), velocity);
    }

	/**
	 * Moves this robot in the specified angle at 
	 * the specified velocity 
	 *
	 * @param angle at which to travel
	 * @param velocity to travel at
	 */
	private void move(double angle, double velocity) {
		double oldHeading = robot.getHeadingRadians();			
		robot.setMaxVelocity(velocity);
		robot.setAhead((orientation = YUtils.backAsFrontDirection(angle, oldHeading)) * 100);
		if (YUtils.backAsFrontTurn(angle, oldHeading) != 0) { robot.setTurnRightRadians(YUtils.backAsFrontTurn(angle, oldHeading)); }
	}																
}																				