package jwst.DAD.Movement;

import robocode.*;
import robocode.util.Utils;

import java.awt.geom.*;     // for Point2D's
import java.util.ArrayList; // for collection of waves
import java.awt.*;

import jwst.DAD.utils.DADUtils;
import jwst.Enemies.*;

/***************************************************************************************************
 * A wave surfer...
 * Movement(One on One): WaveSurfing - it can easily be called the counter to Guess Factor
 * Targeting. The array used in Guess Factor targeting can be described as a graph that could
 * look something like a peak, and the robot uses that peak in the graph to fire at that 
 * certain location. WaveSurfing attempts to make this graph as even as possible for the other
 * robot so that they have no idea where to fire, and their targeting is skewed.
 * 
 * @author Jared Wong
 * @author Shawn Thomas
 * @period Period 2
 * @date June 10th, 2009
 * 
 * Credit: http://testwiki.roborumble.org/w/index.php?title=Wave_Surfing_Tutorial
 * Credit: http://testwiki.roborumble.org/w/index.php?title=BasicSurfer/Code
 ****************************************************************************************************/
public class WaveRider 
{
    // your robot, the enemy robot, and the robot width, given by robocode
    private static AdvancedRobot robot;
    private static AdvancedEnemy enemy;
    
    // given robocode fact
    private final static int ROBOT_WIDTH = 36;
    
    // basically the diagonal distance of the battle field for the purpose of distance segmentation
    private static double MAX_DISTANCE;
    private static int DISTANCE_INDEXES;

    // Indexes are 0&+-1, +-2&+-3, +-4&+-5, +-6&+-7, +-8
    private final static int VELOCITY_INDEXES = 5;
    // merely make the same number of previous velocity indexes as there are velocity indexes
    private final static int PREVIOUS_VELOCITY_INDEXES = VELOCITY_INDEXES;
    
    // 47 bins are used because the max escape angle possible. possible.
    // is calculated by the arcsine( 8 / ( 20 - 3 * 3 ) )
    // it is the biggest angle possible, using the maximum bullet power possible. 3.
    // the angle. in degrees. is 46.658241772778. i wanted to use 1 bin per 2 angles. 
    private final static int BINS = 47;
    private final static int MIDDLE_BIN = (BINS - 1) / 2;
    
    // measure calculated in radians given the maximum possible velocity of the enemy (8 pixel per turn)
    // and the slowest possible bullet speed
    private final static double MAX_ESC_ANGLE = 0.8143399421265254D;
    // calculate the radians measure of the amount of radians each bin encompasses
    private final static double BIN_WIDTH = MAX_ESC_ANGLE / MIDDLE_BIN;

    // create a rectangle2D object representing the robocode playing field
    private static Rectangle2D.Double fieldRect;
    // a wall stick is one that keeps our bot away from the walls when wall smoothing
    private final static double WALL_STICK = ROBOT_WIDTH * 4.5;
    
    // 4 dimensional array with distances, velocities, previous velocities and bins
    private static double[][][][] stats;
    // a stat array to refer to if the regular stats segment has not been visited before
    private static double fastStats[] = new double[BINS];

    // an array list to keep track of enemy waves
    private static ArrayList<EnemyWave> waves = new ArrayList<EnemyWave>();
    // Keep track of my directions (signs), bearings (in relation to the enemy),
    // distance (in relation to my enemy) and velocities
    private static ArrayList<Integer> signs = new ArrayList<Integer>();
    private static ArrayList<Double> bearings = new ArrayList<Double>();
    private static ArrayList<Double> distances = new ArrayList<Double>();
    private static ArrayList<Double> velocities = new ArrayList<Double>();

    // Direction I want to move in
    // not side to side (e.g. left or right)
    // rather forward or backward
    // 1 = forwards
    // -1 = backwards
    /**
     * move direction to be public in the event that we actually hit a wall
     * (which we won't)
     */
    public byte moveDir = 1;

    // merely for the enjoyment of printing out statistics on how much the enemy has hit us
    private int hits = 0;
    private int totalFired = 0;
    private int scanCount = 0;
    
    /**
     * A constructor for the wave rider
     * @param robot - an advancedRobot which will implement this
     */
    public WaveRider(AdvancedRobot robot)
    {
        // set out static reference to a robot, to a robot
        this.robot = robot;
        // make a field where the scanned robot could possibly be
        // taking into account that when a robot is scanned the thing returned is 
        // basically a point at the center of the robot. thus we go ahead and subtract
        // half of the robot width from each side of the robocode field
        fieldRect = new Rectangle2D.Double(ROBOT_WIDTH / 2, ROBOT_WIDTH / 2, 
                                           robot.getBattleFieldWidth() - ROBOT_WIDTH, 
                                           robot.getBattleFieldHeight() - ROBOT_WIDTH);
        // the maximum distance we can ever be from an enemy robot is, for all intents and
        // purposes, the distance from the origin to the battle field width and height respectively
        // although.. in order to calculate a precise maximum distance i would have to, again,
        // take into account the robot width
        MAX_DISTANCE = Point2D.distance( 0,0,
            robot.getBattleFieldWidth(), robot.getBattleFieldHeight() );
        // Each distance index is 100 pixels large because 100 is a cool number. its a square. 10^2 = 100
        DISTANCE_INDEXES = (int)( MAX_DISTANCE / 100 );
        // initialize the stats array given the number of distance indexes based on the
        // maximum distance in the field
        stats = new double[DISTANCE_INDEXES][VELOCITY_INDEXES][PREVIOUS_VELOCITY_INDEXES][BINS];
    }
    
    /**
     * the basic run method called to surf
     * @param passedEnemy - in a 1v1 situation an Advanced Enemy will be passed
     */
    public void run(AdvancedEnemy passedEnemy) 
    {
        // the sole purpose of the scan count is to hold initialize the passed enemy
        // if the enemy field is null
        scanCount++;
        if (scanCount == 1) 
        {
            enemy = passedEnemy;
            hits = 0;
            totalFired = 0;
        }
        // I don't like big numbers...in case the numbers wrap around
        if(scanCount >= 3)
            scanCount--;
        
        // add a sign integer based on my lateral velocity (from the enemy's point of view)
        // to do this, take the sign of my velocity (whether I'm heading relatively forwards
        // or backwards)(and the enemies bearing in relation to me Math.sin returns the 
        // opposite over hypotenuse (as in trig) which essentially returns whether the enemy
        // is to my left or right according to my current heading. seeing as this sin stuff
        // is all in relation to me, I must multiply by the sign of my velocity to get a true
        // direction of me in terms of the enemy and how they view me)
        signs.add(0, DADUtils.sign(robot.getVelocity() * Math.sin(enemy.bearing)));
        
        // add a bearing (from the enemy's point of view)(thus add the pi/2)
        bearings.add(0, robot.getHeadingRadians() + enemy.bearing + Math.PI);
        
        // add the distance I am away from the enemy robot which is the same distance that the
        // enemy robot is away from me
        distances.add(0, Point2D.distance(robot.getX(), robot.getY(), enemy.x, enemy.y));
        
        // add my velocity to the ArrayList velocities
        velocities.add(0, robot.getVelocity());
        
        // calculated fired bullet based on just received information on this scan                                                                                                                                                                                                                                                                     
        if (enemy.firedBullet() && signs.size() > 2) 
        {
            // for the pure amusement of seeing the enemies hit percentage
            totalFired++;
            System.out.println("Their hit percentage: " + (double)hits/(double)totalFired);
            
            // figure out the enemies fire power of their bullets based upon their energy change
            double firePower = enemy.getEnergyChange();
            
            // make a new enemy wave
            EnemyWave w = new EnemyWave();
            
            // The enemy fires the turn before I recognize that his energy has changed
            // therefore I set up lots of variables with previous knowledge
            w.setFireTime(enemy.lastUpdated - 1);
            w.setFiredLocation(new Point2D.Double( enemy.pX, enemy.pY ));
            w.setMyLocationAtFire(new Point2D.Double( robot.getX(),robot.getY()));
            
            // The direction they're using about me is from two ticks ago
            w.setSign(signs.get(2));
            
            // The bearing they're using about me is from two ticks ago
            w.setBearing(bearings.get(2));
            
            // Set the velocity based on their fire power they used
            w.setVelocity(DADUtils.bulletVelocity(firePower));
            
            // Because i recognize one tick after i can just initialize the distance
            // traveled to the bullet velocity(measured in pixels per turn)
            w.setDistanceTraveled(DADUtils.bulletVelocity(firePower));
            
            // set up a distance index based on my distance when the enemy fired 2 ticks ago
            int distanceIndex = (int) (distances.get(2) / (MAX_DISTANCE / DISTANCE_INDEXES));
            
            // set up a velocity index based upon on my velocity 2 ticks ago when he did fire
            int velocityIndex = (int) Math.abs(velocities.get(2) / 2);
            
            // set up a previous velocity index (comparative to when he fired) from 3 ticks ago
            int previousVelocityIndex = (int) Math.abs(velocities.get(3) / 2);
            w.setSegment(stats[distanceIndex][velocityIndex][previousVelocityIndex]);
            
            // Only add the wave if we're still active
            if (robot.getEnergy() >= 0.1)
                waves.add(w);
        }
        
        // Always update waves
        // Updates them based on the current time of the game
        updateWaves();
        
        // Surf them waves!
        doSurfing();
    }
    
    private void updateWaves() 
    {
        for (int i = 0; i < waves.size(); i++) 
        {
            EnemyWave w = waves.get(i);

            // Recalculate the distance traveled based on the time of the game
            // and when the wave was initiated
            w.setDistanceTraveled((robot.getTime() - w.getFireTime()) * w.getVelocity());
            
            // If the bullet wave has reached me and gone 50 pixels behind my center
            // then remove it.
            // The added 50 pixel buffer is in for the event that the robot gets hit at the
            // rear.  In which case if the buffer was not there, a hitByBulletEvent might
            // occur and I wouldn't be tracking it.
            if (w.getDistanceTraveled() > w.getFiredLocation().distance(robot.getX(), robot.getY()) + 50) 
            {
                waves.remove(i);
                i--;
            }
        }
    }
    
    private void doSurfing() 
    {
        // Find the closest wave
        EnemyWave w = getClosestWave();
        // Make sure it exists...
        if (w != null) 
        {
            // Simply check the danger of orbiting left and right when the wave arrives
            double dangerLeft = getDanger(w, -1);
            double dangerRight = getDanger(w, 1);

            Point2D.Double myLocation = new Point2D.Double(robot.getX(), robot.getY());
            double angle = DADUtils.absoluteBearing(w.getFiredLocation(), myLocation);
            // Choose left or right, whichever one is safer
            if (dangerLeft < dangerRight)
                angle = DADUtils.wallSmooth(myLocation, angle - Math.PI/2, -1, WALL_STICK, fieldRect);
            else
                angle = DADUtils.wallSmooth(myLocation, angle + Math.PI/2, 1,  WALL_STICK, fieldRect);

            // Get the actual angle to turn considering the current heading'
            angle = DADUtils.normalizeBearing(angle - robot.getHeadingRadians());
            // Check to see if the absolute value of the angle is greater than 90 degrees
            // If it is couldn't i just turn left or right a little and just move backwards?
            // Why don't I...
            if (Math.abs(angle) > Math.PI/2) 
            {
                if (angle < 0)
                    // if we really want to turn left more than 90 degrees
                    // just turn right a little and then move backwards. much faster
                    robot.setTurnRightRadians(angle + Math.PI);
                else
                    // if we really want to turn right more than 90 degrees
                    // just turn left a little and then move backwards. much faster
                    robot.setTurnLeftRadians(angle - Math.PI);
                // Set your direction as backwards of course
                moveDir = -1;
            }
            else 
            {
                robot.setTurnRightRadians(angle);
                // Set your direction as forwards
                moveDir = 1;
            }
            // MOVE!
            // 100 is just some arbitrarily big number * the direction to move
            // in the event that we actually want to move backwards
            robot.setAhead(100 * moveDir);
        }
    }
    
    /**
     * Loops through an array (segment) to see if it is empty
     * @param array (segment) to check
     * @return true if it's empty, false if it's not
     */
    private boolean isEmpty(double[] segment) 
    {
        // go through all the bins in the array
        for (int i = 0; i < segment.length; i++)
            // if EVER it does not equal 0
            // return false
            if (segment[i] != 0D)
                return false;
        // else return true
        return true;
    }

    
    private EnemyWave getClosestWave() 
    {
        // Assume its the farthest wave from you possible...
        double distance = Double.POSITIVE_INFINITY;
        EnemyWave wave = null;

        // Loop through your array list of waves
        for (int i = 0; i < waves.size(); i++) 
        {
            // get the wave you are looking at currently and its distance
            EnemyWave w = waves.get(i);
            // take the distance the enemy robot is from you
            // and subtract the distance the wave has traveled so far to get
            // how close it is to me
            double dist = w.getFiredLocation().distance(robot.getX(), robot.getY()) - w.getDistanceTraveled();

            // make sure the wave is in front of us
            // then check if the distance of the current wave is less than that
            // of the old distance
            // if so, keep the current wave
            if (distance > w.getVelocity() && dist < distance)
            {
                wave = w;
                distance = dist;
            }
        }
        // return that closest wave
        return wave;
    }
    
    /**
     * Gets the danger of the index we will be at
     * 
     * @param w wave to check the danger of
     * @param sign direction we're moving
     * @return visit count of the bin we'll be at
     */
    private double getDanger(EnemyWave w, int sign) 
    {
        Point2D.Double futureLocation = DADUtils.predictFutureLocation(w, sign, robot, WALL_STICK, fieldRect);

        // Find bin of the predicted location
        int bin = (int) Math.round((DADUtils.normalizeBearing(
        // Absolute bearing to me minus original absolute bearing
            DADUtils.absoluteBearing(w.getFiredLocation(), futureLocation) - w.getBearing()))
        // divided by the size of each bin, but not always positive
                / (w.getSign() * BIN_WIDTH))
        // And put it back in the array
                + MIDDLE_BIN;

        // Limit my bins to the actual number of bins
        bin = (int) DADUtils.limit(0, BINS - 1, bin);
        
        // Use the fast stats if the current segment is empty
        if (!isEmpty(w.getSegment()))
            return w.getSegment()[bin];
        return fastStats[bin];
    }
    
    
    ///////////////////////////////////////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////////////////////////
    ///////////////////////////////////////////////////////////////////////////////////////////

    /**
     * log a hit by bullet event
     * WE work on the thinking that 
     * you hit us once from that angle, NEVER AGAIN will you hit us there again
     * So we only log a stuff in our stats array when the enemy actually hits us
     * Otherwise, what are we doing wrong?
     * @param e - HitByBulletEvent
     */
    public void logHitByBullet(HitByBulletEvent e) 
    {
        hits++;
        // Just make sure there are some waves
        if (!waves.isEmpty()) 
        {
            // get the location where the bullet hit me
            Point2D.Double hitLocation = 
                new Point2D.Double(e.getBullet().getX(), e.getBullet().getY());
            // initiate the hit wave
            EnemyWave hitWave = null;

            // look through the waves, and find one that probably hit us.
            for (int i = 0; i < waves.size(); i++) 
            {
                // set the gotten wave to an ambiguous variable EnemyWave w
                EnemyWave w = waves.get(i);
                
                if (// first check to see if the distance the bullet wave has fired
                    // has reached our location + 50
                    //              the 50 is checked for because the robot could hit the wave
                    //              towards the rear
                    Math.abs(w.getDistanceTraveled() - 
                             w.getFiredLocation().distance(robot.getX(), robot.getY())) < 50
                    // make sure the bullet really is from the wave we are looking at
                    && Math.round(DADUtils.bulletVelocity(e.getBullet().getPower()) * 10) == Math
                                .round(w.getVelocity() * 10)) 
                {
                    // if the wave really is the bullet wave that just hit us.
                    // set it set the hit wave and break from the loop.
                    // as it is unnecessary to keep on going through
                    hitWave = w;
                    break;
                }
            }

            // make sure the wave is not a null wave
            if (hitWave != null) 
            {
                // Find bin of the hit location
                int bin = (int) Math.round((DADUtils.normalizeBearing(
                // Absolute bearing to me minus original absolute bearing
                    DADUtils.absoluteBearing(hitWave.getFiredLocation(), hitLocation)
                                - hitWave.getBearing()))
                        // divided by the size of each bin, but not always
                        // positive
                        / (hitWave.getSign() * BIN_WIDTH))
                        // And put it back in the array as it could currently be very much out of it
                        + MIDDLE_BIN;

                // loop through all the bins
                for (int i = 0; i < BINS; i++) 
                {
                    // Bin smoothing using inverse power
                    //regular segments
                    hitWave.getSegment()[i] += 1D / (Math.pow(bin - i, 2) + 1);
                    //fast segments
                    fastStats[i] += 1D / (Math.pow(bin - i, 2) + 1);
                }

                // We can remove this wave now, of course.
                waves.remove(waves.lastIndexOf(hitWave));
            }
        }
    }

    
    /**
     * on my death, reset the scan count
     * @param e - DeathEvent
     */
    public void onDeath(DeathEvent e)
    {
        scanCount = 0;
    }
    
    /**
     * if i win, reset the scan count
     * @param e - WinEvent
     */
    public void onWin(WinEvent e)
    {
        scanCount = 0;
    }
    
    /**
     * onPaint method of WaveRider
     * @param g - Graphics2D
     */
    public void onPaint(Graphics2D g)
    {
        g.setColor( Color.magenta );
        for(int i = 0; i < waves.size(); i++)
        {
            EnemyWave wave = waves.get( i );
            g.drawOval((int)(wave.getFiredLocation().getX() - wave.getDistanceTraveled()),
                       (int)(wave.getFiredLocation().getY() - wave.getDistanceTraveled()),
                       (int)wave.getDistanceTraveled() * 2,
                       (int)wave.getDistanceTraveled() * 2);
            
            
            int plusMinus = Utils.normalAbsoluteAngle( wave.getBearing() ) > 180 && Utils.normalAbsoluteAngle( wave.getBearing() ) < 270 ? -1 : 1;
            g.drawLine( (int)wave.getFiredLocation().getX(), (int)wave.getFiredLocation().getY(),
                (int)(wave.getFiredLocation().getX() + plusMinus * Math.sin( wave.getBearing() ) * wave.getDistanceTraveled()),
                (int)(wave.getFiredLocation().getY() + plusMinus * Math.cos( wave.  getBearing() ) * wave.getDistanceTraveled()));
        }
        
    }
}
