package djc;

/**
 * MathHelper - Standard Math functions plus functions
 * relating to the standard physics of the robocode arena.
 *
 * Robocode FAQ at:
 *    http://robocode.alphaworks.ibm.com/help/robocode.faq.txt
 *
 * @author Dan Cieslak
 */
public class MathHelper
{
    // Constants
    public static final double TWO_PI = 2.0 * Math.PI;
    public static final double MIN_SHOT_ENERGY = 0.1;
    public static final double MAX_SHOT_ENERGY = 3.0;
    public static final double MAX_ROBOT_VELOCITY = 8.0;  // 8 pixels per time unit.
    public static final double MAX_MELEE_SHOT = 3.0;
    public static final double BOT_RADIUS = 18.0;
    public static final double BOT_ACCELERATION = 1.0; // One pixel per frame per frame acceleration

    //if a heading is not within the 0 to 2pi range, alters it to provide the shortest angle
    public static double normalizeHeading(double ang) {
	return ((ang + (17 * Math.PI)) % (2 * Math.PI));
    }	
    
    //if a bearing is not within the -pi to pi range, alters it to provide the shortest angle
    public static double normalizeBearing(double ang) {
	return ((ang + (17 * Math.PI)) % (2 * Math.PI)) - Math.PI;
    }

    /**
     * Based on RoboCode FAQ.
     *
     * @see <a href="http://robocode.alphaworks.ibm.com/help/robocode.faq.txt">FAQ</a>
     * Author Dan Cieslak
     */
    public static double getBulletDamage(double power) {
	double retval = 4.0 * power;
	if(power > 1.0) retval += 2.0 * (power - 1.0);
	return retval;
    }

    /**
     * Based on RoboCode FAQ.
     *
     * @see <a href="http://robocode.alphaworks.ibm.com/help/robocode.faq.txt">FAQ</a>
     * Author Dan Cieslak
     */
    public static double getPowerToKill(double energy) {
	if(energy < 4.0) 
	    return Math.max(MIN_SHOT_ENERGY, (energy / 4.0));
	else {
	    return Math.min(MAX_SHOT_ENERGY, ((energy - 2.0) / 6.0));
	}
    }

    /**
     * Based on RoboCode FAQ.
     *
     * @see <a href="http://robocode.alphaworks.ibm.com/help/robocode.faq.txt">FAQ</a>
     */
    public static double getShotEnergyLoss(double power) {
	return power;
    }

    /**
     * Based on RoboCode FAQ.
     *
     * @see <a href="http://robocode.alphaworks.ibm.com/help/robocode.faq.txt">FAQ</a>
     */
    public static double getShotVelocity(double power) {
	return 20.0 - 3.0 * power;
    }

    /**
     * Based on RoboCode FAQ.
     *
     * @see <a href="http://robocode.alphaworks.ibm.com/help/robocode.faq.txt">FAQ</a>
     */
    public static double getShotEnergyGain(double power) {
	return 3.0 * power;
    }

    /**
     * Based on RoboCode FAQ.
     *
     * @see <a href="http://robocode.alphaworks.ibm.com/help/robocode.faq.txt">FAQ</a>
     */
    public static double getShotPowerFromEnergyGain(double energy) {
	return energy / 3.0;
    }

    /**
     * Based on RoboCode FAQ.
     *
     * @see <a href="http://robocode.alphaworks.ibm.com/help/robocode.faq.txt">FAQ</a>
     */
    public static double getTurningRate(double velocity)
    {
	return 10.0 - .75 * velocity;
    }

    /**
     * Based on RoboCode FAQ.<br>
     * <br>
     * Uses the Game Physics to try to the place closest to endLoc for the
     * given starting location, time, velocity and heading.  Assumes that
     * arrival velocity is MAX_ROBOT_VELOCITY.
     * <br>
     * Not working, yet...
     * <br>
     * @see <a href="http://robocode.alphaworks.ibm.com/help/robocode.faq.txt">FAQ</a>
     *
     * @param startLoc - Bot starting location
     * @param endLoc - Targeted destination
     * @param t - Time
     * @param v - Current velocity
     * @param heading - current heading (in radians)
     * @return Coordinate closest possible to endLoc based on game physics
     */
    public static Coordinate canRobotMoveHere(Coordinate startLoc, Coordinate endLoc,
					      double t, double v, double heading)
					      
    {
	// First, find the heading from startLoc to endLoc
	double headingToDest = startLoc.headingTo(endLoc);

	// Second, find the change in heading needed to get there
	double headingChangeNeeded = headingToDest - heading;
	if (headingChangeNeeded > Math.PI/2) {
	    headingChangeNeeded -= Math.PI;
	} else if (headingChangeNeeded < -Math.PI/2) {
	    headingChangeNeeded += Math.PI;
	}

	// Third, get the turning rate and time to turn
	double turnRate = MathHelper.getTurningRate(v);  // assume it has not achieved maxvel
	double turnTime = Math.abs(headingChangeNeeded) / Math.toRadians(turnRate); //toRadians???
	
	// Straight-line distance between startLoc and endLoc
	double d = startLoc.distanceFrom(endLoc);

	return new Coordinate(endLoc);
    }

}
