package kc.mini;

import java.awt.Color;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import robocode.AdvancedRobot;
import robocode.HitRobotEvent;
import robocode.RobotDeathEvent;
import robocode.Rules;
import robocode.ScannedRobotEvent;
import robocode.StatusEvent;
import robocode.util.Utils;

/**
* Mirage, by Kevin Clark (Kev).
* Mini-sized melee bot with HoT/Linear avoidance and multiple-choice play-it-forward targeting
* See https://robowiki.net/wiki/Mirage for more information.
*/

public class Mirage extends AdvancedRobot {
  // Game constants
  private static final int MAX_OPPONENTS = 20;
  private static final double FIELD_WIDTH = 1000.0;
  private static final double FIELD_HEIGHT = 1000.0;
  private static final double WALL_MARGIN = 17.5;

  // Gun constants
  private static final int FIRE_ANGLES = 1000;
  private static final int TABLE_SIZE = 1000;

  // Indices of opponent attributes
  private static final int SCAN_TIME = 0;
  private static final int HEADING = 1;
  private static final int VELOCITY = 2;
  private static final int ENERGY = 3;
  private static final int BULLET_POWER = 4;
  private static final int MARKOV_STATE = 5;

  // Aiming data
  private static int[][][] markovTransitionTable = new int[MAX_OPPONENTS][700][TABLE_SIZE + 1];

  // Opponent info
  private static double[][] opponents;
  private static Point2D.Double[] opponentLocs;
  private static Line2D.Double[][][] bullets;
  private static int[] opponentIDs = new int[262144];
  private static int opponentsSeen;
  private static int targetId;
  private static double targetDistance;

  // Our info
  private static int gameTime;
  private static Point2D.Double myLocation;

  @Override
  public void run() {
    setAllColors(new Color(16768200));
    setAdjustGunForRobotTurn(true);
    setTurnRadarRightRadians(targetDistance = Double.POSITIVE_INFINITY);

    opponents = new double[MAX_OPPONENTS][MARKOV_STATE + 1];
    opponentLocs = new Point2D.Double[MAX_OPPONENTS];
    bullets = new Line2D.Double[MAX_OPPONENTS][20000][2];

    // Minimum risk movement with head-on/linear targeting avoidance.
    do {
      Point2D.Double opponentLoc;
      double[] opponent;
      int id = 0;
      double moveDist = Math.min(Math.max(targetDistance * 0.4, 30), 150);
      double minRisk = Double.POSITIVE_INFINITY;
      double bestHeading = 0;
      double candidateHeading = 0;
      while ((candidateHeading += 2 * Math.PI / 100) < Math.PI * 2.01) {
        Point2D.Double location;
        if (fieldContains(location = project(myLocation, candidateHeading, moveDist))) {
          // Stay away from the center
          double risk = -getOthers() * location.distance(500, 500) / 5000;
          try {
            id = 0;
            while (true) {
              if ((opponentLoc = opponentLocs[id]) != null) {
                opponent = opponents[id];

                // Stay away from opponents
                risk += 80000 / location.distanceSq(opponentLoc);

                // Stay parallel to opponents if they are likely shooting at us
                boolean likelyFiringOnUs;
                if (likelyFiringOnUs = (id == targetId || weAreClosestTo(opponentLoc))) {
                  risk += Math.abs(Math.cos(
                      absoluteBearing(location, opponentLoc) - candidateHeading));
                }

                for (int i = 0; i < 2; i++) {  // i == 0: head-on bullets, i == 1: linear bullets
                  // Stay away from virtual bullets
                  Line2D.Double bullet = bullets[id][Math.max(0,
                      (int)(gameTime + (moveDist / 8) -  location.distance(opponentLoc) /
                          Rules.getBulletSpeed(opponent[BULLET_POWER]) + 1))][i];
                  if (bullet != null) {
                    risk += 3000 / (1000 + bullet.ptLineDistSq(location)) / (1 + i);
                  }

                  // Make new virtual bullets
                  if (likelyFiringOnUs && gameTime > 20) {
                    double absBearing;
                    bullets[id][gameTime][i] = new Line2D.Double(
                        opponentLoc, project(opponentLoc,
                            (absBearing = absoluteBearing(opponentLoc, myLocation)) + Math.asin(
                                i * getVelocity() * Math.sin(getHeadingRadians() - absBearing) /
                                Rules.getBulletSpeed(opponent[BULLET_POWER])), 10000));
                  }
                }

                // Predict enemy movements when we aren't scanning them
                if (candidateHeading > 2 * Math.PI - 0.001) {
                  opponentLocs[id] = project(
                    opponentLoc, opponent[HEADING], opponent[VELOCITY] / 2);
                }
              }
              id++;
            }
          } catch(Exception ex) {}

          if (risk < minRisk) {
            minRisk = risk;
            bestHeading = candidateHeading;
          }
        }
      }

      // Move to the lowest risk point
      if (Math.abs(getDistanceRemaining()) < 60 || targetDistance < 100) {
        setTurnRightRadians(Math.tan(bestHeading -= getHeadingRadians()));
        setAhead(moveDist * Math.cos(bestHeading));
      }
      execute();
    } while(true);
  }

  @Override
  public void onStatus(StatusEvent e) {
    myLocation = new Point2D.Double(getX(), getY());
    gameTime = (int)getTime();
  }

  @Override
  public void onScannedRobot(ScannedRobotEvent e) {
    // Update opponent info
    int id = getOpponentID(e.getName());
    double[] opponent = opponents[id];
    double opponentVelocity = e.getVelocity();
    double distance;
    double absoluteBearing;
    Point2D.Double opponentLoc = opponentLocs[id] = project(myLocation, absoluteBearing =
        (getHeadingRadians() + e.getBearingRadians()), distance = e.getDistance());
    double energyDrop;
    if ((energyDrop = (opponent[ENERGY] - (opponent[ENERGY] = e.getEnergy()))) <= 3.0 &&
        energyDrop > 0.0999) {
      opponent[BULLET_POWER] = energyDrop;
    }

    // Update table of opponent's movement
    int state;
    int[][] table = markovTransitionTable[id];
    if (gameTime > 1 && opponent[SCAN_TIME] == (opponent[SCAN_TIME] = gameTime) - 1) {
      state = makeState(
          opponentVelocity, e.getHeadingRadians() - opponent[HEADING], opponent[VELOCITY]);
      if (opponent[MARKOV_STATE] > 0) {
        int[] nextStates = table[(int)opponent[MARKOV_STATE]];
        nextStates[nextStates[TABLE_SIZE] < TABLE_SIZE ?
            nextStates[TABLE_SIZE]++ : (int)(Math.random() * TABLE_SIZE)] = state;
      }
      opponent[MARKOV_STATE] = state;
    } else {
      state = makeState(opponentVelocity, 0, opponentVelocity);
      opponent[MARKOV_STATE] = -1;
    }

    // Update rest of opponent info; done later so we can compute delta heading/turn
    opponent[VELOCITY] = opponentVelocity;
    opponent[HEADING] = e.getHeadingRadians();

    // We are scanning the targeted opponent
    if (distance < targetDistance || targetId == id) {
      targetDistance = distance;
      targetId = id;

      // Radar lock
      if (getGunHeat() < 1) {
        setTurnRadarRightRadians(Double.POSITIVE_INFINITY * Utils.normalRelativeAngle(
          absoluteBearing - getRadarHeadingRadians()));
      }

      // Fire
      double bulletPower;
      if ((bulletPower = Math.min(getEnergy() / 5, Math.min(opponent[ENERGY] / 4,
          Math.min(3, 600 / distance)))) < getEnergy() - 0.1 && getGunTurnRemaining() == 0) {
        setFire(bulletPower);
      }

      // Play-it-forward + multiple-choice aiming
      int bestBin = 0;
      double[] aimBins = new double[FIRE_ANGLES];
      for (int i = 0; i < 200; i++) {
        Point2D.Double location = opponentLoc;
        id = state;
        int ticks = 0;
        double h = opponent[HEADING];
        double v = opponentVelocity;
        double weight = 1;
        while (++ticks * Rules.getBulletSpeed(bulletPower) < myLocation.distance(location)) {
          // Sample a random next state and play forward the movement
          int tableSize;
          id = ((tableSize = table[id][TABLE_SIZE]) == 0 ? id :
            table[id][(int)(Math.random() * tableSize)]);
          if (tableSize == 0) {
            weight = 0.1;
          }
          if (!fieldContains(location = project(
              location,
              h += ((id >> 7) - 2) * Rules.getTurnRateRadians(v) / 2,
              v = ((id >> 2) & 31) - 8))) {
            weight = 0.01;
            break;
          }
        }

        // Update kernel densities and check if new best bin is found
        int width;
        int aimBin;
        aimBins[aimBin = (int)(FIRE_ANGLES * Utils.normalAbsoluteAngle(absoluteBearing(
            myLocation, location)) / (2 * Math.PI))] += weight / 5;
        for (int bin = -(width = (int)Math.round(
            FIRE_ANGLES * Math.atan(18 / distance) / (2 * Math.PI))); bin <= width; bin++) {
          int b;
          if ((aimBins[b = (aimBin + bin + FIRE_ANGLES) % FIRE_ANGLES] +=
              weight) > aimBins[bestBin]) {
            bestBin = b;
          }
        }
      }

      setTurnGunRightRadians(Utils.normalRelativeAngle(
          2 * Math.PI * bestBin / FIRE_ANGLES - getGunHeadingRadians()));
    }
  }

  @Override
  public void onRobotDeath(RobotDeathEvent e) {
    int id = getOpponentID(e.getName());
    if (id == targetId) {
      targetDistance = Double.POSITIVE_INFINITY;
    }
    opponentLocs[id] = null;
  }

  @Override
  public void onHitRobot(HitRobotEvent e) {
    targetId = getOpponentID(e.getName());
    targetDistance = 0;
  }

  private static int getOpponentID(String name) {
    int code = name.hashCode() & 262143;
    int id = opponentIDs[code];
    return opponentIDs[code] = (id == 0 ? ++opponentsSeen : id);
  }

  private static Point2D.Double project(Point2D.Double source, double angle, double length) {
    return new Point2D.Double(source.x + Math.sin(angle) * length,
       source.y + Math.cos(angle) * length);
  }

  private static double absoluteBearing(Point2D.Double source, Point2D.Double target) {
    return Math.atan2(target.x - source.x, target.y - source.y);
  }

  private static boolean fieldContains(Point2D.Double location) {
    return new Rectangle2D.Double(WALL_MARGIN, WALL_MARGIN,
        FIELD_WIDTH - (2 * WALL_MARGIN), FIELD_HEIGHT - (2 * WALL_MARGIN)).contains(location);
  }

  private static int makeState(double v, double h, double lastV) {
    return  (int)Math.signum(v - lastV) + 1 +  // acceleration
        ((int)Math.rint(8 + v) << 2) +  // velocity
        (((int)Math.rint(  // delta heading
            2 * Utils.normalRelativeAngle(h) / Rules.getTurnRateRadians(lastV)) + 2) << 7);
  }

  private static boolean weAreClosestTo(Point2D.Double opponentLoc) {
    try {
      int id = 0;
      Point2D.Double other;
      while (true) {
        if ((other = opponentLocs[id++]) != null && other != opponentLoc &&
            0.9 * opponentLoc.distance(myLocation) > opponentLoc.distance(other)) {
          return false;
        }
      }
    } catch (Exception ex) {}
    return true;
  }
}