package nat.gun.meteor;

import java.awt.Color;
import java.awt.event.KeyEvent;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;

import robocode.RobotStatus;
import robocode.ScannedRobotEvent;
import robocode.StatusEvent;
import robocode.util.Utils;
import nat.base.BotBase;
import nat.base.EventListener;
import nat.base.Gun;
import nat.base.Logger;
import nat.base.Time;
import nat.base.actors.GunActor;
import nat.gfx.GraphicsLayer;
import nat.gfx.RobocodeGraphics;
import nat.gfx.renderables.Arrow;
import nat.gfx.renderables.Bot;
import nat.gun.meteorite.Trace;
import nat.utils.KdTree;
import nat.utils.M;
import nat.utils.Point;

public class Meteor extends EventListener implements Gun {
	// Constant
	private static final int PREDICTED_LOCATION = 100;
	private static final int WAITING_TRACES = 2;
	private static final double BOT_WIDTH = 20d;

	public RobotStatus status;
	public Time time;
	public static HashMap<String, Enemy> enemies;
	public Logger logger;
	public GraphicsLayer g;

	public boolean isAiming = false;
	public double firePower = 0;

	public double[][] aimingArray;
	public Point[] locationArray;
	public int bestIndex;

	static {
		enemies = new HashMap<String, Enemy>();
	}

	public Meteor(Logger logger, RobocodeGraphics g) {
		this.g = g.getLayer("Gun", KeyEvent.VK_G);
		this.logger = logger;
	}

	@Override
	public void doGun(GunActor actor) {

		Point fireLocation = new Point(status.getX(), status.getY());
		// ready to fire?
		if (!isAiming && status.getGunHeat() < 0.2 && status.getEnergy() > 0d) {

			// get predictions
			int predictEachEnemy = (int) M.ceil((double) PREDICTED_LOCATION
					/ time.getOthers());

			ArrayList<FireLocation> predictedLocation = new ArrayList<FireLocation>(
					predictEachEnemy * time.getOthers());

			for (Enemy enemy : enemies.values()) {

				if (!enemy.isAlive)
					continue;

				calcFirePower(enemy);
				double bulletV = M.getBulletSpeed(enemy.firePower);

				int timeDelta = (int) (time.getTime() - enemy.last.time);

				List<KdTree.Entry<Trace>> nearestNeighbors = enemy.data
						.nearestNeighbor(enemy.location, predictEachEnemy * 4,
								true);

				int j = 0;
				for (int i = (nearestNeighbors.size() - 1); i >= 0
						&& j < predictEachEnemy; i--) {

					Point p = playItForward(fireLocation, bulletV,
							nearestNeighbors.get(i).value, enemy.last,
							timeDelta - 1);

					if (p != null) {
						FireLocation a = new FireLocation();
						a.power = enemy.firePower;
						a.distance = p.distance(fireLocation);
						a.location = p;
						a.angle = fireLocation.angle(p);
						a.gunTurnNeed = M.abs(M.normalRelativeAngle(status
								.getGunHeadingRadians()
								- a.angle));
						a.tolerance = BOT_WIDTH / a.distance;
						predictedLocation.add(a);
						j++;
					}
				}
			}
			aimingArray = new double[predictedLocation.size()][6];
			locationArray = new Point[predictedLocation.size()];
			bestIndex = 0;
			for (int i = 0; i < predictedLocation.size(); i++) {
				FireLocation loc = predictedLocation.get(i);
				locationArray[i] = loc.location;
				aimingArray[i][0] = loc.angle;
				aimingArray[i][1] = loc.tolerance;
				aimingArray[i][3] = (1 / M.cbrt(loc.distance))
						- M.limit(0, M.qur(loc.gunTurnNeed / M.PI), 0.1);
				aimingArray[i][2] = aimingArray[i][3];
				aimingArray[i][4] = loc.distance;
				aimingArray[i][5] = loc.power;

				for (int j = 0; j < i; j++) {
					if (M.abs(aimingArray[j][0] - aimingArray[i][0]) < aimingArray[j][1]) {
						aimingArray[j][2] += aimingArray[i][3];// 1;
					}
					if (M.abs(aimingArray[j][0] - aimingArray[i][0]) < aimingArray[i][1]) {
						aimingArray[i][2] += aimingArray[j][3];// 1;
					}
				}
			}

			bestIndex = 0;
			for (int i = 1; i < locationArray.length; i++) {
				if (aimingArray[bestIndex][2] < aimingArray[i][2]) {
					bestIndex = i;
				}
			}

			if (aimingArray.length == 0) {
				// no data, shot head-on
				Point closest = null;
				for (Enemy enemy : enemies.values()) {
					if (closest == null
							|| closest.distanceSq(fireLocation) > enemy
									.distanceSq(fireLocation))
						closest = enemy;
				}
				actor.turnGunTo(closest);
			} else {
				isAiming = true;
				actor.turnGunTo(aimingArray[bestIndex][0]);
			}
		}

		if (aimingArray != null && aimingArray.length > 0) {
			g.add(new Bot(locationArray[0], Color.red.darker().darker()
					.darker()));
			for (int i = 1; i < locationArray.length; i++) {
				g.add(new Bot(locationArray[i], null));
			}
			if (isAiming)
				for (int i = 0; i < locationArray.length; i++) {
					g.add(new Arrow(fireLocation, aimingArray[i][0],
							aimingArray[i][4] * .66d, null));
				}

			g.add(new Bot(locationArray[bestIndex], Color.yellow));

			if (isAiming)
				g.add(new Arrow(fireLocation, aimingArray[bestIndex][0],
						aimingArray[bestIndex][4] * .66d + 20d, null));
		}

		if (isAiming && status.getGunTurnRemaining() == 0
				&& status.getGunHeat() == 0) {
			actor.fire(aimingArray[bestIndex][5]);
			firePower = 0;
			isAiming = false;
		}

		if (!isAiming && time.getOthers() == 1 && enemies.values().size() > 0) {
			actor.turnGunTo(enemies.values().iterator().next());
		}
	}

	private final void calcFirePower(Enemy enemy) {
		double bulletPower = 3d;
		bulletPower = Math.min(status.getEnergy() / 6, 1300 / enemy.distance);
		bulletPower = Math
				.min(bulletPower, M.damageToBulletPower(enemy.energy));
		bulletPower = Math.max(0.1, Math.min(bulletPower, 2.999));
		bulletPower = Math.min(bulletPower, status.getEnergy());
		enemy.firePower = bulletPower;
	}

	@Override
	public void onScannedRobot(ScannedRobotEvent e) {
		Enemy enemy = enemies.get(e.getName());
		if (enemy == null) {
			enemies.put(e.getName(), enemy = new Enemy());
		}

		double absBearing = e.getBearingRadians() + status.getHeadingRadians();

		enemy.isAlive = true;
		enemy.x = status.getX() + e.getDistance() * M.sin(absBearing);
		enemy.y = status.getY() + e.getDistance() * M.cos(absBearing);

		// interpolation
		if (enemy.last != null) {
			int timeDelta = (int) (e.getTime() - enemy.last.time);
			if (enemy.last.time > e.getTime())
				timeDelta = 1;
			if (timeDelta == 1) {
				Trace trace = new Trace();
				trace.x = enemy.x;
				trace.y = enemy.y;
				trace.time = (int) time.getTime();
				trace.heading = e.getHeadingRadians();
				enemy.last.next = trace;
				trace.previous = enemy.last;
				enemy.last = trace;
			} else {
				double headingDelta = (e.getHeadingRadians() - enemy.last.heading)
						/ (double) timeDelta;
				double xDelta = (enemy.x - enemy.last.x) / (double) timeDelta;
				double yDelta = (enemy.y - enemy.last.y) / (double) timeDelta;
				while (timeDelta-- > 0) {
					Trace trace = new Trace();
					trace.x = enemy.last.x + xDelta;
					trace.y = enemy.last.y + yDelta;
					trace.heading = enemy.last.heading + headingDelta;
					enemy.last.next = trace;
					trace.previous = enemy.last;
					enemy.last = trace;
					trace.time = (int) time.getTime();
				}
			}
		} else {
			Trace trace = new Trace();
			trace.x = enemy.x;
			trace.y = enemy.y;
			trace.heading = e.getHeadingRadians();
			enemy.last = trace;
			enemy.first = trace;
			trace.time = (int) time.getTime();
		}

		double[] location = new double[9];

		double dir = M.signum(e.getVelocity()
				* M.sin(e.getHeadingRadians() - absBearing));

		location[0] = M.abs(e.getVelocity()) / 8d;
		location[1] = time.getOthers() / 10d;

		// Wall distances
		location[2] = wallDistance(enemy.x, enemy.y, e.getDistance(),
				absBearing, dir, true);
		location[3] = wallDistance(enemy.x, enemy.y, e.getDistance(),
				absBearing, -dir, true);
		location[4] = wallDistance(enemy.x, enemy.y, e.getDistance(),
				absBearing, dir, false);
		location[5] = wallDistance(enemy.x, enemy.y, e.getDistance(),
				absBearing, -dir, false);

		Trace t = enemy.last;
		for (int i = 0; i <= 32 && t.previous != null; i++, t = t.previous) {
			if (i == 8) {
				location[6] = M.min(64, M.max(-36d, angledDistance(t,
						enemy.last) + 36d)) / 100d;
			}
			if (i == 16) {
				location[7] = M.min(128, M.max(-100d, angledDistance(t,
						enemy.last) + 100d)) / 228d;
			}
			if (i == 32) {
				location[8] = M.min(256, M.max(-228d, angledDistance(t,
						enemy.last) + 228d)) / 484d;
			}
		}

		KdEntry a = new KdEntry();
		a.location = enemy.location = location;
		a.trace = enemy.last;

		enemy.queue.offer(a);

		// adding data to tree
		while (enemy.queue.size() > WAITING_TRACES) {
			KdEntry entry = enemy.queue.poll();
			enemy.data.addPoint(entry.location, entry.trace);
		}

		enemy.distance = e.getDistance();
		enemy.energy = e.getEnergy();
	}

	@Override
	public void onRobotDeath(robocode.RobotDeathEvent e) {
		try {
			enemies.get(e.getName()).isAlive = false;
		} catch (NullPointerException ignored) {
		}
	}

	@Override
	public void onStatus(StatusEvent e, Time time) {
		this.time = time;
		status = e.getStatus();
	}

	private Point playItForward(Point fireLocation, double bulletVelocity,
			Trace start, Trace current, int extraTick) {
		int currentRound = start.round;
		double distance = current.distance(fireLocation);
		double angle = M.getAngle(current, fireLocation);
		double deltaHeading = start.heading - current.heading;
		Point projectedMyLocation = M.project(start, angle + deltaHeading,
				distance);
		double bulletDistance = -bulletVelocity * extraTick;
		int estimatedBFT = (int) M.ceil(distance / bulletVelocity) + extraTick;
		for (int i = 0; i < estimatedBFT; i++) {
			start = start.next;
			if (start == null || start.round != currentRound)
				return null;
			bulletDistance += bulletVelocity;
		}
		distance = start.distance(projectedMyLocation);
		int newBFT = (int) M.ceil(distance / bulletVelocity) + extraTick;
		distance *= distance;
		if (newBFT < estimatedBFT) {
			do {
				start = start.previous;
				bulletDistance -= bulletVelocity;
				distance = start.distanceSq(projectedMyLocation);
			} while (M.sqr(bulletDistance) > distance);
			start = start.next;
			bulletDistance += bulletVelocity;
			distance = start.distanceSq(projectedMyLocation);
		} else if (newBFT > estimatedBFT) {
			do {
				start = start.next;
				if (start == null || start.round != currentRound)
					return null;
				bulletDistance += bulletVelocity;
				distance = start.distanceSq(projectedMyLocation);
			} while (M.sqr(bulletDistance) < distance);
		}
		angle = M.getAngle(projectedMyLocation, start);
		distance = M.sqrt(distance);
		Point resultLocation = M.project(fireLocation, angle - deltaHeading,
				distance);
		if (!BotBase.BATTLEFIELD.contains(resultLocation)) {
			return null;
		}
		return resultLocation;
	}

	@SuppressWarnings("serial")
	private static class Enemy extends Point {
		public Enemy() {
			super(0, 0);
		}

		KdTree<Trace> data = new KdTree.SqrEuclid<Trace>(9, null);
		Queue<KdEntry> queue = new LinkedList<KdEntry>();
		boolean isAlive = true;
		@SuppressWarnings("unused")
		Trace first, last;
		double[] location;
		double firePower = 0;
		double distance, energy;
	}

	private static class KdEntry {
		Trace trace;
		double[] location;
	}

	private static class FireLocation {
		Point location;
		double power;
		double distance;
		double gunTurnNeed;
		double angle;
		double tolerance;
	}

	// eDist = the distance from you to the enemy
	// eAngle = the absolute angle from you to the enemy
	// oDir = 1 for the clockwise orbit distance
	// -1 for the counter-clockwise orbit distance
	// returns: the positive orbital distance (in radians) the enemy can travel
	// before hitting a wall (possibly infinity).
	private final double wallDistance(double x, double y, double eDist,
			double eAngle, double oDir, boolean a) {
		final double N = BotBase.BATTLEFIELD.getMaxX();
		final double E = BotBase.BATTLEFIELD.getMaxY();
		final double S = BotBase.BATTLEFIELD.getMinX();
		final double W = BotBase.BATTLEFIELD.getMinY();
		return Math.min(Math.min(Math.min(distanceWest(N - y, eDist, eAngle
				- M.HALF_PI, oDir, a), distanceWest(E - x, eDist, eAngle
				+ Math.PI, oDir, a)), distanceWest(6 - S, eDist, eAngle
				+ M.HALF_PI, oDir, a)), distanceWest(x - W, eDist, eAngle,
				oDir, a));
	}

	private final double distanceWest(double toWall, double eDist,
			double eAngle, double oDir, boolean a) {
		if (eDist <= toWall) {
			return Double.POSITIVE_INFINITY;
		}
		double wallAngle = Math.acos(-oDir * toWall / eDist) + oDir
				* (a ? M.HALF_PI : M.PI);
		return Utils.normalAbsoluteAngle(oDir * (wallAngle - eAngle));
	}

	private final double angledDistance(Trace a, Trace b) {
		return a.distance(b) * Math.cos(a.angle(b) - a.heading);
	}
}
