/*
 * VirtualBullet - utility class for keeping bullet data
 * Copyright (C) 2002  Joachim Hofer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * You can contact the author via email (qohnil@johoop.de) or write to
 * Joachim Hofer, Feldstr. 12, D-91052 Erlangen, Germany.
 */

package qohnil.blot;

import robocode.AdvancedRobot;

import java.io.Serializable;

import qohnil.util.BotMath;
import qohnil.util.Coords;

public abstract class VirtualBullet extends MovableData implements Serializable {
    double firePower = 0.0;
    Coords originCoords = null;
    long hitTime = 0;
    static double reliability = 0.3;

    public VirtualBullet(double angle, double firePower, Coords coords, long time) {
        super();

        this.time = time;
        this.firePower = firePower;
        velocity = BotMath.getShotVelocity(firePower);
        heading = angle;
        this.coords = new Coords(
                coords.getX() + Math.sin(heading) * 22.5,
                coords.getY() + Math.cos(heading) * 22.5
        );
        this.originCoords = new Coords(coords);
    }

    public void update(long currentTime) {
        coords = getLinearPredictedCoords(currentTime - time, time);
        time = currentTime;
    }

    public boolean hasHit(Coords coords) {
        return (coords.distance(this.coords) <= 22.5);
    }

    public double willHitProbability(Coords coords, int numNextTicks) {
        double minDistance = Double.POSITIVE_INFINITY;
        Coords start = null;
        Coords end = null;
        for (int i = 0; i < numNextTicks; i++) {
            Coords nextCoords = getLinearPredictedCoords(i, time);
            if (i == 0) {
                start = nextCoords;
            } else if (i == numNextTicks - 1) {
                end = nextCoords;
            }
            double distance = coords.distance(nextCoords);
            if (distance <= minDistance) {
                minDistance = distance;
            }
        }
        //BlotBot.getInstance().addLine(start.getX(), start.getY(), end.getX(), end.getY());

        return 22.5 / minDistance;
    }

    public double getFirePower() { return firePower; }

    public boolean hasPassed(Coords coords, double width, double height) {
        double distOpponent = coords.distance(originCoords.getX(), originCoords.getY());
        double distBullet = this.coords.distance(originCoords.getX(), originCoords.getY());

        return (distBullet > distOpponent + 22.5
                || coords.getX() < 0.0 || coords.getY() < 0.0
                || coords.getX() > width
                || coords.getY() > height);
    }

    /**
     * Should be called when a bullet hits us...
     */
    public void adaptReliability(double realAngle) {
        double dAngle = Math.abs(BotMath.normalizeRelativeAngle(heading - realAngle));

        // simple moving average with previous reliabilities
        reliability = 0.67 * reliability + 0.33 * (Math.PI / 2 - dAngle) / (Math.PI / 2);
        if (this instanceof LinearEnemyBullet) {
            System.out.println("Linear reliability: " + getReliability());
        } else {
            System.out.println("Pattern reliability: " + getReliability());
        }
    }

    public double getReliability() {
        return reliability * reliability;
    }

    public long getHitTime() {
        return hitTime;
    }
}
