/*
 * Written by Kinsen Choy
 */

package kinsen.melee.Guns;
import kinsen.melee.Details;
import java.awt.Point;
import java.util.ArrayList;

/*
 * GunAvoidance - Class that generates likely simple gun bullet locations.
 */
public class GunAvoidance extends Gun
{
	// Contains gun classes to calculate firing parameters
	private ArrayList gunFunctions = new ArrayList();
	private static final byte NUMBER_GUNS = 8;

	// This is needed because enemies fire one tick behind
	private Details previousMe = null;

	// Data that is returned
	private ArrayList allFireAt = new ArrayList();

	// Data for weighting bullet hit types
	private int bulletTypeHits[] = new int[NUMBER_GUNS];
	private int totalBulletHits = 0;

	/*
	 * Guns
	 * 0: Head On
	 * 1: Linear
	 * 2: Half-Linear
	 * 3: Circular
	 * 4: Stop and Go
	 * 5: Oscillator
	 * 6: Averaged Circular
	 * 7: Pattern Matching
	 */

	/*
	 * GunAvoidance: Creates a new instance of GunAvoidance, new instances of the gun classes, and resets hit rates
	 */
	public GunAvoidance()
	{
		// Remove any existing gun classes
		gunFunctions.clear();

		// Add gun classes
		gunFunctions.add(new HeadOn());
		gunFunctions.add(new Linear());
		gunFunctions.add(new HalfLinear());
		gunFunctions.add(new Circular());
		gunFunctions.add(new StopAndGo());
		gunFunctions.add(new Oscillator());
		gunFunctions.add(new AveragedCircular());
		gunFunctions.add(new PatternMatching());
	}

	/*
	 * recordData: Records data for each gun
	 */
	public void recordData(Details enemy, Details me, long time)
	{
		// If this is first data then feed current data
		if (previousMe == null)
			previousMe = me;
		// Go through all gun classes and give data
		for (int i = 0; i < NUMBER_GUNS; i++)
		{
			((Gun) gunFunctions.get(i)).recordData(previousMe, enemy, time);
		}
		previousMe = me;
	}

	/*
	 * calculateGun: Calculates location to fire at using the best gun
	 */
	public void calculateGun(double initialFirePower)
	{
		allFireAt.clear();

		// Calculate all possible bullet locations
		for (int i = 0; i < NUMBER_GUNS; i++)
		{
			Gun currentGun = (Gun) gunFunctions.get(i);
			currentGun.calculateGun(initialFirePower, false);
			allFireAt.add(currentGun.getFireAt());
		}
	}

	public void calculateGun(double initialFirePower, boolean canChangePower)
	{
		calculateGun(initialFirePower);
	}

	/*
	 * getAllFireAt: Returns all the possible locations the bullet was fired at
	 */
	public ArrayList getAllFireAt()
	{
		return allFireAt;
	}

	/*
	 * recordHit: Records a hit by a bullet for avoidance weighting
	 */
	public void recordHit(int bulletType)
	{
		// Increase number of hits for bullet type
		bulletTypeHits[bulletType]++;
		// Increase total number of bullet hits
		totalBulletHits++;
	}

	/*
	 * getWeights: Returns the proportion of hits for each bullet
	 */
	public double[] getWeights()
	{
		// Calculate bullet type proportions by dividing number of type by total
		double bulletWeights[] = new double[NUMBER_GUNS];
		for (int i = 0; i < bulletWeights.length; i++)
		{
			// Check if it has been hit before
			if (totalBulletHits == 0)
				bulletWeights[i] = 1;
			else
				bulletWeights[i] = Math.max((double) bulletTypeHits[i] / (double) totalBulletHits, 0.01);
		}
		return bulletWeights;
	}
}