package voidious.utils.genetic;

import java.util.ArrayList;
import java.util.HashMap;

/**
 * Copyright (c) 2011 - Voidious
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *    1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software.
 *
 *    2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 *    3. This notice may not be removed or altered from any source
 *    distribution.
 */

public class DnaSequence {
    protected ArrayList<Gene> _geneLayout;
    protected HashMap<String, Gene> _geneMap;
    protected int _length = 0;

    public DnaSequence(ArrayList<Gene> genes) {
        _geneLayout = new ArrayList<Gene>();
        _geneMap = new HashMap<String, Gene>();

        if (genes != null) {
            for (Gene g : genes) {
                addGene((Gene)g.clone());
            }
        }
    }

    public DnaSequence() {
        this(null);
    }

    public void addGene(Gene g) {
        // TODO: handle existing Gene with this name somehow
        //       replace it? throw exception? overwrite it? delete/clear Genes?
        _geneMap.put(g.name, g);
        _geneLayout.add(g);
        g.position = _length;
        _length += g.length;
    }

    public Gene getGene(String name) {
        return _geneMap.get(name);
    }

    public int length() {
        return _length;
    }

    public ArrayList<Gene> getGeneLayout() {
        return _geneLayout;
    }

    public static class Gene implements Cloneable {
        public String name;
        public int type;
        public int length;
        public int position;
        public long min;
        public long max;
        public boolean negatives;

        public static final int BIT = 0; 
        public static final int BYTE = 1; 
        public static final int SHORT = 2; 
        public static final int INTEGER = 3; 
        public static final int LONG = 4; 
        public static final int FLOAT = 5; 
        public static final int DOUBLE = 6;
        public static final int[] typeSizes = new int[]{1, 8, 16, 32, 64, 32, 64};

        public Gene(String name, int type, long min, long max, 
                boolean negatives) {
            this.name = new String(name);
            this.type = type;
            this.min = min;
            this.max = max;
            this.negatives = negatives;
            length = typeSizes[type];
            position = 0;
        }

        public Gene(String name, int type) {
            this(name, type, Long.MAX_VALUE);
        }

        public Gene(String name, int type, boolean negatives) {
            this(name, type, Long.MAX_VALUE, negatives);
        }

        public Gene(String name, int type, long max, boolean negatives) {
            this(name, type, Long.MIN_VALUE, max, negatives);
        }

        public Gene(String name, int type, long max) {
            this(name, type, Long.MIN_VALUE, max, false);
        }

        public Gene(String name, int type, long min, long max) {
            this(name, type, min, max, false);
        }

        public Object clone() {
            return new Gene(name, type, min, max, negatives);
        }

        public boolean equals(Object o) {
            if (!(o instanceof Gene)) {
                return false;
            }

            Gene that = (Gene)o;

            if (this.name.equals(that.name) && this.type == that.type &&
                this.length == that.length && this.position == that.position &&
                this.min == that.min && this.max == that.max &&
                this.negatives == that.negatives) {
                return true;
            } else {
                return false;
            }
        }
    }
}
