package rc.yoda.plugin;

import robocode.*;
import java.io.File;
import java.awt.*;
import java.util.*;
import java.awt.geom.Point2D;

import rc.yoda.plugin.guns.Circular;
import rc.yoda.plugin.guns.Electrum;
import rc.yoda.plugin.guns.HeadOn;
import rc.yoda.plugin.guns.Linear;
import rc.yoda.plugin.guns.Shoto;
import rc.yoda.utils.*;

/** 
 * GunArray- by Robert Codd (Gorded) 
 *
 * This code is here by released under the RoboWiki Public Code Licence (RWPCL),
 * datailed on: http://robowiki.net/?RWPCL
 * (Basically it means you must keep the code public if you base any bot on it)
 *
 * GunArray.java : v1.0 -- 2007/05/12
 */


/**
 * GunArray is a class that loads and 
 * manages an array of virtual guns that 
 * trys to determine which gun is best suited 
 * for defeating the current oppenent
 *
 * @author Robert Codd
 * @version v1.0
 */
public class GunArray implements Plugin
{
	/**
	 * The name of the folder all gun plugins are stored
	 */
	private static final String FOLDER = "guns";
	
	/**
	 * How often to fire virual bullets from the guns
	 * 0 : only when firing 
	 * 1: every turn 
	 * 2 : every two turns
	 */	
	private static final int VIRTUAL_FIRE_SPEED = 0;
	
	/**
	 * A HashMap that contains the hit count stats for 
	 * each gun, index by the guns name - to avoid stat swapping
	 */
	private static HashMap gunStats;	
		
	/**
	 * An array containing the names of all guns in use
	 */
	private String[] gunNames;

	/**
	 * A list of all the VirtualBullets on the battle field
	 */
	private ArrayList _Bullets;
	
	/**
	 * A HashMap of all the guns in use this round indexed by gun name
	 */
	private HashMap guns;
	
	/**
	 * A HashMap of the colors of all the guns in use indexed by gun name
	 */	
	private HashMap gunColors;
	
	/**
	 * The location of the robot 
	 */
	private Point2D.Double robotsLocation;
	
	/**
	 * The location of the enemy to target
	 */
	private Point2D.Double enemyLocation;
	
	/**
	 * A reference to the Advanced Robot the Gun Array is working for
	 */
	private AdvancedRobot robot;
	
	/** 
     * Class constructor specifying the AdvancedRobot it works for
     */
	public GunArray(AdvancedRobot robot) {
		this.robot = robot;		
		init();
	}

	/**
	 * Once a round management for the Gun Array
	 */
	public void init() {
		if (robot.getRoundNum() == 0) { gunStats = new HashMap(); }
		_Bullets = new ArrayList();
		guns = new HashMap();
		gunColors = new HashMap();
		populateMap();
	}

	/**
	 * Event method called by this robot when it detects
	 * that the enemy fired a bullet
	 *
	 * @param double deltaEnergy the power of the bullet fired
	 */
	public void onRobotFire(double deltaEnergy) {} 
	
	/**
	 * Event method called by Robocode when this robot's scanner
	 * passes over another robot
	 * 
	 * @param ScannedRobotEvent information about the scanned robot
	 */		
	public void onScannedRobot(ScannedRobotEvent e) {	
		if (guns.size() == 0) { return; }	
		
		String bestGun = bestGun();
		robotsLocation = new Point2D.Double(robot.getX(), robot.getY());
						
		double enemyBearing = robot.getHeadingRadians() + e.getBearingRadians(),
			enemyDistance = e.getDistance(),
			robotGunHeat = robot.getGunHeat();
		boolean	canFire = (VIRTUAL_FIRE_SPEED != 0 ? (robot.getTime() % VIRTUAL_FIRE_SPEED == 0 || robotGunHeat == 0) : robotGunHeat == 0);	
						
		enemyLocation = YUtils.project(robotsLocation, enemyBearing, enemyDistance);	
		double fireAngle = 0, firePower = 0;
										
																																
		for (int count = 0; count < gunNames.length; count++)
		{
			double angle = 0, power = 0;
			Gun gun = (Gun) guns.get(gunNames[count]);
			gun.onScannedRobot(e);
			
			if (canFire || gunNames[count].equals(bestGun))
			{
				angle = gun.getFireAngle();
				power = gun.getBulletPower();
				
				VirtualBullet bullet = new VirtualBullet(gunNames[count], robotsLocation,
					YUtils.normalRelativeAngle(enemyBearing + angle), Laws.getBulletSpeed(power), (robotGunHeat == 0 && gunNames[count].equals(bestGun)));

				if (canFire) { _Bullets.add(bullet); }
				if (gunNames[count].equals(bestGun)) { fireAngle = enemyBearing + angle; firePower = power; }
			}
		}
		fire(fireAngle, firePower);
		updateBullets();
	}
	
	/**
	 * Event method called by Robocode when this robot
	 * gets hit by a bullet
	 * 
	 * @param HitByBulletEvent information about ther bullet
	 * that hit this robot
	 */	
	public void onHitByBullet(HitByBulletEvent e) {}
	
	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired hits another robot
	 * 
	 * @param BulletHitEvent information about the robot 
	 * that got hit by the bullet
	 */	
	public void onBulletHit(BulletHitEvent e) {		
		if (guns.size() == 0) { return; }
		Gun firingGun = firingGun(e.getBullet(), false);			
		if (firingGun != null) { firingGun.onBulletHit(e); }			
	}
	
	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired hit a wall
	 *
	 * @param BulletMissedEvent information about the bullet
	 */	
	public void onBulletMissed(BulletMissedEvent e) {
		if (guns.size() == 0) { return; }	
		Gun firingGun = firingGun(e.getBullet(), false);	
		if (firingGun != null) { firingGun.onBulletMissed(e); }	
	}
	
	/**
	 * Event method called by Robocode when a bullet this
	 * robot fired collides with a bullet fired by another robot
	 *
	 * @param BulletHitBulletEvent information about the bullets
	 */	
	public void onBulletHitBullet(BulletHitBulletEvent e) {
		if (guns.size() == 0) { return; }	
		Gun firingGun = firingGun(e.getBullet(), false);
		if (firingGun != null) { firingGun.onBulletHitBullet(e); }
	}
	
	/**
	 * Event method called by Robocode when this robot hits a wall
	 * 
	 * @param HitWallEvent information about the wall
	 */	
	public void onHitWall(HitWallEvent e) {}
	
	/**
	 * Event method called by Robocode when this robot 
	 * collides with another robot
	 *
	 * @param HitRobotEvent information about the collision and 
	 * the other robot in the crash
	 */	
	public void onHitRobot(HitRobotEvent e) {}
	
	/**
	 * Event method called by Robocode when a robot dies
	 *
	 * @param RobotDeathEvent name of decaesed robot
	 */	
	public void onRobotDeath(RobotDeathEvent e) {}
	
	/**
	 * Event method called by Robocode when this robot
	 * wins a round
	 *
	 * @param WinEvent
	 */	
	public void onWin(WinEvent e) {
		if (guns.size() == 0) { return; }
		_Bullets.clear();
		for (int count = 0; count < gunNames.length; count++) { ((Gun)guns.get(gunNames[count])).onWin(e); }			
	}
	
	/**
	 * Event method called by Robocode when this robot dies
	 *
	 * @param DeathEvent 
	 */	
	public void onDeath(DeathEvent e) {	
		if (guns.size() == 0) { return; }			
		_Bullets.clear();
		for (int count = 0; count < gunNames.length; count++) { ((Gun)guns.get(gunNames[count])).onDeath(e); }		
	}
	
	/**
	 * Event method called by Robocode when this robot is
	 * allowed to draw debugging graphics to the screen
	 *
	 * @param Graphics2D graphics that provides drawing method for painting
	 */	
	public void onPaint(Graphics2D g) {	
		if (guns.size() == 0) { return; }
		for (int count = 0; count < gunNames.length; count++)
		{
			g.setColor(Color.white);
			if (gunNames[count] == bestGun()) {	g.drawString(String.format("%s - hits : %d <---",gunNames[count], hits(gunNames[count])), 20, 5 + count*15); }
				else { g.drawString(String.format("%s - hits : %d",gunNames[count], hits(gunNames[count])), 20, 5 + count*15); }
			g.setColor((Color)gunColors.get(gunNames[count]));
			g.fillOval(5, 5 + count * 15, 10, 10);						
		}
		for (int count = 0;  count < _Bullets.size(); count++)
		{
			VirtualBullet bullet = (VirtualBullet) _Bullets.get(count);
			g.setColor((Color)gunColors.get(bullet.gunName));
			g.fillOval((int) bullet.location.x - 3,(int) bullet.location.y - 3, 6, 6);
		}
		for (int count = 0; count < gunNames.length; count++) { ((Gun)guns.get(gunNames[count])).onPaint(g); }
	}

	/**
	 * returns the number of hits of the gun 
	 * at the key gunName 
	 *
	 * @param String the key of the gun to find its hit count
	 * @return int the hit count of the gun with key gunName
	 */
	private int hits(String gunName) {
		return ((Integer)gunStats.get(gunName)).intValue();
	}

	/**
	 * Points the gun at the desired angle and fires a bullet
	 * with the desired power
	 *
	 * @param double the angle at which the gun should be pointed
	 * @param double the power of the bullet to be shot
	 */
	private void fire(double angle, double power) {			
		robot.setTurnGunRightRadians(YUtils.normalRelativeAngle(angle - robot.getGunHeadingRadians()));		
		robot.setFire(power);
	}

	/**
	 * Advances all Virtual Bullets on the field, removes
	 * bullets that are no longer active and logHits for
	 * those that hit the enemy
	 */
	private void updateBullets() {
		for (int count = 0; count < _Bullets.size(); count++)
		{
			VirtualBullet bullet = (VirtualBullet) _Bullets.get(count);
			bullet.update();
			
			if (!YUtils.field.contains(bullet.location)) { _Bullets.remove(count--); }
			else if (enemyLocation.distance(bullet.location) < 18) { logHit(bullet); _Bullets.remove(count--); }
		}
	}

	/**
	 * Adds to the hit count of the gun who shoot
	 * the bullets that hit the enemy
	 * 
	 * @param VirtualBullet the bullet that hit the enemy
	 */
	private void logHit(VirtualBullet bullet) {
		int value = ((Integer)gunStats.get(bullet.gunName)).intValue();
		gunStats.put(bullet.gunName, new Integer(++value));
	}

	/**
	 * The closest Virtual Bullet to the specified Bullet and returns
	 * the gun that fired it and removes from list of VirtualBullets if desired
	 * 
	 * @param Bullet the bullet that you what the firing Gun for
	 * @param boolean whether to remove the found bullet from the list of VirtualBullets
	 * @return Gun the gun that fire the Bullet
	 */
	private Gun firingGun(Bullet e, boolean remove) {
		int index = -1;
		double closestDistance = Double.POSITIVE_INFINITY;
		
		for (int count = 0; count < _Bullets.size(); count++)
		{
			VirtualBullet bullet = (VirtualBullet) _Bullets.get(count);
			if (bullet.real)
			{
				double value = bullet.location.distance(e.getX(), e.getY());
				closestDistance = (value < closestDistance && (index = count) == index ? value : closestDistance);
			}
		}
		try { 
			VirtualBullet bullet = (VirtualBullet) _Bullets.get(index); 
			if (remove) { _Bullets.remove(index); }
			return (Gun) guns.get(bullet.gunName);
		} catch(ArrayIndexOutOfBoundsException ex) { return null; }
	}

	/**
	 * Returns a String key linking to the 
	 * gun with the most amount of hits
	 *
	 * @return String the key to the Gun object with the best stats
	 */
	private String bestGun() {
		String key = "";
		int lowestHR = -1;
		
		for (int count = 0; count < gunNames.length; count++)
		{
			int value = ((Integer)gunStats.get(gunNames[count])).intValue();
			lowestHR = (value > lowestHR && (key = gunNames[count]) == key ? value : lowestHR);
		}
		return key;
	}

	/**
	 * Retrives Gun classes from the plugin folder
	 * and populate this rounds HashMap of guns
	 * and indexes them by their class name
	 */
	private void populateMap() {		
		Object[] parameters = { (Object)robot };
		String canonicalPath = "",
			packageName = "";
		
		/*{
			Class Claas = getClass(); 
			String dir = robot.getDataDirectory().getPath();
			
			packageName =  Claas.getName().replaceAll(Claas.getSimpleName(), FOLDER);
			canonicalPath = dir.split(robot.getName().split(" ")[0].concat(".data"))[0] 
				+ packageName.replace(".", File.separator); 
		}*/
	
		Factory factory = new Factory((new Gun(robot)).getClass(), parameters, new Class[]{
				Circular.class,
				Electrum.class,
				HeadOn.class,
				Linear.class,
				Random.class,
				Shoto.class
		});
		Gun[] temp = (Gun[]) factory.getClasses();
		gunNames = new String[temp.length];							
								
		for (int count = 0; count < temp.length; count++)
		{
			String key = temp[count].getClass().getSimpleName();
			if (!gunStats.containsKey(key)) { gunStats.put(key, new Integer(0)); }
			gunColors.put(key, new Color(Color.HSBtoRGB(count/(float)temp.length,1,1)));
			guns.put(key, temp[count]);
			gunNames[count] = key;
		}
	}
}