/**
 * 
 */
package banshee.micro;

import static java.lang.Math.PI;
import static robocode.util.Utils.normalAbsoluteAngle;
import static robocode.util.Utils.normalRelativeAngle;

import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import robocode.AdvancedRobot;
import robocode.Rules;
import robocode.ScannedRobotEvent;

/**
 * version: 0.3.0 (733 bytes)
 * @author banshee
 */
public class Nexus6 extends AdvancedRobot {

	public List<Detractor> detractors = new ArrayList<Detractor>();
	
	private static double BATT_WIDTH;
	private static double BATT_HEIGHT;
	private static double BATT_DIAG;
	
	// enemy properties
	private String target;
	private double targetX;
	private double targetY;
	private double targetAbsAngle;
	private double targetVelo;
	private double targetHead; // absolute: 0, 2 * PI
	private double targetBear;
	
	private double targetHeadDelta; // delta from last 2 observations in target heading
//	private double targetVeloDelta; // delta from last 2 observations in target velocity


	@Override
	public void run() {

		// set some general properties
		BATT_WIDTH = getBattleFieldWidth();
		BATT_HEIGHT = getBattleFieldHeight();
		BATT_DIAG = Point2D.distance(0, 0, BATT_WIDTH, BATT_HEIGHT);
		
		// set radar & gun independent from movement
		setAdjustRadarForGunTurn(true); // will automatically call setAdjustRadarForRobotTurn(true);
		setAdjustGunForRobotTurn(true);
		
		// main loop
		while (true) {
			
			double x = getX();		
			double y = getY();

			if (target != null) {
				// lock radar on target, TODO: make the arc wider
				setTurnRadarRightRadians(normalRelativeAngle(targetAbsAngle - getRadarHeadingRadians()));
			} else {
				// scan continuously
				setTurnRadarRightRadians(Rules.RADAR_TURN_RATE_RADIANS);
			}

			detractors.add(new Detractor(0, y, detPower(x)));
			detractors.add(new Detractor(BATT_WIDTH, y, detPower(BATT_WIDTH - x)));
			detractors.add(new Detractor(x, 0, detPower(y)));
			detractors.add(new Detractor(x, BATT_HEIGHT, detPower(BATT_HEIGHT - y)));				
			
			// now find new direction to move, counting weighted average from detractors
			double newRelX = 0;
			double newRelY = 0;

			for (Detractor d : detractors) {
				double power = d.power;
				newRelX += power * (x - d.x);
				newRelY += power * (y - d.y);					
			}

			double gotoAngleRel = normalRelativeAngle(Math.atan2(newRelX - x, newRelY - y) - getHeadingRadians());			
			
			// a bit of move randomization
			if (getTime() % 3 == 0) {
				Random rand = new Random();
				gotoAngleRel = (rand.nextFloat() > 0.5F) ? 1 : -1 * Math.toRadians(rand.nextInt((int) Rules.MAX_TURN_RATE));
			}
			setTurnRightRadians(gotoAngleRel);
			
			// if about to turn sharp (> 120 degrees), stop
			if (Math.abs(gotoAngleRel) > 2 * PI / 3) {
				setAhead(0);
			} else {
				setAhead(160); // some large number...
			}
			
			// now setting gun
			if (target != null) {
				
				// determine bullet power, energy management only
				double life = getEnergy();
				double power = life < 10D ? life / 10 : 3;

				// linear targeting, using law of sines
//				setTurnGunRightRadians(normalRelativeAngle(targetAbsAngle - getGunHeadingRadians() + 
//						Math.asin(Math.sin(targetHead - targetAbsAngle) * targetVelo / Rules.getBulletSpeed(power))));
				
				// circular targeting, taking into account changes in target heading only
				int i = 1;
				double bulletSpeed = Rules.getBulletSpeed(power);
				double tHeadPred = targetHead; // predicted target heading at turn = now + i;
				double tXPred = targetX; // predicted target X at turn = now + i;
				double tYPred = targetY; // predicted target Y at turn = now + i;
				
				while (Point2D.distance(x, y, tXPred, tYPred) > i * bulletSpeed) {
					// predictions
					tHeadPred += targetHeadDelta;
					tXPred = tXPred + targetVelo * Math.sin(tHeadPred);
					tYPred = tYPred + targetVelo * Math.cos(tHeadPred);
					i++;
				}
				
				// TODO: it costs almost 50 bytes, some more golfing needed to fit into micro
//				// correct to battlefield dimension
//				if (tXPred > BATT_WIDTH) tXPred = BATT_WIDTH;
//				if (tXPred < 0) tXPred = 0;
//				if (tYPred > BATT_HEIGHT) tYPred = BATT_HEIGHT;
//				if (tYPred < 0) tYPred = 0;
				
				setTurnGunRightRadians(normalRelativeAngle(Math.atan2(tXPred - x, tYPred - y) - getGunHeadingRadians()));				
				setFire(power);
			}
			detractors.clear();
			execute();
		}
	}

	/**
	 * @param dist
	 * @return
	 */
	private double detPower(double dist) {
		// let detracting power be inversely proportional to distance 
		return Math.pow(1 / (dist / BATT_DIAG), 2);
	}
	
	@Override
	public void onScannedRobot(ScannedRobotEvent e) {
		// update enemy info
		targetHeadDelta = e.getHeadingRadians() - targetHead;

		targetBear = e.getBearingRadians();
		targetAbsAngle = normalAbsoluteAngle(targetBear + getHeadingRadians());
		targetVelo = e.getVelocity();
		targetHead = e.getHeadingRadians();
		target = e.getName();
		
		// find point where are enemy is
		double dist = e.getDistance();
		targetX = getX() + dist * Math.sin(targetAbsAngle);
		targetY = getY() + dist * Math.cos(targetAbsAngle);
		
		// update this enemy detractor info
		detractors.add(new Detractor(targetX, targetY, detPower(dist)));
	}

	
	private static class Detractor {
		
		double x;
		double y;
		double power;

		public Detractor(double x, double y, double power) {
			this.x = x;
			this.y = y;
			this.power = power;
		}

	}	

}
