package lancel.components;
import java.awt.*;
import java.awt.geom.*;
import java.text.NumberFormat;
import java.util.*;
import static java.lang.Math.*;

/**************************************************************************************************
 Lynx - A duelist, based on Dynamic Clustering (DC) and WaveSurfing
 Author: lancel
 Current Version: 1.09
 Date: 6/12/2010
 License: Open Source under the RoboWiki Public Code License RWPCL (http://robowiki.net/wiki/RWPCL)
 Robocode Version: 1.6.1.4 (because of RoboRumble)
 Techniques this bot is based on:
    Gun: a k-nearest neighbors search with a kd-tree using a simple kernel density estimation 
         (dynamic clustering)
    Movement: basic WaveSurfing using the closest wave as a surfing wave
 CREDITS: Much of the ideas and code is based on various sources. A few to mention:
    wiki.BasicGFSurfer by Voidius, Pez and Bayen: initial structure for Lynx and the implementation
        of WaveSurfing
    Komarious by Voidius: the segmentation for WaveSurfing
    ABC: inventor of WaveSurfing
    Rednaxela: the implementation of the kd-tree
    rozu: precise prediction used in the Lynx's movement system
    DrussGT by Skilgannon: normalizing and weighting "features" that describe states for the gun
        ("segmentation" in DC) 
    Raiko by Jamougha: choosing a bullet power
    PEZ: Simple Iterative Wall Smoothing
    Voidious and others: good tutorials for getting into robocoding
**************************************************************************************************/

public class DebugPaint {
// This class with static methods is for painting and printing debugging information.

    public static void movementPaint(Graphics2D g, State state, Movement mov) {
    //---------------------------------------------------------------------------------------------
    // The main function for painting movement related debugging information
        //movPaintWaves(g, s, mov);
        movPaintText(g, state, mov);
        movPaintWallStick(g, state, mov);
        movPaintPredictedPosition(g, mov);
        movPaintOppCircle(g, state, mov);
    }

    public static void gunPaint(Graphics2D g, State state, Gun gun) {
    //---------------------------------------------------------------------------------------------
    // The main function for painting gun related debugging information
        gunPaintText(g, state, gun);
    }

    public static void movPaintWaves(Graphics2D g, State state, Movement mov) {
    //---------------------------------------------------------------------------------------------
        g.setStroke(new BasicStroke(2));

        for(Movement.OppWave ow : mov.oppWaves) {
            int waveCounter = 1;
            double escRad = MyUtils.maxEscapeAngle(ow.bulletVelocity);
            double escDeg = escRad * 180 / PI;
            double binAngle = escDeg * 2.0 / Movement.BINS;
            Point2D.Double fireLoc = ow.fireLocation;
            double arcStart = ow.directAngle * 180 / PI;
            arcStart = (arcStart - escDeg) - 90;
            Arc2D.Double arc = new Arc2D.Double(Arc2D.OPEN);
            Color color = new Color(0x70,0x70, 0x70, 0x70);
            g.setColor(color);
            arc.setArcByCenter(fireLoc.x, fireLoc.y, ow.distanceTraveled, arcStart, escDeg * 2.0,
                               Arc2D.OPEN);
            g.draw(arc);

            double highestRisk = 0.0;
            int highestRiskBin = Movement.BINS / 2; // default
            int curBin = 0;
            
            if (mov.surfWave != null)
                for (double risk : mov.surfWave.waveGuessFactors) {
                    if (risk > highestRisk) {
                        highestRisk = risk;
                        highestRiskBin = curBin;
                    }
                    curBin++;
                }

            color = new Color(255, 0, 0, 0xff);
            g.setColor(color);
            arc.setArcByCenter(fireLoc.x, fireLoc.y, ow.distanceTraveled +2.0, arcStart +
                (highestRiskBin * binAngle), binAngle, Arc2D.OPEN);
            g.draw(arc);
            color = new Color(255, 255, 0, 0xff);
            g.setColor(color);
            arc.setArcByCenter(fireLoc.x, fireLoc.y, ow.distanceTraveled, arcStart +
                (highestRiskBin * binAngle), binAngle, Arc2D.OPEN);
            g.draw(arc);

            // emphasize current surfing wave
            if (mov.surfWave != null && ow == mov.surfWave) {
                g.setColor(new Color(0x80, 0x80, 0xff, 0x80));
                arc.setArcByCenter(fireLoc.x, fireLoc.y, ow.distanceTraveled -2.0, arcStart,
                                   escDeg * 2.0, Arc2D.OPEN);
                g.draw(arc);
            }
        }
    }

    public static void movPaintText(Graphics2D g, State state, Movement mov) {
    //---------------------------------------------------------------------------------------------
        NumberFormat percent = NumberFormat.getPercentInstance();
        NumberFormat nf      = NumberFormat.getInstance();
        percent.setMinimumFractionDigits(2);
        percent.setMaximumFractionDigits(2);
        nf.setMinimumFractionDigits(2);
        nf.setMaximumFractionDigits(2);

        ArrayList<String> strList = new ArrayList<String>();
        strList.add("movement related values");
        strList.add("Opp hitrate estimate: " +
            percent.format((double) mov.hitsByOpp / mov.oppBulletsFired) + " ("  + mov.hitsByOpp +
            "/" + mov.oppBulletsFired + ")");
        strList.add("myHeading: " + nf.format(state.myHeading) + " = " + (int)(state.myHeading *
                    180 / PI) + " deg");
        strList.add("velocity: " + nf.format(state.myVelocity));
        strList.add("goAngle: " + nf.format(mov.goAngle));
        strList.add("plusDanger: " + nf.format(mov.debugPlusDanger));
        strList.add("minusDanger: " + nf.format(mov.debugMinusDanger));
        strList.add("oppSurroundingRadius: " + nf.format(mov.oppSurroundingCircleRadius));
        strList.add("advancingVelocityOpp: " + nf.format(mov.advancingVelocityOpp));
        g.setColor(new Color(0x00, 0xff, 0x00, 0x80));
        int top = strList.size() * 11;
        int left = 600;
        int fontHeight = 11;
        for (String str : strList) {
            if (str != null) g.drawString(str, left, top);
            top -= fontHeight;
        }
        if(mov.advancingVelocityOpp > 6.5) {
            g.setColor(new Color(0xff, 0x00, 0x00, 0x80));
            g.drawString("Opponent is ramming", left - 150, top + fontHeight);
        }
            
    }

    public static void movPaintWallStick(Graphics2D g, State state, Movement mov) {
    //---------------------------------------------------------------------------------------------
        Point2D.Double endPoint;
        Line2D.Double  stick;

        // Stick at front
        endPoint = MyUtils.project(state.myLocation, state.myHeading, mov.WALL_STICK);
        stick    = new Line2D.Double(state.myLocation, endPoint);
        g.setColor(new Color(0, 0xff, 0, 0x80));
        g.draw(stick);

        // Stick at back
        endPoint = MyUtils.project(state.myLocation, state.myHeading + PI, mov.WALL_STICK);
        stick    = new Line2D.Double(state.myLocation, endPoint);
        g.setColor(new Color(0xa0, 0xa0, 0xa0, 0x80));
        g.draw(stick);
    }

    public static void movPaintPredictedPosition(Graphics2D g, Movement m) {
    //---------------------------------------------------------------------------------------------
        Point2D.Double position;
        if (m.debugPlusDanger >= m.debugMinusDanger) {
            g.setColor(new Color(0xa0, 0x0, 0x0, 0x80));
            g.draw(new Ellipse2D.Double(m.predictedPositionPlus.x - 10, m.predictedPositionPlus.y
                - 10, 20, 20));
            g.setColor(new Color(0x0, 0xa0, 0x0, 0x80));
            g.draw(new Ellipse2D.Double(m.predictedPositionMinus.x - 10, m.predictedPositionMinus.y
                - 10, 20, 20));
        }
        else {
            g.setColor(new Color(0x0, 0xa0, 0x0, 0x80));
            g.draw(new Ellipse2D.Double(m.predictedPositionPlus.x - 10, m.predictedPositionPlus.y
                - 10, 20, 20));
            g.setColor(new Color(0xa0, 0x0, 0x0, 0x80));
            g.draw(new Ellipse2D.Double(m.predictedPositionMinus.x - 10, m.predictedPositionMinus.y
                - 10, 20, 20));
        }
    }

    public static void movPaintOppCircle(Graphics2D g, State state, Movement mov) {
    //---------------------------------------------------------------------------------------------
        g.setColor(new Color(0, 0x7f, 0, 0x40));
        double radius;
        radius = mov.oppSurroundingCircleRadius;
        g.draw(new Ellipse2D.Double(state.oppLocation.x-radius, state.oppLocation.y-radius, 
            2*radius, 2*radius));
    }

    public static void gunPaintText(Graphics2D g, State state, Gun gun) {
    //---------------------------------------------------------------------------------------------
        g.setColor(new Color(0x00, 0xff, 0x00, 0x80));
        NumberFormat percent = NumberFormat.getPercentInstance();
        NumberFormat nf      = NumberFormat.getInstance();
        percent.setMinimumFractionDigits(2);
        percent.setMaximumFractionDigits(2);
        nf.setMinimumFractionDigits(2);
        nf.setMaximumFractionDigits(2);

        ArrayList<String> strList = new ArrayList<String>();
        strList.add(state.myName);
        strList.add("gun related values");
        strList.add("kd-tree size: " + nf.format(gun.featureSpace.size()));
        strList.add("feat[0] distancel: " + nf.format(gun.featurePoint[0]));
        strList.add("feat[1] latVel: " + nf.format(gun.featurePoint[1]));
        strList.add("feat[2] adv: " + nf.format(gun.featurePoint[2]));
        strList.add("feat[3] distLast10: " + nf.format(gun.featurePoint[3]));
        strList.add("feat[4] dirTime: " + nf.format(gun.featurePoint[4]));
        strList.add("feat[5] decTime " + nf.format(gun.featurePoint[5]));
        strList.add("NEIGHBORS: " + Integer.toString(gun.NEIGHBORS));
        strList.add("botWidth: " + Integer.toString(gun.targetBotWidth));
        strList.add("hitrate: " + percent.format((double)gun.bulletHits / gun.bulletsFired) + " (" +
            gun.bulletHits  + "/" + gun.bulletsFired + ")");

        String ffStr = "featureWeights: ";
        for (int i = 0; i < gun.featureWeights.length; i++) {
            ffStr += nf.format(gun.featureWeights[i]) + " ";
        }
        strList.add(ffStr);
        g.setColor(new Color(0, 0xff, 0, 0x80));

        int fontHeight = 11;
        int top = strList.size() * fontHeight;
        int left = 10;

        for (String str : strList) {
            if (str != null) g.drawString(str, left, top);
            top -= fontHeight;
        }
    }
} // end class DebugPaint
